% loc2mods2
%
% This script converts an LOC netCDF file created by HydroBase to
% a MODS formatted ASCII file.

%-------------------------------------------------------------------------------
% Set run parameters.
%
%    basename.....Base name for input/output files.
%    mincnt.......Minimum acceptible number of data points that went into
%                 creating a climatology point.
%    in_suff......Suffix for input files.
%    out_suff.....Suffix for input files.
%    ttl..........Descriptive title for MODS formatted output file.
%    time.........Time to write to Climatology file. [yr mon day hr min sec]
%    julian_off...Off set to subtract from Julian dates.
%-------------------------------------------------------------------------------

% Names for shallow portion.
%
basename = ['7306.nmfs   '; ...
            '7307.nmfs   '; ...
            '7405.nodc.qc'; ...
            '7406.nmfs   '; ...
            '7407.mwra   '];

% Names for deep portion.
%
%basename = ['7305.nodc.qc'; ...
%            '7306.nmfs   '; ...
%            '7307.nmfs   '; ...
%            '7405.nodc.qc'; ...
%            '7406.nmfs   '; ...
%            '7407.nmfs   '];

mincnt = 5;

in_suff  = '.nc';
out_suff = '.mods';
if (mincnt>1),
   out_suff = ['.',num2str(mincnt),out_suff];
end;

out_suff = ['.p25',out_suff];

ttl = 'LOC 1/4 degree climatology, June';
if (mincnt>1),
   ttl = [ttl,', ',num2str(mincnt),' data vals. min'];
end;

time = [2001 6 15 12 0 0];

julian_off = 2440000;

%-------------------------------------------------------------------------------
% Set julian date for casts.
%-------------------------------------------------------------------------------

jdate = julian(time) - julian_off;

clear time;

%-------------------------------------------------------------------------------
% Set netCDF parameters.
%-------------------------------------------------------------------------------

status = mexcdf ('setopts',0);

%-------------------------------------------------------------------------------
% Loop over all files.
%-------------------------------------------------------------------------------

[nfile nchar] = size(basename);

clear nchar

for m = 1:nfile,

   %=========================
   %=== Open netCDF file. ===
   %=========================

   ncfile =  [deblank(basename(m,:)),in_suff];
   disp (' ');
   disp (['   Working on: ',ncfile]);
   ncid   = netcdf (ncfile);

   if isempty(ncid),
      disp (' ');
      disp ('***Error:  LOC2MODS - unable to open file:');
      disp (['           ',setstr(34),ncfile,setstr(34)]);
      disp (' ');
      return;
   end;

   %============================
   %=== Get file dimensions. ===
   %============================

   %----------------------------------
   %--- Number of latitude points. ---
   %----------------------------------

   name = 'lat';
   nlat = length(ncid(name));

   if (nlat==0),
      disp (' ');
      disp (['***Error:  LOC2MODS - unable to read dimension ',setstr(34), ...
                                            name,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncfile,setstr(34)]);
      disp (' ');
      return;
   end;

   %-----------------------------------
   %--- Number of longitude points. ---
   %-----------------------------------

   name = 'lon';
   nlon = length(ncid(name));

   if (nlon==0),
      disp (' ');
      disp (['***Error:  LOC2MODS - unable to read dimension ',setstr(34), ...
                                            name,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncfile,setstr(34)]);
      disp (' ');
      return;
   end;

   %------------------------------
   %--- Number of time levels. ---
   %------------------------------

   name = 'time';
   ntim = length(ncid(name));

   if (ntim==0),
      disp (' ');
      disp (['***Error:  LOC2MODS - unable to read dimension ',setstr(34), ...
                                            name,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncfile,setstr(34)]);
      disp (' ');
      return;
   end;

   %========================
   %=== Get fill values. ===
   %========================

   %-------------------------------
   %--- Temperature fill value. ---
   %-------------------------------

   name = 'te';
   temp_fill = ncid{name}.FillValue_(:);

   if isempty(temp_fill),
      disp (' ');
      disp (['***Error:  LOC2MODS - unable to read attribute ',setstr(34), ...
                               name,':_FillValue',setstr(34),' in file:']);
      disp (['           ',setstr(34),ncfile,setstr(34)]);
      disp (' ');
      return;
   end;

   %----------------------------
   %--- Salinity fill value. ---
   %----------------------------

   name = 'sa';
   salt_fill = ncid{name}.FillValue_(:);

   if isempty(salt_fill),
      disp (' ');
      disp (['***Error:  LOC2MODS - unable to read attribute ',setstr(34), ...
                               name,':_FillValue',setstr(34),' in file:']);
      disp (['           ',setstr(34),ncfile,setstr(34)]);
      disp (' ');
      return;
   end;

   %==============================================
   %=== Extract useful casts from netCDF file. ===
   %==============================================

   %----------------------------
   %--- Extract cast depths. ---
   %----------------------------

   name = 'de';
   zlev = ncid{name}(:);

   if isempty(zlev),
      disp (' ');
      disp (['***Error:  LOC2MODS - unable to read variable ',setstr(34), ...
                                           name,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncfile,setstr(34)]);
      disp (' ');
      return;
   end;

   %--------------------------------------------
   %---       Extract ending latitude.       ---
   %--- yes, those bozos decrement latitude. ---
   %--------------------------------------------

   name = 'latmax';
   eval(['latmax = ncid.',name,'(:);']);

   if isempty(latmax),
      disp (' ');
      disp (['***Error:  LOC2MODS - unable to read attribute ',setstr(34), ...
                                        ':',name,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncfile,setstr(34)]);
      disp (' ');
      return;
   end;

   %-----------------------------------
   %--- Extract latitude step size. ---
   %-----------------------------------

   name = 'latincr';
   eval(['dlat = ncid.',name,'(:);']);

   if isempty(dlat),
      disp (' ');
      disp (['***Error:  LOC2MODS - unable to read attribute ',setstr(34), ...
                                        ':',name,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncfile,setstr(34)]);
      disp (' ');
      return;
   end;

   %-----------------------------------
   %--- Extract starting longitude. ---
   %-----------------------------------

   name = 'lonmin';
   eval(['lonmin = ncid.',name,'(:);']);

   if isempty(lonmin),
      disp (' ');
      disp (['***Error:  LOC2MODS - unable to read attribute ',setstr(34), ...
                                        ':',name,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncfile,setstr(34)]);
      disp (' ');
      return;
   end;

   %------------------------------------
   %--- Extract longitude step size. ---
   %------------------------------------

   name = 'lonincr';
   eval(['dlon = ncid.',name,'(:);']);

   if isempty(dlon),
      disp (' ');
      disp (['***Error:  LOC2MODS - unable to read attribute ',setstr(34), ...
                                        ':',name,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncfile,setstr(34)]);
      disp (' ');
      return;
   end;

   %----------------------------------------
   %--- Loop over all positions & times. ---
   %----------------------------------------

   castid = 0;

   for n = 1:ntim,
      for j = 1:nlat,
         for i = 1:nlon,

            zwk = zlev';

            %----------------------------
            %--- Extract temperature. ---
            %----------------------------

            name = 'te';
            twk = squeeze(ncid{name}(n,j,i,:));

            if isempty(twk),
               disp (' ');
               disp (['***Error:  LOC2MODS - unable to read variable ', ...
                           setstr(34),name,setstr(34),' in file:']);
               disp (['           ',setstr(34),ncfile,setstr(34)]);
               disp (['           (i,j,n) = (',num2str(i),',', ...
                              num2str(j),',',num2str(n),')']);
               disp (' ');
               return;
            end;

            if (mincnt>1),
               name = [name,'_cnt'];
               tcnt = squeeze(ncid{name}(n,j,i,:));

               if isempty(tcnt),
                  disp (' ');
                  disp (['***Error:  LOC2MODS - unable to read variable ', ...
                              setstr(34),name,setstr(34),' in file:']);
                  disp (['           ',setstr(34),ncfile,setstr(34)]);
                  disp (['           (i,j,n) = (',num2str(i),',', ...
                                 num2str(j),',',num2str(n),')']);
                  disp (' ');
                  return;
               end;
            end;

            %-------------------------
            %--- Extract salinity. ---
            %-------------------------

            name = 'sa';
            swk = squeeze(ncid{name}(n,j,i,:));

            if isempty(swk),
               disp (' ');
               disp (['***Error:  LOC2MODS - unable to read variable ', ...
                           setstr(34),name,setstr(34),' in file:']);
               disp (['           ',setstr(34),ncfile,setstr(34)]);
               disp (['           (i,j,n) = (',num2str(i),',', ...
                               num2str(j),',',num2str(n),')']);
               disp (' ');
               return;
            end;

            if (mincnt>1),
               name = [name,'_cnt'];
               scnt = squeeze(ncid{name}(n,j,i,:));

               if isempty(scnt),
                  disp (' ');
                  disp (['***Error:  LOC2MODS - unable to read variable ', ...
                              setstr(34),name,setstr(34),' in file:']);
                  disp (['           ',setstr(34),ncfile,setstr(34)]);
                  disp (['           (i,j,n) = (',num2str(i),',', ...
                                 num2str(j),',',num2str(n),')']);
                  disp (' ');
                  return;
               end;
            end;

            %------------------------
            %--- Remove bad data. ---
            %------------------------

            if (mincnt>1),
               ind = find ( (zwk<0) | (twk==temp_fill) | (swk==salt_fill) | ...
                            (tcnt<mincnt) | (scnt<mincnt) );
             else
               ind = find ( (zwk<0) | (twk==temp_fill) | (swk==salt_fill) );
            end;

            if (~isempty(ind)),
               no_pt  = length(zwk);
               if (length(ind)==no_pt),
                  zwk = [];
                  twk = [];
                  swk = [];
                elseif (min(ind)~=1),
                  ind2 = min(ind):no_pt;
                  zwk(ind2) = [];
                  twk(ind2) = [];
                  swk(ind2) = [];
                  clear ind2;
                else,
                  dind = diff(ind);
                  nst  = min(find(dind>1));
                  ind2 = ind(nst+1):no_pt;
                  zwk(ind2) = [];
                  twk(ind2) = [];
                  swk(ind2) = [];
                  ind2 = ind(1:nst);
                  zwk(ind2) = [];
                  twk(ind2) = [];
                  swk(ind2) = [];
                  clear dind ind2 nst;
               end;
               clear no_pt;
            end;

            %------------------
            %--- Save cast. ---
            %------------------

            if (~isempty(zwk)),

               castid = castid + 1;

               hinfo(castid,1) = 3;
               hinfo(castid,2) = length(zwk);
               hinfo(castid,3) = castid;
               hinfo(castid,4) = lonmin + (i-0.5).*dlon;
               hinfo(castid,5) = latmax - (j-0.5).*dlat;

               name = 'bottom';
               zbot = squeeze(ncid{name}(n,j,i));

               if isempty(zbot),
                  disp (' ');
                  disp (['***Error:  LOC2MODS - unable to read variable ', ...
                                 setstr(34),name,setstr(34),' in file:']);
                  disp (['           ',setstr(34),ncfile,setstr(34)]);
                  disp (['           (i,j,n) = (',num2str(i),',',num2str(j), ...
                                                ',',num2str(n),')']);
                  disp (' ');
                  return;
               end;

               hinfo(castid,6) = zbot;
               hinfo(castid,7) = jdate;
               hinfo(castid,8:10) = [0.1 1e-3 1e-3];
               hinfo(castid,11) = 0;

               dummy = [setstr(39),'CTD: z t s',setstr(39)];
               ind = 1:length(dummy);
               htype(castid,ind) = dummy;

               ind = 1:length(zwk);
               z(castid,ind) = zwk;
               t(castid,ind) = twk;
               s(castid,ind) = swk;

               zwk = [];
               twk = [];
               swk = [];

            end;
         end;
      end;
   end;

   clear castid dlat dlon i j latmax lonmin n name nlat nlon ntim ...
         salt_fill swk temp_fill twk zbot zwk zlev;
   if (mincnt>1)
      clear scnt tcnt
   end;

   %=====================================
   %=== Close input netCDF data file. ===
   %=====================================

   status = close (ncid);

   if ~isempty(status),
      disp (' ');
      disp ('***Error:  LOC2MODS - unable to close file:');
      disp (['           ',setstr(34),ncfile,setstr(34)]);
      disp (' ');
      return;
   end;

   clear ncfile status ncid;

   %========================
   %=== Write cast file. ===
   %========================

   if (exist('z')==1),

      %---------------------------
      %--- Create file header. ---
      %---------------------------

      dummy = [' title = ',ttl,':  ',deblank(basename(m,:))];
      ind = 1:length(dummy);
      header (1,ind) = dummy;

      [nsta, mxpts] = size(z);
      dummy = [' stations = ',num2str(nsta)];
      ind = 1:length(dummy);
      header (2,ind) = dummy;

      dummy = [' str_time = ',num2str(jdate,16),', ',pjh_greg(jdate)];
      ind = 1:length(dummy);
      header (3,ind) = dummy;

      dummy = [' end_time = ',num2str(jdate,16),', ',pjh_greg(jdate)];
      ind = 1:length(dummy);
      header (4,ind) = dummy;

      dummy = [' Jday_offset = ',num2str(julian_off,16)];
      ind = 1:length(dummy);
      header (5,ind) = dummy;

      dummy = [' lng_min = ',num2str(min(hinfo(:,4)),16)];
      ind = 1:length(dummy);
      header (6,ind) = dummy;

      dummy = [' lng_max = ',num2str(max(hinfo(:,4)),16)];
      ind = 1:length(dummy);
      header (7,ind) = dummy;

      dummy = [' lat_min = ',num2str(min(hinfo(:,5)),16)];
      ind = 1:length(dummy);
      header (8,ind) = dummy;

      dummy = [' lat_max = ',num2str(max(hinfo(:,5)),16)];
      ind = 1:length(dummy);
      header (9,ind) = dummy;

      dummy = ' format = ascii, record interleaving';
      ind = 1:length(dummy);
      header (10,ind) = dummy;

      dummy = ' type = CTD';
      ind = 1:length(dummy);
      header (11,ind) = dummy;

      dummy = ' fields = depth, temperature, salinity';
      ind = 1:length(dummy);
      header (12,ind) = dummy;

      dummy = ' units  = meter, Celsius, PSU';
      ind = 1:length(dummy);
      header (13,ind) = dummy;

      dummy = [' creation_date = ',date_stamp];
      ind = 1:length(dummy);
      header (14,ind) = dummy;

      dummy = 'END';
      ind = 1:length(dummy);
      header (15,ind) = dummy;

      clear dummy ind mxpts nsta;

   %------------------------
   %--- Write cast file. ---
   %------------------------

      hydnam = [deblank(basename(m,:)),out_suff];
      disp (['   ... Creating: ',hydnam]);
      status = whydro (hydnam,header,hinfo,htype,z,t,s);

      if (status<0),
         disp (' ');
         disp ('***Error:  LOC2MODS - unable to write to file:');
         disp (['           ',setstr(34),hydnam,setstr(34)]);
         disp (' ');
         return;
      end;

      clear hydnam header hinfo htype z t s status;

    else

      disp (['   ... Eliminated all data from: ', ...
             [deblank(basename(m,:)),in_suff]]);

   end;

end;

clear basename in_suff jdate julian_off m mincnt nfile out_suff ttl
