function status = rm_ovrlp (fnefiles,fneres,crsin,crsout);
%
% function status = rm_ovrlp (fnefiles,fneres,crsin,crsout);
%
% This function removes coarse data positions which overlap fine positions.
%
% ------
% Input:
% ------
%
%    FNEFILES   Names of fine resolution files.
%    FNERES     Resolutions of fine resolution files.
%    CRSIN      Name of original coarse resolution file.
%    CRSOUT     Name for new coarse resolution file with no overlap.
%
% -------
% Output:
% -------
%
%    STATUS     Exit status.
%                 [-1]  Error creating new file.
%                  [0]  Normal exit.
%                  [1]  No coarse data eliminated.
%                  [2]  All coarse data eliminated.

%-------------------------------------------------------------------------------
% Set output & counters.
%-------------------------------------------------------------------------------

status = 0;
nfine  = length(fneres);
nrm    = 0;

%-------------------------------------------------------------------------------
% Read in coarse data.
%-------------------------------------------------------------------------------

[header,hinfo,htype,z,t,s] = rhydro (crsin,1);

loncrs = hinfo(:,4);
latcrs = hinfo(:,5);

%-------------------------------------------------------------------------------
% Remove overlap with each fine resolution file.
%-------------------------------------------------------------------------------

n = 0;

while ( (n<nfine) & ~isempty(loncrs))

   %--------------------------
   %--- Increment counter. ---
   %--------------------------

   n = n + 1;

   %------------------------------------------
   %--- Read current fine resolution file. ---
   %------------------------------------------

   ind = find(~isspace(fnefiles(n,:)));

   [head,hin,hty,zf,tf,sf] = rhydro (fnefiles(n,ind),1);

   lonfne = hin(:,4);
   latfne = hin(:,5);

   clear head hin hty zf tf sf ind;

   %---------------------------------------------
   %--- Remove coarse data overlap with fine. ---
   %---------------------------------------------

   rmind = find_overlap (lonfne,fneres(n),latfne,fneres(n),loncrs,latcrs);

   if ~isempty(rmind)
      nrm            = nrm + length(rmind);
      hinfo(rmind,:) = [];
      htype(rmind,:) = [];
      z(rmind,:)     = [];
      t(rmind,:)     = [];
      s(rmind,:)     = [];
      loncrs(rmind)  = [];
      latcrs(rmind)  = [];
   end;

end;

%-------------------------------------------------------------------------------
% Terminate under special cases.
%-------------------------------------------------------------------------------

if (nrm==0)
   status = 1;
end;

if isempty(loncrs)
   status = 2;
end;

if (status~=0)
   clear header hinfo htype z t s loncrs latcrs
   return;
end;

%-------------------------------------------------------------------------------
% Update file header.
%-------------------------------------------------------------------------------

nlst  = max(find(~isspace(header(1,:))));
dummy = [header(1,1:nlst),' (no overlap)'];
header(1,:) = setstr(zeros(size(header(1,:))));
ind = 1:length(dummy);
header (1,ind) = dummy;

[nsta, mxpts] = size(z);
dummy = [' stations = ',num2str(nsta)];
header(2,:) = setstr(zeros(size(header(2,:))));
ind = 1:length(dummy);
header (2,ind) = dummy;

jdate = min(hinfo(:,7));
dummy = [' str_time = ',num2str(jdate,16),', ',pjh_greg(jdate)];
header(3,:) = setstr(zeros(size(header(3,:))));
ind = 1:length(dummy);
header (3,ind) = dummy;

jdate = max(hinfo(:,7));
dummy = [' end_time = ',num2str(jdate,16),', ',pjh_greg(jdate)];
header(4,:) = setstr(zeros(size(header(4,:))));
ind = 1:length(dummy);
header (4,ind) = dummy;

dummy = [' lng_min = ',num2str(min(hinfo(:,4)),16)];
header(6,:) = setstr(zeros(size(header(6,:))));
ind = 1:length(dummy);
header (6,ind) = dummy;

dummy = [' lng_max = ',num2str(max(hinfo(:,4)),16)];
header(7,:) = setstr(zeros(size(header(7,:))));
ind = 1:length(dummy);
header (7,ind) = dummy;

dummy = [' lat_min = ',num2str(min(hinfo(:,5)),16)];
header(8,:) = setstr(zeros(size(header(8,:))));
ind = 1:length(dummy);
header (8,ind) = dummy;

dummy = [' lat_max = ',num2str(max(hinfo(:,5)),16)];
header(9,:) = setstr(zeros(size(header(9,:))));
ind = 1:length(dummy);
header (9,ind) = dummy;

dummy = [' creation_date = ',date_stamp];
header(14,:) = setstr(zeros(size(header(14,:))));
ind = 1:length(dummy);
header (14,ind) = dummy;

%-------------------------------------------------------------------------------
% Write new file.
%-------------------------------------------------------------------------------

status = whydro (crsout,header,hinfo,htype,z,t,s);

if (status~=0)
   status = -1;
end;

clear header hinfo htype z t s
