/*  ctdconvert.c
................................................................................
.   Reads WHOI ctd (NODCexch format file(s))
.   extracts :    header info
.                 p,t,s,ox,  
.   applies a nominal filter to each (5 pts for p,t,s; 11 pts for ox)
.   and outputs p,d,t,s and o at the specified prs_int to the output
.   file.
.
.   USAGE: ctdconvert infile_list -Ooutfile -P<prs_int> [-D<dir>] [-E<extent>]
................................................................................
*/

#include <stdio.h>
#include <math.h>
#include <string.h>
#include <malloc.h>
#include "hydrobase.h"


#define   LF        0x0a   /* ascii code for linefeed char */
#define   MISSING   -9.0   /* missing value flag */
#define   FILTWIDTH  5     /* # of pts in gaussian filter */
#define   FILTOX    11     /* # of pts in gaussian filter */

#define   DIR    ""
#define   EXTENT ""

  /* define possible modes for output files ... */
#define   OVERWRITE 0
#define   NO_CLOB   1
#define   APPEND    2


  /* globally defined variables */
  
struct HYDRO_HDR  hdr;
struct HYDRO_DATA data;
double *p, *d, *t, *s, *o;
int deltap;          /* pressure interval of ctd cast */

/*  functions in hydro_utils.c */

extern int create_hydro_file();
extern int write_hydro_station();
extern int ms10();

/*  functions in filters.c */
extern void filter();

main (argc, argv)
int  argc;
char **argv;
{
   int error, nobs, nprops;
   int outfile;
   int  i, j, curfile = 1, nfiles = 0; 
   short oflag, pflag; 
   char  ox_avail;
   char *dir, *extent;
   double dlat;
   int prsint, npts;
   FILE *infile, *openfile();
   void print_usage();
   double depth_();
   void readdata();
   int parse_header();
   
/* check for command line arguments */

   if (argc < 2) {
     print_usage(argv[0]);
     exit(1);
   }
   
 
/*  set these default values */

    dir = DIR;
    extent = EXTENT;
    error = 0;
    pflag = oflag  = 0;
 
 
/*  parse the command line arguments */

   for (i = 1; i < argc; i++) {
      if (argv[i][0] == '-') {
            switch (argv[i][1]) {
               case 'D':                   /* get input dir */
                        dir = &argv[i][2];
                        break;
               case 'E':                    /* get file extent */
                        extent = &argv[i][2];
                        break;

                case 'O':                    /* get output file  */
                        oflag = 1;
                        outfile = create_hydro_file(&argv[i][2], OVERWRITE);
                        if (outfile < 1) {
                           fprintf(stderr,"\nError opening output file: %s\n", 
                                  &argv[i][2]);
                           exit(1);
                        }
                        break;

               case 'P':                    /* get pressure interval */
                        pflag = 1;
                        error = (sscanf(&argv[i][2],"%d", &prsint) != 1);
                        if ((prsint % 2) > 0) {
                          prsint = (prsint >> 1) << 1;
                          fprintf(stderr,"\n Pressure interval must be a multiple of 2.");
                          fprintf(stderr,"\n Using %d db.\n", prsint);
                        }
                        break;

               default:
                        error = 1;

          }    /* end switch */

          if (error ) {
             fprintf(stderr,"\nError parsing command line args.\n");
             fprintf(stderr,"     in particular: '%s'\n", argv[i]);
             exit(1);
          }

       }  /* end if */

       else  {
           ++nfiles;
       }
   }  /* end for */

   if (! (oflag && nfiles && pflag)) {
       fprintf(stderr,"\nYou must specify input file(s), output file and pressure interval.\n");
       exit(1);
   }

/* loop for each input file */

   do {
   
     infile = openfile(dir, argv[curfile], extent);
     if (infile == NULL) goto NEXTFILE;
      
      /* parse the header info */
      
     nobs = parse_header(infile);
     nprops = hdr.nprops;  
     ox_avail = (nprops > 4) ? 1 : 0;   
     
     /* adjust some of the header info ...   */
     
     dlat = (double) hdr.lat;
     strncpy(hdr.country,"--",3);
     hdr.ms10 = ms10(hdr.lat, hdr.lon, &hdr.ms1);
     hdr.prop_id = (int *) malloc(hdr.nprops * sizeof(int));
     hdr.prop_id[0] = (int)PR;
     hdr.prop_id[1] = (int)DE;
     hdr.prop_id[2] = (int)TE;
     hdr.prop_id[3] = (int)SA;
     if (ox_avail) 
        hdr.prop_id[4] = (int)OX;
     
     /*  allocate space */
     
     p = (double *) malloc(nobs * sizeof(double));
     d = (double *) malloc(nobs * sizeof(double));
     t = (double *) malloc(nobs * sizeof(double));
     s = (double *) malloc(nobs * sizeof(double));
     if (ox_avail) 
        o = (double *) malloc(nobs * sizeof(double));
        
    /* read and filter each property */    
        
     readdata(infile, nprops, nobs);
     filter('g',p, nobs, FILTWIDTH);
     filter('g',t, nobs, FILTWIDTH);
     filter('g',s, nobs, FILTWIDTH);
     if (ox_avail)
        filter('g',o, nobs, FILTOX);
        

     /* decimate the arrays according to specified prsint */
     
     npts = prsint / deltap;  /*  get # of pts per interval*/
     if (npts == 0) {
        fprintf(stderr,"Bad number of points per interval: %d \n", npts);
        exit(1);
     }
    
     j = 0;
     for (i = 0; i < nobs; i += npts) {
       p[j] = p[i];
       d[j] = depth_(&p[i], &dlat);
       t[j] = t[i];
       s[j] = s[i];
       if (ox_avail)
          o[j] = o[i];
       ++j;
     }
     
     /* add bottom observation  ... */
     
     if ((i - npts) != (--nobs)) {
        p[j] = p[nobs];
        d[j] = depth_(&p[nobs], &dlat);
        t[j] = t[nobs];
        s[j] = s[nobs];
        if (ox_avail)
           o[j] = o[nobs];
        ++j;
     }
     
     hdr.nobs = data.nobs = j;
     data.nprops = hdr.nprops; 
     
     data.observ[(int)PR] = p;   /* set these pointers */
     data.observ[(int)DE] = d;
     data.observ[(int)TE] = t;
     data.observ[(int)SA] = s;
     if (ox_avail)
        data.observ[(int)OX] = o;
        
     if (hdr.nobs > 0 )
        write_hydro_station(outfile, hdr, data);
     
     free(hdr.prop_id);   
     free(p);
     free(d);
     free(t);
     free(s);
     if (ox_avail)
       free(o);

NEXTFILE:
         fclose(infile);

 
   } while (curfile++ < nfiles );

   exit(0);
        
} /* end main() */ 
  
/*****************************************************************************/

void print_usage(program)
char *program;
{
   fprintf(stderr,"\nUsage:  %s filelist -Ooutfile -Pprs_int [-D<dirname>] [-E<file_extent>]", program);
   fprintf(stderr,"\n\n  List of filenames MUST be first argument!");
   fprintf(stderr,"\n    -D  : specifies dirname for input files (default is current directory) ");
   fprintf(stderr,"\n            ex: -D../data/ ");
   fprintf(stderr,"\n    -E  : specifies input_file extent (default is no extent)");  
   fprintf(stderr,"\n            ex: -E.dat ");

   fprintf(stderr,"\n\n");  
   return;
}
   
/*****************************************************************************/
FILE *openfile(dir, root, extent)
char *dir;
char *root;
char *extent;
{
   char st[80];
   int i;
   FILE *infile;
   
   strcpy(st, dir);
   if ((i=strlen(dir)) != 0) {
      if (dir[i-1] != '/')
         strncat(st,"/",1);
   }
   strcat(st, root);
   strcat(st, extent);
   infile = fopen(st,"r");
   if (infile == NULL)
         fprintf(stderr,"\n Unable to open %s \n", st);
   else
         fprintf(stderr,"\n   opened %s ...\n", st);
   
   return(infile);
   
}  /* end openfile() */
/*****************************************************************************/
int parse_header(fptr)
FILE *fptr;
   /* reads the header info from an already opened ctd data file and returns
      the number of obs.  The info is used to fill in appropriate values in
      the struct HYDRO_HDR hdr, a global variable.  */
{
   int n=0, error;
   char line[80];
   
   /* line 1 */
   if (fscanf(fptr,"%[^\n]", line) != 1) {
        fprintf(stderr,"\n Error attempting to read 1st header line. \n");
        exit(1);   
   } 
   if (sscanf(&line[6],"%2s", hdr.ship) != 1) {
        fprintf(stderr,"\n error in sscanf attempt to read shipcode \n");
        exit(1);
   }
   if (sscanf(&line[15],"%d", &hdr.cruise) != 1) {
        fprintf(stderr,"\n error in sscanf attempt to read cruise # \n");
        exit(1);
   }   
   if (sscanf(&line[24],"%d", &hdr.station) != 1) {
        fprintf(stderr,"\n error in sscanf attempt to read station # \n");
        exit(1);
   } 
   error = getc(fptr);
   if (error != (int)'\n')
        fprintf(stderr,"\n expected a LF(0x10) but got %x in parse_header()\n",
                error);
      
   
   /* line 2 */
   if (fscanf(fptr,"%[^\n]", line) != 1) {
        fprintf(stderr,"\n Error attempting to read 2nd header line. \n");
        exit(1);   
   }    
   if (sscanf(&line[5],"%d-%d-%d", &hdr.year, &hdr.month, &hdr.day) != 3) {
        fprintf(stderr,"\n error in sscanf attempt to read date \n");
        exit(1);
   } 
   error = getc(fptr);
    
    
   /* line 3 */
   if (fscanf(fptr,"%[^\n]", line) != 1) {
        fprintf(stderr,"\n Error attempting to read 3rd header line. \n");
        exit(1);   
   }         
   error = getc(fptr);
     
    
   /* line 4 */
   if (fscanf(fptr,"%[^\n]", line) != 1) {
        fprintf(stderr,"\n Error attempting to read 4th header line. \n");
        exit(1);   
   }
   if (sscanf(&line[10],"%f", &hdr.lat) != 1) {
        fprintf(stderr,"\n error in sscanf attempt to read lat \n");
        exit(1);
   } 
   if (sscanf(&line[24],"%f", &hdr.lon) != 1) {
        fprintf(stderr,"\n error in sscanf attempt to read lon \n");
        exit(1);
   } 
   error = getc(fptr);
    
   /* line 5 */
   if (fscanf(fptr,"%[^\n]", line) != 1) {
        fprintf(stderr,"\n Error attempting to read 5th header line. \n");
        exit(1);   
   }        
   error = getc(fptr);
      
    /* line 6 */
   if (fscanf(fptr,"%[^\n]", line) != 1) {
        fprintf(stderr,"\n Error attempting to read 6th header line. \n");
        exit(1);   
   }        
   error = getc(fptr);
   /* line 7 */
   if (fscanf(fptr,"%[^\n]", line) != 1) {
        fprintf(stderr,"\n Error attempting to read 7th header line. \n");
        exit(1);   
   }   
   error = getc(fptr);
   /* line 8 */
   if (fscanf(fptr,"%[^\n]", line) != 1) {
        fprintf(stderr,"\n Error attempting to read 8th header line. \n");
        exit(1);   
   }    
   if (sscanf(&line[27],"%d", &hdr.pdr) != 1) {
        fprintf(stderr,"\n error in sscanf attempt to read pdr \n");
        exit(1);
   } 
   error = getc(fptr);
        
   /* line 9 */
   if (fscanf(fptr,"%[^\n]", line) != 1) {
        fprintf(stderr,"\n Error attempting to read 9th header line. \n");
        exit(1);   
   }    
   error = getc(fptr);
   if (sscanf(&line[5],"%d", &deltap) != 1) {
        fprintf(stderr,"\n error in sscanf attempt to read pr_int \n");
        exit(1);
   }
   /* line 10 */
   if (fscanf(fptr,"%[^\n]", line) != 1) {
        fprintf(stderr,"\n Error attempting to read 10th header line. \n");
        exit(1);   
   }    
   if (sscanf(&line[5],"%d", &hdr.nobs) != 1) {
        fprintf(stderr,"\n error in sscanf attempt to read nobs \n");
        exit(1);
   } 
   if (sscanf(&line[17],"%d", &hdr.nprops) != 1) {
        fprintf(stderr,"\n error in sscanf attempt to read nobs \n");
        exit(1);
   } 
   error = getc(fptr);
   /* line 11 */
   if (fscanf(fptr,"%[^\n]", line) != 1) {
        fprintf(stderr,"\n Error attempting to read 11th header line. \n");
        exit(1);   
   }    
   error = getc(fptr);

   return(hdr.nobs);
}  /* end parse_header() */
/*****************************************************************************/
void readdata(fptr, nprops, nobs)
FILE *fptr;
int nprops, nobs;  
{
   int i, error;
   char line[80];
   
   for (i = 0; i < nobs; ++i) {
      error = fscanf(fptr, "%[^\n]", line) != 1;
      if (error){
         fprintf(stderr,"\nError reading properties at data line #%d.\n", i);
         exit(1);
      }
      error = getc(fptr);
      
      error = sscanf(&line[0], "%lf", &p[i]) != 1;
      if (nprops > 2) 
         error += sscanf(&line[9], "%lf", &t[i]) != 1;
      if (nprops > 3) 
         error += sscanf(&line[21], "%lf", &s[i]) != 1;
      if (nprops > 4) 
         error += sscanf(&line[33], "%lf", &o[i]) != 1;
      
      if (error){
         fprintf(stderr,"\nError reading %d properties at data line #%d.\n",
            error, i);
         exit(1);
      }
   }
   return;
}  /* end readdata() */
