/* extract.c
................................................................................
.  Extracts stations from the HydroBase files by a variety of criteria 
.
Usage:  extract  filename_root(s) -T<p[^]/minlon/maxlon/minlat/maxlat>/<m[^]/m1/m2/...>/<y[^]/minyear/maxyear> [-Ddirname] [-Eextent] [-H] > outfile

    -D  : specifies dirname (default is ./) 
        ex: -D../data/ 
    -E  : specifies file extent (default is no extent)
        ex: -E.dat 
    -H  : extract the header information only 
    -T  : specifies type of extraction and criteria
        n = nation (country code)
        s = ship
        c = cruise
        d = station depth (deepest observation)
        g = geographic position  ex:  -Tp/-80/-10/0/60
        m = month  ex:  -Tm/1/2/3/12
        y = year   ex:  -Ty/1965/1980
        combination of above:  -Tp/-80/-10/0/160/m/1/2/3/12

        Use ^ after the  extraction type to extract all stations EXCEPT
        the ones which meet the criteria.  ex: -Ts^VE/y1982 will extract
        all stations from 1982 except those collected by the ship VE.

.          
. Output: to stdout device
................................................................................
................................................................................

*/
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "hydrobase.h"

#define    DIR     ""
#define    EXTENT   ""
#define    MAXCRIT  12   /* maximum # of criteria for any of the options */
#define    STDOUT    1
#define    PRINT_MSG 1

extern int open_hydro_file();     /* in hydro_utils.c */
extern int get_station();
extern int write_hydro_station();

main (argc, argv)
int   argc;
char *argv[];
{
   int     curfile = 1, nfiles = 0; 
   int     i, infile;
   int     g_flag=0, h_flag=0, m_flag=0, y_flag=0, d_flag=0;
   int     n_flag=0, s_flag=0, c_flag=0, o_flag=0;
   int     ld_flag=0, lp_flag=0, xdateline = 0;
   int     clist[MAXCRIT], ncru = 0;
   int     ncountry=0, nship = 0;
   char    ship[MAXCRIT][3], country[MAXCRIT][3];
   int     status, staOK, topt = 0;
   int     nout = 0, nskip = 0;
   int     mask, month, error = 0;
   int     minyr, maxyr;
   int     mindepth, maxdepth;
   int     outcastfile;
   float   top, bot, left, right;
   float   minlev, maxlev;
   struct HYDRO_HDR h;
   struct HYDRO_DATA data;
   char   *dir, *extent;
   char   *s;
   int     checkpos(), checkmonth(), checkyear();
   int     checkship(), checkcruise(), checknation();
   int     checkdepth(), checklevs();
   void    print_usage(), report_status();

/* are there command line arguments? */

   if (argc < 2) {
      print_usage(argv[0]);
      exit(1);
   }

/*  set these default values */

    dir = DIR;
    extent = EXTENT;

/*  parse the command line arguments */

   for (i = 1; i < argc; i++) {
      if (argv[i][0] == '-') {
            switch (argv[i][1]) {
               case 'D':
                        dir = &argv[i][2];
                        break;
               case 'E':
                        extent = &argv[i][2];
                        break;
               case 'H':
                        h_flag = 1;
                        break;

               case 'O':
                        o_flag = 1;
                        outcastfile = create_hydro_file(&argv[i][2], APPEND);
                        break;
               case 'T':
                        topt = 1;
                        s = &argv[i][2];
                        do {
                          switch (*s) {
                            case 'c' :
                                     c_flag = 1;
                                     ++s;
                                     if (*s == '^') {
                                          c_flag = -1;
                                          ++s;
                                     }
                                     if (*s == '/')
                                          ++s;
                                     
                                     do {
                                       error = sscanf(s,"%d", &clist[ncru++]) != 1;
                                       if (error)   
                                           break;
                                       if ((s = strchr(s,'/')) == NULL) 
                                          break;
                                       if (*s == '/')
                                         ++s;
                                     } while ((ncru < MAXCRIT) && (isdigit(*s)));
                                     error = (ncountry >= MAXCRIT);
                                     break;
                            case 'd' :
                                     d_flag = 1;
                                     ++s;  /* move past d */
                                     if (*s == '^') {
                                          d_flag = -1;
                                          ++s;
                                     }
                                     if (*s == '/')
                                          ++s;

                                     error += (sscanf(s,"%d", &mindepth) != 1);
                                     s = strchr(s,'/');  /* check for another delimiter*/
                                     error = (s == NULL);
                                     if (s != NULL) {
                                        ++s;  /* move past delimiter */
                                        error += (sscanf(s,"%d", &maxdepth) != 1);
                                     }
                                     if (error) break;
                                     if (maxdepth < mindepth) {
                                        fprintf(stderr,"\nmindepth exceeds maxdepth\n");
                                        exit(1);
                                     }
                                     if ((s = strchr(s,'/')) == NULL)  /* check for another delimiter */
                                          break;

                                     if (*s == '/')
                                          ++s;
                                     break;
                            case 'l' :
                                     ++s;  /* move past l */
                                     if (*s == 'd')
                                         ld_flag = 1;
                                     else if (*s == 'p')
                                         lp_flag = 1;
                                     else {
                                         fprintf(stderr, "\n Specify p or d after the 'l' extraction type.\n");
                                         exit(1);
                                     }
                                     ++s; 
                                     if (*s == '^') {
                                          if (ld_flag)
                                            ld_flag = -1;
                                          else
                                            lp_flag = -1;
                                          ++s;
                                     }
                                     if (*s == '/')
                                          ++s;
                                     error += (sscanf(s,"%f", &minlev) != 1);
                                     s = strchr(s,'/');  /* check for another delimiter*/
                                     error = (s == NULL);
                                     if (s != NULL) {
                                        ++s;  /* move past delimiter */
                                        error += (sscanf(s,"%f", &maxlev) != 1);
                                     }
                                     if (error) break;
                                     if (maxlev < minlev) {
                                        fprintf(stderr,"\nminlev exceeds maxlev\n");
                                        exit(1);
                                     }
                                     if ((s = strchr(s,'/')) == NULL)  /* check for another delimiter */
                                          break;

                                     if (*s == '/')
                                          ++s;

                                     break;
                            case 'n' :
                                     n_flag = 1;
                                     ++s;  /* move past n */
                                     if (*s == '^') {
                                          n_flag = -1;
                                          ++s;
                                     }
                                     if (*s == '/')
                                          ++s;
                                     do {
                                        error = (sscanf(s,"%2s", country[ncountry++]) != 1);
                                        if (error)   
                                           break;
                                        if ((s = strchr(s,'/')) == NULL) 
                                           break;
                                        if (*s == '/')
                                          ++s;
                                     }  while ((ncountry < MAXCRIT) && (isdigit(*s)));
                                     error += (ncountry >= MAXCRIT);
                                     break;
                            case 's' :
                                     s_flag = 1;
                                     ++s;  /* move past s */
                                     if (*s == '^') {
                                          s_flag = -1;
                                          ++s;
                                     }

                                     if (*s == '/')
                                          ++s;
                                     do {
                                        error = (sscanf(s,"%2s", ship[nship++]) != 1);
                                        if (error)   
                                           break;
                                        if ((s = strchr(s,'/')) == NULL) 
                                           break;
                                        if (*s == '/')
                                          ++s;
                                     } while ((nship < MAXCRIT) && (*s < 'a'));
                                     error = (nship >= MAXCRIT);
                                     break;
                            case 'm' :
                                     m_flag = 1;
                                     ++s;  /* move past m */
                                     if (*s == '^') {
                                          m_flag = -1;
                                          ++s;
                                     }

                                     if (*s == '/')
                                          ++s;
                                     do {
                                       error = sscanf(s,"%d", &month) != 1;
                                       if (error)   
                                           break;
                                       mask = (1 << --month) | mask;  /* bitwise or */
                                       if ((s = strchr(s,'/')) == NULL) 
                                          break;
                                       if (*s == '/')
                                         ++s;
                                     } while (isdigit(s[0]));
                                     break;
                            case 'g':
                                      g_flag = 1;
                                     ++s;  /* move past p */
                                     if (*s == '^') {
                                           g_flag = -1;
                                          ++s;
                                     }

                                     if (*s == '/')
                                          ++s;
                                     error = (sscanf(s,"%f", &left) != 1);
                                     while (*(++s) != '/')
                                            ;  
                                     ++s;  /* move past delimiter */
                                     error += (sscanf(s,"%f", &right) != 1);
                                     while (*(++s) != '/')
                                            ;  
                                     ++s;
                                     error += (sscanf(s,"%f", &bot) != 1);
                                     while (*(++s) != '/')
                                            ;  
                                     ++s;  /* move past delimiter */
                                     error += (sscanf(s,"%f", &top) != 1);
                                     
                                     if (left > 0 && right < 0)
                                         right += 360.;
                                     if (right > 180)
                                         xdateline = 1;

                                     if ((s = strchr(s,'/')) == NULL) 
                                          break;

                                     if (*s == '/')
                                          ++s;
                                     break;

                            case 'y' :
                                     y_flag = 1;
                                     ++s;  /* move past y */
                                     if (*s == '^') {
                                          y_flag = -1;
                                          ++s;
                                     }

                                     if (*s == '/')
                                          ++s;

                                     error += (sscanf(s,"%d", &minyr) != 1);
                                     s = strchr(s,'/');  /* check for another delimiter*/
                                     error = (s == NULL);
                                     if (s != NULL) {
                                        ++s;  /* move past delimiter */
                                        error += (sscanf(s,"%d", &maxyr) != 1);
                                     }
                                     if (error) break;
                                     if (maxyr < 1000) 
                                          maxyr +=  1900;
                                     if (minyr < 1000) 
                                          minyr += 1900;
                                     if (maxyr < minyr) {
                                        fprintf(stderr,"\nminyr exceeds maxyr\n");
                                        exit(1);
                                     }
                                     if ((s = strchr(s,'/')) == NULL)  /* check for another delimiter */
                                          break;

                                     if (*s == '/')
                                          ++s;
                                     break;
                             default :
                                     error = 1;

                          }  /* end switch */
  
                          if (error)  {
                              fprintf(stderr,"\nError parsing command line");
                              fprintf(stderr,"\n in particular: %s\n", argv[i]);
                              exit(1);          
                          }  
                         
                           
                        } while ((s != NULL) && (*s != '\0'));
                        break;
               default  :
                        fprintf(stderr,"\nError parsing command line");
                        fprintf(stderr,"\n in particular: %s\n", argv[i]);
                        exit(1);
            }  /* end switch */
       }  /* end if */
       else  {
           ++nfiles;
       }
   }  /* end for */

   if (!topt || !nfiles) {
       print_usage(argv[0]);
       fprintf(stderr,"\n\nYou must specify input file(s) and -T option(s)!\n");
       exit(1);
   }


 /* loop for each input file */

   do {

     infile = open_hydro_file(dir, argv[curfile], extent, PRINT_MSG);
     if (infile  < 0) 
       goto NEXTFILE;
     

     /* loop for each station */

     while ((status = get_station(infile, &h, &data)) == 0) { 

       if (xdateline && h.lon < 0)
          h.lon += 360.;
          
       staOK = 1;
       if ( g_flag && staOK){
               staOK = checkpos(h, top, bot, left, right);
               if ( g_flag < 0)
                   staOK = !staOK;
        }
       if (m_flag && staOK) {
               staOK = checkmonth(h, mask);
               if (m_flag < 0)
                   staOK = !staOK;
        }

       if (y_flag && staOK) {
               staOK = checkyear(h,minyr,maxyr);
               if (y_flag < 0)
                   staOK = !staOK;
        }
       if (c_flag && staOK) {
               staOK = checkcruise(h, clist, ncru);
               if (c_flag < 0)
                   staOK = !staOK;
        }
       if (d_flag && staOK) {
               staOK = checkdepth(h, data.observ[(int)DE], mindepth, maxdepth);
               if (d_flag < 0)
                   staOK = !staOK;
        }
       if (s_flag && staOK) {
               staOK = checkship(h, ship, nship);
               if (s_flag < 0)
                   staOK = !staOK;
        }
       if (n_flag && staOK) {
               staOK = checknation(h, country, ncountry);
               if (n_flag < 0)
                   staOK = !staOK;
        }

       if ((ld_flag || lp_flag) && staOK) {
          staOK = checklevs(&h, &data, ld_flag, lp_flag, minlev, maxlev);
       }

       if (staOK) {
          if (h_flag) 
             error = write_hydro_hdr(STDOUT, h);
          else 
             error = write_hydro_station(STDOUT, h, data);
             
          if (error) {
             fprintf(stderr,"\nError in write_hydro_station(): error code = %d.\n", error);
             exit(1);
          }
          ++nout;
       }
       else {
          ++nskip;
          if (o_flag) {
              if (h_flag) 
                error = write_hydro_hdr(outcastfile, h);
              else 
                error = write_hydro_station(outcastfile, h, data); 
          }
       }

     }  /* end while */

     report_status(status, stderr);
     close(infile);

NEXTFILE:
     ;
   } while (curfile++ < nfiles);

   fprintf(stderr,"\n\n%d stations extracted   %d stations skipped", 
           nout, nskip);
   fprintf(stderr,"\n\nEnd of extract.\n\n");
   exit(0);

} /* end main() */




/****************************************************************************/

void print_usage(program)
char *program;
{
   fprintf(stderr,"\nUsage:  %s filename_root(s)", program);

   fprintf(stderr," [-Ddirname] [-Eextent] [-H] [-Ooutcast_file] ");
   fprintf(stderr,"-T[c[^]/cruise1/cruise2/...]/[d[^]/depthmin/depthmax]");
   fprintf(stderr,"/[l<d_or_p>[^]/minlev/maxlev]");
   fprintf(stderr,"/[m[^]/m1/m2/...]/[n[^]/code1/code2/...]/");
   fprintf(stderr,"[g[^]/minlon/maxlon/minlat/maxlat]/");
   fprintf(stderr,"[s[^]/ship1/ship2/...]/[y[^]/minyear/maxyear]\n");
   fprintf(stderr,"\n    -D  : specifies dirname (default is ./) ");
   fprintf(stderr,"\n        ex: -D../data/ ");
   fprintf(stderr,"\n    -E  : specifies file extent (default is no extent)");  
   fprintf(stderr,"\n        ex: -E.dat ");
   fprintf(stderr,"\n    -H  : output header only");  
   fprintf(stderr,"\n    -O  : specifies file to which rejected stations are written");  
   fprintf(stderr,"\n        ex: -OVE1982.sta");
   fprintf(stderr,"\n    -T  : specifies type of extraction and criteria");
   fprintf(stderr,"\n        c = NODC cruise #  ex:  -Tc/8103/7202/83");
   fprintf(stderr,"\n        d = if deepest level sampled falls between min/max ex:  -Td/4000/10000");
   fprintf(stderr,"\n        g = geographic position  ex:  -Tg/-80/-10/0/60");
   fprintf(stderr,"\n        ld = depth level  ex:  -Tld1000/2000");
   fprintf(stderr,"\n        lp = pressure level  ex:   -Tlp^1000/2000");
   fprintf(stderr,"\n        m = by month  ex:  -Tm/1/2/3/12");
   fprintf(stderr,"\n        n = nation (NODC country code)ex:  -Tn31/90");
   fprintf(stderr,"\n        s = ship  ex:  -TsVE/6N");
   fprintf(stderr,"\n        y = by year   ex:  -Ty/1965/1980");
   fprintf(stderr,"\n        combination of above:  -Tg/-80/-10/0/160/m/1/2/3/12");
   fprintf(stderr,"\n\n  Use ^ after the extraction type to extract stations");
   fprintf(stderr,"\n  which DO NOT meet the criteria. ");
   fprintf(stderr,"\n   ex: -Ts^/VE/y1982  to extract all stations from 1982");
   fprintf(stderr,"\n       except those collected by the ship VE.");
   fprintf(stderr,"\n\n");  
   return;
}

/****************************************************************************/
/****************************************************************************/

int checkmonth(hdr, mask)
struct HYDRO_HDR  hdr;
int   mask;

/*  Returns a nonzero value if the station was done during one of the 
    specified months  or 0 if it does not */
{
   int  m = 1;

   return ((m << (hdr.month -1)) & mask);    /* bitwise and */

} /* end checkmonth() */

/****************************************************************************/




int checkpos(hdr, top, bot, left, right)
struct HYDRO_HDR  hdr;
float top, bot, left, right;

{

   if ((hdr.lat > top) || (hdr.lat < bot))
        return (0);

   if ((hdr.lon > right) || (hdr.lon < left))
        return (0);

   return (1);

} /* end checkpos() */

/****************************************************************************/
int checkyear(hdr, min, max)
struct HYDRO_HDR  hdr;
int   min, max;

{

   return ((hdr.year <= max) && (hdr.year >= min));

} /* end checkyear() */


/****************************************************************************/
int checknation(hdr, list, n)
struct HYDRO_HDR  hdr;
char list[MAXCRIT][3];    /* list of country codes */
int  n;                   /* number of country codes in list */
{
     register i;

   i = -1;
   while (++i < n) {
        if (strncmp(hdr.country,list[i],2) == 0)
            return(1);
   }
   return(0);

} /* end checknation() */
/****************************************************************************/
int checkship(hdr, list, n)
struct HYDRO_HDR  hdr;
char list[MAXCRIT][3];    /* list of ship codes */
int  n;                   /* number of ship codes in list */
{
     register i;

   i = -1;
   while (++i < n){
        if (strncmp(hdr.ship,list[i],2) == 0)
            return(1);
   }
   return(0);

} /* end checkship() */
/****************************************************************************/
int checkdepth(hdr, z, min, max)
struct HYDRO_HDR  hdr;
double *z;               /* pointer to depth values */
int   min, max;          /* mindepth, maxdepth */

   /* Returns a 1 if the bottommost observation is at a level between
      min and max (inclusive); or a 0 if not. */

{

   return ((z[hdr.nobs-1] <= max) && (z[hdr.nobs-1] >= min));

} /* end checkdepth() */

/****************************************************************************/
int checkcruise(hdr, list, n)
struct HYDRO_HDR  hdr;
int *list;        /* list of NODC cruise #s */
int  n;                   /* number of items in list */
{
     register i;

   i = -1;
   while (++i < n) {
      if (list[i] == hdr.cruise)
            return(1);
   }
   return(0);

} /* end checkcruise() */

/****************************************************************************/
int checklevs(hptr, dptr, d_flag, p_flag, min, max)
struct HYDRO_HDR *hptr;
struct HYDRO_DATA *dptr;
int d_flag, p_flag;      /* -1, 0, or 1 for depth/pressure levels */
float min, max;

   /*  Determines which pressure or depth levels meet the min/max criteria
       and orders the data accordingly, adjusting nobs if necessary.
       Returns a 1 if any levels meet the criteria or 0 if no levels do. */

{
   int i, j, k, ii, not_flag, levOK;
   double *prop;

   not_flag = 0;

   if (d_flag) {
      if (d_flag < 0)  
         not_flag = 1;
      prop = dptr->observ[(int)DE];
   }
   else {
      prop = dptr->observ[(int)PR];
      if (p_flag < 0)
         not_flag = 1;
   }

   k = 0;
   for (j = 0; j < hptr->nobs; ++j) {
      levOK = (prop[j] <= max) && (prop[j] >= min);
      if (not_flag)
           levOK = !levOK;
      if (levOK) {
        for (i = 0; i < hptr->nprops; ++i) {
            ii = hptr->prop_id[i];
            dptr->observ[ii][k] = dptr->observ[ii][j];
        }
        ++k;
      }
   }
   hptr->nobs = dptr->nobs = k;

   if (k == 0) 
       return(0);

   return(1);

} /* end checklevs() */
