/*  grid3d.c
                *****************************
                      H Y D R O B A S E
                *****************************
.
.  USAGE:
. grid3d filename_root(s)  -B/west/east/south/north -I<gridspacing>     -P<property_list>  -Ccdf_outfile_name [-N] [-S<sigma_series_filename>] [-D<dirname>] [-E<file_extent>] [-Z<std_depth_file>] [-T<time_bin_file>]
.
..........................................................................
. Computes an average profile for each property specified at each lat/lon
. gridnode from randomly spaced stations and outputs a netCDF file of hydro 
. properties gridded as a function of lat, lon, and depth.  With the -T option
. multiple 3-D matrices will be created, one representing all the data, 
. while each of the others represents a user-specified range of years.  Without
. this time option, one matrix including all data will be generated.  The depth
. dimension consists of a series of standard depths, but these values also
. can be optionally specified by the user with -Z.  The averaging is done 
. on density surfaces and interpolated back onto the depth levels.  A
. default set of sigma surfaces is provided, but the user can (and should!)
. supply the sigma values with the -S option.  Optimum results will be
. obtained by customizing the sigma series for a particular part of the
. ocean.  The goal is to specify sigma values which span the entire water
. column with appropriate sampling as a function of depth.  
..........................................................................
*/

#include <stdio.h>
#include <string.h>
#include <math.h>
#include "hydro_cdf.h"
#include "hydrobase.h"

#define    NINT(x)	((int)((x) + 0.5))
#define    UI   unsigned int
#define    MISSINGVAL  -999999.  /* fill value for cdf file */
#define    PREFILL      0        /* turn off prefilling of cdf variables */
#define    ADD_PROP_COUNTS  1    /* turn on option to include # of obs info */
#define    STDERR 2
#define    PRINT_MSG  1          /* turn on message printing to stderr */
#define    MAX_TIME_BINS  20     /* arbitrary number of time bins supported */

/* input_file pathnames */

#define    EXTENT   ""
#define    DIR      ""

/***************************************************************/
/*  define the standard sigma levels on which data will be averaged;
     sigma series is divided by reference levels ...*/

#define  MAXSIGLEVS 500    /* max size of sigma series -- arbitrary number */
#define  NREFLEVS  5       /* # of ref levs defining sigma values */

double siglevs[MAXSIGLEVS];   /* stores the sigma series  */
int  nsiglevs;                /* number of elements in full sigma series */

int ref_prop_id[NREFLEVS] = {(int)S0,(int)S1,(int)S2,(int)S3,(int)S4};
int nsig[NREFLEVS];       /* number of elements of sigseries for each ref lev */
int zmin[NREFLEVS] = {-1,500,1500,2500,3500}; /* depth ranges for ref levs */
int zmax[NREFLEVS] = {500,1500,2500,3500,99999};
double *sigptr[NREFLEVS];   /* points to station arrays of referenced sigma values */

#define  NSIG0   86        /* # of values in sig0_default */

double sig0_default[NSIG0] = 
{18.0, 19.0, 20.0, 21.0, 22.0, 
 23.0, 23.2, 23.4, 23.6, 23.8,
 24.0, 24.2, 24.4, 24.6, 24.8,
 25.0, 25.2, 25.4, 25.6, 25.8,
 26.0, 26.1, 26.2, 26.3, 26.4, 26.5, 26.6, 26.7, 26.8, 26.9,
 27.0, 27.02, 27.04, 27.06, 27.08,
 27.1, 27.12, 27.14, 27.16, 27.18,
 27.2, 27.22, 27.24, 27.26, 27.28,
 27.3, 27.32, 27.34, 27.36, 27.38,
 27.4, 27.42, 27.44, 27.46, 27.48,
 27.5, 27.52, 27.54, 27.56, 27.58,
 27.6, 27.62, 27.64, 27.66, 27.68,
 27.7, 27.72, 27.74, 27.76, 27.78,
 27.8, 27.82, 27.84, 27.86, 27.88,
 27.9, 27.92, 27.94, 27.96, 27.98,
 28.0, 28.02, 28.04, 28.06, 28.08,
 28.1
};

#define  NSIG1  39           /* # of values in sig1_default */

double sig1_default[NSIG1] = 
{30.0, 30.1, 30.2, 30.3, 30.4, 30.5, 30.6, 30.7, 30.8, 30.9,
 31.0, 31.1, 31.2, 31.3, 31.4, 31.5, 31.6, 31.7, 31.8, 31.9, 
 32.0, 32.1, 32.2, 32.3, 32.4, 32.5, 32.6, 32.7, 
 32.8, 32.81, 32.82, 32.83, 32.84, 32.85, 32.86, 32.87, 32.88, 32.89,
 32.9};

#define   NSIG2  44          /* # of values in sig2_default */

double sig2_default[NSIG2] =
{ 36.0, 36.1, 36.2, 36.3, 36.4, 
  36.5, 36.52, 36.54, 36.56, 36.58,
  36.6, 36.62, 36.64, 36.66, 36.68,
  36.7, 36.72, 36.74, 36.76, 36.78,
  36.8, 36.82, 36.84, 36.86, 36.88,
  36.9, 36.92, 36.94, 36.96, 36.98,
  37.0, 37.02, 37.04, 37.06, 37.08,
  37.1, 37.12, 37.14, 37.16, 37.18,
  37.2, 37.3, 37.4, 37.5 };

#define  NSIG3   44         /* # of values in sig3_default */

double sig3_default[NSIG3] = 
{41.0, 41.1, 
 41.2, 41.22, 41.24, 41.26, 41.28,
 41.3, 41.32, 41.34, 41.36, 41.38,
 41.4, 41.42, 41.44, 41.46, 41.48,
 41.5, 41.52, 41.54, 41.56, 41.58,
 41.6, 41.62, 41.64, 41.66, 41.68,
 41.7, 41.72, 41.74, 41.76, 41.78,
 41.8, 41.82, 41.84, 41.86, 41.88,
 41.9, 41.92, 41.94, 41.96, 41.98,
 42.0, 42.02
};

#define   NSIG4  36          /* # of values in sig4_default */

double sig4_default[NSIG4] =
{45.60, 45.62, 45.64, 45.66, 45.68, 
 45.70, 45.72, 45.74, 45.76, 45.78, 
 45.80, 45.82, 45.84, 45.86, 45.88, 
 45.90, 45.91, 45.92, 45.93, 45.94, 45.95, 45.96, 45.97, 45.98, 45.99,
 46.00, 46.02, 46.04, 46.06, 46.08,
 46.10, 46.12, 46.14, 46.16, 46.18,
 46.20};


/***************************************************************/

/* globally referenced variables for input station data */

struct HYDRO_DATA sta;
struct HYDRO_HDR hdr;
double *pr, *de, *te, *sa;
double s_pref;

int report_gaps;           /* switch turns on reporting of vertical datagaps */

/************ structures to represent grid node ****************/

struct surfrec {
          double  density, depth;
          double *prop;
              UI *count;
              UI  n;
  struct surfrec *next;
};

struct gridnode {
         double **prop;
         double  *d;
             UI **count;
             UI  *nobs;
 struct surfrec  *mix_layer;
};
/***************************************************************/

extern int   get_prop_indx();     /* functions in prop_subs.c */
extern char *get_prop_units();
extern char *get_prop_mne();
extern void  print_prop_menu();
extern void  compute_sigma(), compute_height(), compute_theta();
extern double potvort(), buoyancy();

extern int std_depth_init();      /* functions in hydro_utils.c */
extern int open_hydro_file();
extern int get_station();


extern int  write_prop_cdf();     /* functions in hydro_cdf.c */
extern int  write_prop_count_cdf();
extern int  write_std_depths_cdf();
extern int  write_bottom_depth_cdf();
extern int  cdf_init();
extern int  cdf_define();
extern int  get_indices();


/***************************************************************/

main (argc, argv)
int   argc;
char *argv[];
{
   short   bopt, iopt, popt, copt;
   short   deriv_prop_flag;
   int     curfile = 1, nfiles = 0; 
   int     print_mess, xdateline; 
   int     nlevs, nprops = 0, prop_indx[MAXPROP];
   int     i, j, error, test_outcrop, prop_avail;
   int     nstdlevs;
   FILE   *sigfile[NREFLEVS];
   FILE   *z_file;
   FILE   *t_file;
   int     row, col, nrows, ncols, tbin, ntbins;
   int     include_counts;
   int     out_of_bounds=0;
   char   *extent, *dir, *st;
   int     infile, cdf_file;
   float  **data, *bottomdepth;  
   short  **count; 
   struct CDF_HDR  h;
   struct gridnode **grid[MAX_TIME_BINS];
   struct gridnode **alloc_grid();
   int     get_sig_series();
   int     get_time_bins();
   int     parse_p_option();
   void    parse_s_option();
   void    insert_data();
   void    insert_derivative();
   void    insert_surf_vals();
   void    compute_avg();
   void    print_usage();
   void    define_sigma_transitions();
   void    free_and_alloc();
   int     do_std_depth();


/* are there command line arguments? */

   if (argc < 2) {
      print_usage(argv[0]);
      exit(1);
   }
 
/*  set these default values */

    dir = DIR;
    extent = EXTENT;
    bopt = iopt = popt = copt = 0;
    z_file = t_file = NULL;
    for (j = 0; j < NREFLEVS; ++j)
       sigfile[j] = NULL;
    include_counts = ADD_PROP_COUNTS;
    error = 0;
    print_mess = PRINT_MSG;
    report_gaps = 0;
    xdateline = 0;
    

/*  parse the command line arguments */

   for (i = 1; i < argc; i++) {
      if (argv[i][0] == '-') {
            switch (argv[i][1]) {
               case 'D':                   /* get input directory */
                        dir = &argv[i][2];
                        break;
               case 'E':                    /* get file extent */
                        extent = &argv[i][2];
                        break;

               case 'G':                    /* report vertical gaps */
                        report_gaps = 1;
                        break;
               case 'B':                    /* get grid bounds */
                        bopt = 1;
                        st = &argv[i][2];
                           if (*st == '/')
                               ++st;
                        error = (sscanf(st,"%f", &h.xmin) != 1);
                        while (*(st++) != '/')
                           ;  
                        error += (sscanf(st,"%f", &h.xmax) != 1);
                        while (*(st++) != '/')
                           ;  
                        error += (sscanf(st,"%f", &h.ymin) != 1);
                        while (*(st++) != '/')
                           ;  
                        error += (sscanf(st,"%f", &h.ymax) != 1);
                        
                        if (h.xmin > 0 && h.xmax < 0)
                           h.xmax += 360.;
                        if (h.xmax > 180)
                           xdateline = 1;
                           
                        break;

               case 'I':
                        iopt = 1;
                        error = (sscanf(&argv[i][2],"%f", &h.xincr) == 1) ? 0 : 1;
                        h.yincr = h.xincr;
                        while (*(st++) != '\0') {
                            if (*st == '/') {
                              ++st;
                              error = (sscanf(st,"%f", &h.yincr) == 1) ? 0 : 1;
                              break;
                            }
                        }
                        
                        break;
                        

               case 'S':
                        parse_s_option(&argv[i][2], sigfile);
                        break;

               case 'Z':
                        z_file = fopen(&argv[i][2],"r");
                        if (z_file == NULL) {
                           fprintf(stderr,"\nError opening stddep file: %s\n",&argv[i][2]);
                           exit(1);
                        }
                        break;

               case 'T':
                        t_file = fopen(&argv[i][2],"r");
                        if (t_file == NULL) {
                           fprintf(stderr,"\nError opening time bin file: %s\n",&argv[i][2]);
                           exit(1);
                        }
                        break;

               case 'P':
                        popt = 1;
                        nprops = parse_p_option(&argv[i][2], prop_indx);
                        break;

               case 'C':
                        cdf_file = cdf_init(&argv[i][2]);
                        copt = 1;
                        break;

               case 'N':
                        include_counts = 0;
                        break;

               default:
                        error = 1;

          }    /* end switch */

          if (error ) {
             fprintf(stderr,"\nError parsing command line args.\n");
             fprintf(stderr,"     in particular: '%s'\n", argv[i]);
             print_usage(argv[0]);
             exit(1);
          }

       }  /* end if */

       else  {
           ++nfiles;
       }
   }  /* end for */

   if (!bopt || !iopt || !nfiles || !popt || !copt) {
       fprintf(stderr,"\nYou must specify input file(s), bounds, properties, cdf_output_file and gridspacing!\n");
       print_usage(argv[0]);
       exit(1);
   }


/* initialize global array of std_depths */

     nstdlevs = std_depth_init(z_file);


/* define values for sigma series ... */

   nsiglevs = 0;
   for (i = 0; i < NREFLEVS; ++i ) {
      nsig[i] = get_sig_series(ref_prop_id[i], sigfile[i], &siglevs[nsiglevs]);
      nsiglevs += nsig[i];
      if (nsiglevs >= MAXSIGLEVS) {
          fprintf(stderr,"\n # of sigma levels exceeds space allocated");
          fprintf(stderr,"\n Recompile program with larger MAXSIGLEVS.\n");
          exit(1);
      }
   }
   fprintf(stderr,"\n total number of sigma levels: %d\n", nsiglevs);


/* compute dimensions of grid and allocate space for computation ...*/

   nrows = (int) ((h.ymax - h.ymin) / h.yincr);
   ncols = (int) ((h.xmax - h.xmin) / h.xincr);
   h.node_offset = 1;       /* 0 for node grid, 1 for pixel grid */

   ntbins = get_time_bins(t_file, &h);

   if (ntbins > MAX_TIME_BINS) {
          fprintf(stderr,"\n # of time bins exceeds space allocated");
          fprintf(stderr,"\n Recompile program with larger MAX_TIME_BINS.\n");
          exit(1);
   }

   fprintf(stderr,"\n allocating memory for grid #", ntbins);

   for (tbin = 0; tbin < ntbins; ++tbin) {
     fprintf(stderr,"   %d", tbin);
     grid[tbin] = alloc_grid(ncols, nrows, nsiglevs, nprops);
   }
   fprintf(stderr,"\n");


/* loop for each input_file */

   do {
     if ((infile = open_hydro_file(dir, argv[curfile], extent, print_mess)) 
          < 0) {
       goto NEXTFILE;
     }

     
     /* loop for each station */

    while ((error = get_station(infile, &hdr, &sta)) >= 0) {

       /* is station within bounds ?   */
       
       if (xdateline && hdr.lon < 0)
          hdr.lon += 360;

       if (get_indices(h, hdr.lat, hdr.lon, &row, &col) < 0) {
           ++out_of_bounds;
       }
       else {

         pr = sta.observ[(int)PR];   /* set frequently used pointers */
         de = sta.observ[(int)DE];
         te = sta.observ[(int)TE];
         sa = sta.observ[(int)SA];
         if (pr == NULL || te == NULL || sa == NULL || de == NULL) {
           fprintf(stderr, "\nThe requisite parameters: pr, te, sa, de are not available at this station.\n");
           write_hydro_hdr(STDERR, hdr);
           continue;
         }

         /* compute referenced sigmas for this station... */

         free_and_alloc(&sta.observ[(int)S0], hdr.nobs);
         compute_sigma(0., hdr.nobs, sta.observ[(int)S0], pr, te, sa);
         free_and_alloc(&sta.observ[(int)S1], hdr.nobs);
         compute_sigma(1000., hdr.nobs, sta.observ[(int)S1], pr, te, sa);
         free_and_alloc(&sta.observ[(int)S2], hdr.nobs);
         compute_sigma(2000., hdr.nobs, sta.observ[(int)S2], pr, te, sa);
         free_and_alloc(&sta.observ[(int)S3], hdr.nobs);
         compute_sigma(3000., hdr.nobs, sta.observ[(int)S3], pr, te, sa);
         free_and_alloc(&sta.observ[(int)S4], hdr.nobs);
         compute_sigma(4000., hdr.nobs, sta.observ[(int)S4], pr, te, sa);

         for (i = 0; i < NREFLEVS; ++i) {
            sigptr[i] = sta.observ[ref_prop_id[i]];
         }



         /* compute other appropriate properties for this station ... */

         for (i = 0; i < nprops; ++i) {
            test_outcrop = 0;
            deriv_prop_flag = 0;
            prop_avail = 1;

         /* !**! Special cases for individual properties... */
            switch ((enum property) prop_indx[i]) {

               case PR:
                  test_outcrop = (de[0] <= 51.) ? 1 : 0;
                  break;

               case OX: /* fall through */
               case N2:
               case N3:
               case P4:
               case SI:
                    prop_avail = available((enum property) prop_indx[i], hdr);
                    break;

               case S_:
                  free_and_alloc(&sta.observ[(int)S_], hdr.nobs);
                  compute_sigma(s_pref, hdr.nobs, sta.observ[(int)S_],pr,te,sa);
                  break;

               case TH:
                  free_and_alloc(&sta.observ[(int)TH], hdr.nobs);
                  compute_theta(hdr.nobs, sta.observ[(int)TH], pr, te, sa);
                  break;

               case HT:
                  free_and_alloc(&sta.observ[(int)HT], hdr.nobs);
                  compute_height(hdr.nobs, pr, te, sa, sta.observ[(int)HT]);
                  break;

               case SV:
                  free_and_alloc(&sta.observ[(int)SV], hdr.nobs);
                  compute_sp_vol(hdr.nobs, sta.observ[(int)SV], pr, te, sa);
                  break;

               case VA:
                  free_and_alloc(&sta.observ[(int)VA], hdr.nobs);
                  compute_svan(hdr.nobs, sta.observ[(int)VA], pr, te, sa);
                  break;


               case BF:   /* fall through */
               case PV:
                  deriv_prop_flag = 1;
                  break;

               default:
                   break;
            } /* end switch */


            for (tbin = 0; tbin < ntbins; ++tbin) {
              if (hdr.year >= h.tmin[tbin] && hdr.year <= h.tmax[tbin]) {

                if (deriv_prop_flag) {
                 insert_derivative(prop_indx[i], grid[tbin][row][col].prop[i], grid[tbin][row][col].count[i], (double)hdr.lat);
                }
                else if (prop_avail) {
                  insert_data(sta.observ[prop_indx[i]], hdr.nobs,  grid[tbin][row][col].prop[i], grid[tbin][row][col].count[i], test_outcrop);
                }

              }
            } /* end for tbin */

         } /* end for */

      /* insert depth values and deal with the sea surface ... */

         test_outcrop = (de[0] <= 51.) ? 1 : 0;
         for (tbin = 0; tbin < ntbins; ++tbin) {
             if (hdr.year >= h.tmin[tbin] && hdr.year <= h.tmax[tbin]) {
                insert_data( de, hdr.nobs, grid[tbin][row][col].d, 
                            grid[tbin][row][col].nobs, test_outcrop);
                insert_surf_vals(&grid[tbin][row][col], prop_indx, nprops);
             }
         }
       } /* end else */

     }  /*end while !eof */ 

NEXTFILE:
     close(infile);

   } while (curfile++ < nfiles );             

   if (error > 0) {
         report_status(error, stderr);
         exit(1);
   }

/******** end of input phase *********/

   if (out_of_bounds) {
      fprintf(stderr,"\n  %d out-of-bounds stations were read and ignored.\n", out_of_bounds);
   }

   fprintf(stderr,"\n  constructing header ...\n");


/* construct the cdf header  */

   h.nx = ncols;
   h.ny = nrows;
   h.nz = nstdlevs;
   h.nt = ntbins;
   h.nprops = nprops;
   h.fill_value = MISSINGVAL;
   strncpy(h.x_units, "degrees", 8);
   strncpy(h.y_units, "degrees", 8);
   strncpy(h.z_units, "meters", 7);
   strncpy(h.title,"HydroBase", 10);
   strcpy(h.command, *argv);
   for (i = 1; i < argc; ++i) {
      strncat(h.command, " ", 1);
      strcat(h.command, argv[i]);
   }
   h.prop_id = (char **) malloc(nprops * sizeof(char *));
   h.prop_units = (char **) malloc(nprops * sizeof(char *));
   for (i = 0; i < nprops; ++i) {
      h.prop_id[i] = (char *) malloc(3);
      h.prop_units[i] = (char *) malloc(50);
      strncpy(h.prop_id[i], get_prop_mne(prop_indx[i]),3);
      strcpy(h.prop_units[i], get_prop_units(prop_indx[i]));
   }
   if ( h.prop_units[nprops-1] == NULL) {
      fprintf(stderr, "\nUnable to malloc memory for the cdf header.\n");
      exit(1);
   }

   error = cdf_define(cdf_file, h, PREFILL, include_counts);
   error = write_std_depths_cdf(cdf_file, h);
   error = write_time_bins_cdf(cdf_file, &h);


   data = (float **) malloc ((UI) nprops  * sizeof(float *));
   count = (short **) malloc ((UI) (nprops ) * sizeof(short *));
   bottomdepth = (float *) malloc ((UI) ncols * sizeof(float));

   for (i = 0; i < nprops; ++i) {
      data[i] = (float *) malloc((UI) (nstdlevs * ncols) * sizeof(float));
      count[i] = (short *) malloc((UI) (nstdlevs * ncols) * sizeof(short));
   }

   if (data[nprops-1] == NULL) {
      fprintf(stderr, "\nUnable to allocate memory for data & count arrays.\n");
      exit(1);
   }

   fprintf(stderr,"  computing averages ...\n");

/* for each gridnode, compute means at all sigma levels ... */

   for (tbin = 0; tbin < ntbins; ++tbin) {
      for (row = 0; row < nrows; ++row) {
         for (col = 0; col < ncols; ++col) {
            for (i = 0; i < nprops; ++i) {
               compute_avg(grid[tbin][row][col].prop[i], 
                           grid[tbin][row][col].count[i], nsiglevs);
            }
            compute_avg(grid[tbin][row][col].d, grid[tbin][row][col].nobs, 
                        nsiglevs);
            define_sigma_transitions(&grid[tbin][row][col], nprops);
         }
      }
   }


/* interpolate the sigma series back onto the standard depth levels and 
   output the property data to the netcdf file... */

   fprintf(stderr,"  writing data ...\n");

   for (tbin = 0; tbin < ntbins; ++tbin) {
      for (row = 0; row < nrows; ++row) {
         for (col = 0; col < ncols; ++col) {
            nlevs = do_std_depth(grid[tbin][row][col], nsiglevs, prop_indx, 
                  nprops, ncols, col, data, count, &bottomdepth[col]);
         }

         for (i = 0; i < nprops; ++i) {
            write_prop_cdf(cdf_file, data[i], h.prop_id[i], row, 0, tbin, 0,
                           1, ncols, 1, nlevs);
            if (include_counts) {
               write_prop_count_cdf(cdf_file, count[i], h.prop_id[i],
                           row, 0, tbin, 0, 1, ncols, 1, nlevs);
            }
         }
         write_bottom_depth_cdf(cdf_file, row, 0, tbin, 1, ncols, 1,
                                bottomdepth);
      }
   }
   cdf_close(cdf_file);
   free((char *) data);
   free((char *) count);
   free((char *) bottomdepth);
   exit(0);
} /* end main */


/****************************************************************************/

void print_usage(program)
char *program;
{
   fprintf(stderr,"\nUsage:  %s filename_root(s)  -B/west/east/south/north -C<cdf_output_file> -I<gridspacing> -P<list_of_properties> [-S<ref_id>/<sigma_series_file>] [-Z<std_depth_file>] [-T<time_bin_file>] [-N] [-G] [-D<dirname>] [-E<file_extent>] \n", program);
   fprintf(stderr,"\n    -B  : specifies grid bounds");
   fprintf(stderr,"\n    -C  : name of netCDF output file.");

   fprintf(stderr,"\n    -I  : specifies grid increment in degrees;  ex: -I0.5");
   fprintf(stderr,"\n          OR specify separate x,y increments with a slash");
   fprintf(stderr,"\n          to delimit xincr/yincr;  ex: -I2.0/0.5\n");
   fprintf(stderr,"\n    -P  : list of properties to project onto surface;");
   fprintf(stderr,"\n          ex:  -Ppr/th/sa/ox/ht");
   fprintf(stderr,"\n               -P (by itself) produces a list of available properties\n");
   fprintf(stderr,"\n    -S  : ref_id = 0,1, 2,3, or 4 ");
   fprintf(stderr,"\n          Specify each ref level with its own -S argument");
   fprintf(stderr,"\n          filename = file containing sigma series definitions.");
   fprintf(stderr,"\n          Each line in file contains sigmin, sigmax, incr");
   fprintf(stderr,"\n          (sigmin and sigmax are INCLUSIVE in generating series).");
   fprintf(stderr,"\n          Values MUST be monotonically INCREASING.");
   fprintf(stderr,"\n          -S (by itself) lists the default sigma series.\n");
   fprintf(stderr,"\n    -Z  : file containing list of standard depths.");
   fprintf(stderr,"\n          Values MUST be monotonically INCREASING.\n");
   fprintf(stderr,"\n    -T  : file describing time bins.");
   fprintf(stderr,"\n          Each line in file contains minyear maxyear");
   fprintf(stderr,"\n    -N  : suppress inclusion of observation counts.\n");
   fprintf(stderr,"\n    -G  : report vertical datagaps in output profile.\n");
   fprintf(stderr,"\n    -D  : specifies directory for input data files (default is current directory) ");
   fprintf(stderr,"\n            ex: -D../data/\n ");
   fprintf(stderr,"\n    -E  : specifies input_file extent (default is no extent)");  
   fprintf(stderr,"\n            ex: -E.dat \n");
   fprintf(stderr,"\n\n");  
   return;
} /* end print_usage() */
/****************************************************************************/

void parse_s_option(str, file_ptr)
char *str;
FILE **file_ptr;
{
  int j;

  switch (*str) {
    case '\0' :
        fprintf(stdout,"\nDefault Sigma Levels:");
        fprintf(stdout,"\nsigma-0:");
           for (j = 0; j < NSIG0; ++j) {
               if ((j % 10) == 0) {
                   fprintf(stdout,"\n");
               }
               fprintf(stdout, "%.2lf ", sig0_default[j]);
           }
           fprintf(stdout,"\n\nsigma-1:");
           for (j = 0; j < NSIG1; ++j) {
              if ((j % 10) == 0) {
                fprintf(stdout,"\n");
              }
              fprintf(stdout, "%.2lf ", sig1_default[j]);
           }

           fprintf(stdout,"\n\nsigma-2:");
           for (j = 0; j < NSIG2; ++j) {
              if ((j % 10) == 0) {
                fprintf(stdout,"\n");
              }
              fprintf(stdout, "%.2lf ", sig2_default[j]);
           }
           fprintf(stdout,"\n\nsigma-3:");
           for (j = 0; j < NSIG3; ++j) {
              if ((j % 10) == 0) {
                fprintf(stdout,"\n");
              }
              fprintf(stdout, "%.2lf ", sig3_default[j]);
           }
           fprintf(stdout,"\n\nsigma-4:");
           for (j = 0; j < NSIG4; ++j) {
              if ((j % 10) == 0) {
                fprintf(stdout,"\n");
              }
              fprintf(stdout, "%.2lf ", sig4_default[j]);
           }

           fprintf(stdout,"\n");
           exit(0);

    case '0':
           j = 0;
           break;
    case '1':
           j = 1;
           break;
    case '2':
           j = 2;
           break;
        case '3':
           j = 3;
           break;
        case '4':
           j = 4;
           break;
        default:
           fprintf(stderr,"\n Error parsing -S option\n");
           fprintf(stderr,"USAGE: -S<ref_lev_id>/<file_name>");
           fprintf(stderr,"\n ex: -S4/sig4levs.list\n");
           exit(1);
           break;

    }  /* end switch */

    if (*(++str) == '/')
        ++str;

    file_ptr[j] = fopen(str,"r");
    if (file_ptr[j] == NULL) {
      fprintf(stderr,"\nError opening %s \n", str);
      exit(1);
    }

   return;
}   /* end parse_s_option() */

/*****************************************************************************/
int parse_p_option(st, prop_indx)
char *st;
int *prop_indx;
{
  char prop[4];
  int n;

  if (*st == '\0') {
          print_prop_menu();
         exit(0);
  }

  n = 0;
  do {
     if (*st == '/')
         ++st;
     sscanf(st,"%2s", prop);
     prop_indx[n] = get_prop_indx(prop);
     if (prop_indx[n] < 0)  {
       fprintf(stderr,"\n Unknown property '%.2s' specified in -P%s\n", prop, st);
       exit(1);
     }
       
        
     ++st;
     ++st;

     if (prop_indx[n] == (int)SF) {
       fprintf(stderr,"\n Stream Function cannot be computed here.");
       fprintf(stderr,"\n A separate program will do this but requires grid3d");
       fprintf(stderr,"\n output containing a minimum of these properties: ");
       fprintf(stderr,"                pr, sv, ht  \n");
       exit(1);
     } 

     if (prop_indx[n] == (int)S_ ) {
        if (sscanf(st, "%lf", &s_pref) != 1) {
          fprintf(stderr, "\n Specify a ref pressure for  %.2s", prop);
          fprintf(stderr,"\n   ex: -P%.2s1500/th/sa\n", prop);
          exit(1);
        }
        while (!(*st=='/' || *st==0 || *st==' '))
           ++st;
     }

     if (prop_indx[n] != (int) DE)  {
         /* depth is automatically done so don't count it here */
            ++n;
     }

   } while (*st == '/');
   return (n);
}  /* end parse_p_option() */

/*****************************************************************************/
struct gridnode **alloc_grid(nx, ny, nz, nprops)
int nx, ny, nz, nprops;
{
   int i, j, k, n;
   struct gridnode **g;

   g = (struct gridnode **) malloc(ny * sizeof(struct gridnode *));
   if (g == NULL) {
      fprintf(stderr,"\nUnable to allocate memory for grid.\n");
      exit(1);
   }
   for (i = 0; i < ny; ++i) {
         g[i] = (struct gridnode *) malloc(nx * sizeof(struct gridnode));
         if (g[i] == NULL) {
           fprintf(stderr,"\nUnable to allocate memory for grid[%d]\n", i);
           exit(1);
         }
         for (j = 0; j < nx; ++j) {

            g[i][j].prop = (double **) malloc(nprops * sizeof(double *));
            if (g[i][j].prop == NULL) {
                fprintf(stderr,"\nUnable to allocate memory for grid[%d][%d].prop\n", i, j);
                exit(1);
            }

            g[i][j].count = (UI **) malloc(nprops * sizeof(UI *));
            if (g[i][j].count == NULL) {
                fprintf(stderr,"\nUnable to allocate memory for grid[%d][%d].count\n", i, j);
                exit(1);
            }

            g[i][j].d = (double *) malloc(nz * sizeof(double));
            if (g[i][j].d == NULL) {
               fprintf(stderr,"\nUnable to allocate memory for g[%d][%d].d\n", i, j);
               exit(1);
            }
            g[i][j].nobs = (UI *) malloc(nz * sizeof(UI));
            if (g[i][j].nobs == NULL) {
               fprintf(stderr,"\nUnable to allocate memory for g[%d][%d].nobs\n", i, j);
               exit(1);
            }

            g[i][j].mix_layer = (struct surfrec *) NULL;
          

            for (n = 0; n < nz; ++n) {       /* initialize depth,count array */
                 g[i][j].d[n] = 0.0;
                 g[i][j].nobs[n] = 0;
            }

            for (k = 0; k < nprops; ++k) {

              g[i][j].prop[k] = (double *) malloc(nz * sizeof(double));
              if (g[i][j].prop[k] == NULL) {
                fprintf(stderr,"\nUnable to allocate memory for grid[%d][%d].prop[%d]\n", i, j, k);
                exit(1);
              }

              g[i][j].count[k] = (UI *) malloc(nz * sizeof(UI));
              if (g[i][j].count[k] == NULL) {
                fprintf(stderr,"\nUnable to allocate memory for grid[%d][%d].count[%d]\n", i, j, k);
                exit(1);
              }

                 /* zero each prop, count */
              for (n = 0; n < nz; ++n) {     
                 g[i][j].prop[k][n] = 0.0;
                 g[i][j].count[k][n] = 0;
              }
            } /* end for */
         } /* end for */
   }  /* end for */

   return g;
} /* end alloc_grid */

/*****************************************************************************/
int get_sig_series(ref_id, fptr, siglist)
int  ref_id;       /* defines the sigma variable: (int) enum property */
FILE *fptr;        /* pointer to file containing list of sigma values OR nil */
double *siglist;   /* starting addr of array to insert values */
    /* if fptr is NULL, the default sigma series corresponding to ref_id
       will be inserted starting at the address pointed to by siglist.
       Otherwise, the file will be read for (min, max, incr) triplets
       from which a sigma series will be generated and inserted at siglist.
       The number of sigma values is returned. */
{
   double  min, max, incr;
   double  next;
   int i, count;


   if (fptr == NULL) {
      switch ((enum property) ref_id) {
        case S0:
           fprintf(stderr,"Using default levels for sigma-0 \n");
           for (i = 0; i < NSIG0; ++i) {
              *siglist++ = sig0_default[i];
           }
           return (NSIG0);
        case S1:
           fprintf(stderr,"Using default levels for sigma-1 \n");
           for (i = 0; i < NSIG1; ++i) {
              *siglist++ = sig1_default[i];
           }
           return (NSIG1);

        case S2:
           fprintf(stderr,"Using default levels for sigma-2 \n");
           for (i = 0; i < NSIG2; ++i) {
              *siglist++ = sig2_default[i];
           }
           return (NSIG2);

        case S3:
           fprintf(stderr,"Using default levels for sigma-3 \n");
           for (i = 0; i < NSIG3; ++i) {
              *siglist++ = sig3_default[i];
           }
           return (NSIG3);

        case S4:
           fprintf(stderr,"Using default levels for sigma-4 \n");
           for (i = 0; i < NSIG4; ++i) {
              *siglist++ = sig4_default[i];
           }
           return (NSIG4);

        default:
           return (0);

      }  /* end switch */
   }
   else {

     count = 0;
     while( fscanf(fptr,"%lf %lf %lf", &min, &max, &incr) == 3) {
         *siglist = min;
         ++count;
         while ( ( next = *siglist + incr) <= max) {
           ++count;
           *(++siglist) = next; 
         }
         ++siglist;
     }

     fclose(fptr);
     return (count);
   }

} /* end get_sig_series */
/*****************************************************************************/
int get_time_bins( fptr, hptr)
FILE *fptr;        /* pointer to file containing list of time ranges OR nil */
struct CDF_HDR *hptr;  
    /* if fptr is NULL, a single time bin with min = 0, max = 9999 is created.
       Otherwise, the file will be read for (minyear, maxyear) couplets
       from which additional time bins will be constructed.
       The number of time bins is returned. */
{
   int min, max;
   int count;

   count = 1;
   if (fptr != NULL) {
      while( fscanf(fptr,"%d %d", &min, &max) == 2) {
          ++count;
      }
      fseek(fptr, 0L, 0);      /* rewind file */
   }

   hptr->tmin = (int *) malloc(count * sizeof(int));
   hptr->tmax = (int *) malloc(count * sizeof(int));

   hptr->tmin[0] = 0;         /* the first bin includes all years */
   hptr->tmax[0] = 9999;

   if (count == 1)
       return (1);

   count = 1;                 /* set subsequent time bins */
   while( fscanf(fptr,"%d %d", &min, &max) == 2) {
         hptr->tmin[count] = min;
         hptr->tmax[count++] = max;
   }

   fclose(fptr);
   return (count);

} /* end get_time_bins */

/*****************************************************************************/
void insert_data( y, ny, ysig, count, test_outcrop)
double *y;      /* array of y observed values */
int     ny;    /* dimension of y */
double *ysig;   /* sum of yvalues on each sigma surface */
int    *count;  /* counts # of observations on each surface */
int     test_outcrop;  /* 0 or 1: if set, tests for outcropping surfaces*/

/* For the y property at a station with ny observation levels, interpolate
   to find the yvals at each level in the sigma series,  add each
   yval to its appropriate sum, and increment the counter.  Checks for vertical
   data gaps using the globally defined de array and does not interpolate over
   gaps which exceed 200 m in the thermocline (upper 1000 m) or 1000 m 
   elsewhere.  The sigma series is specified in the global array, siglevs.
   The observed sigma values at this station are subdivided by ref levels, 
   and stored at global addresses pointed to by sigptr. 
*/
{
   int  i, j, datagap, n;
   int  refindx;
   double *xtmp[NREFLEVS], *ytmp, *dtmp, z;
   double interpolate();
   
   for (i = 0; i < NREFLEVS; ++i )
      xtmp[i] = (double *) malloc((UI) (ny * sizeof(double)));
   ytmp = (double *) malloc((UI) (ny * sizeof(double)));
   dtmp = (double *) malloc((UI) (ny * sizeof(double)));
   if (dtmp == NULL) {
       fprintf(stderr,"\nUnable to allocate memory in insert_data()\n");
       exit(1);
   }

/* Ensure continuous (no missing values) x,  y, and depth arrays */
   n = 0;
   for (i = 0; i < ny; ++i) {
      if ( y[i] > -8.9)  {
         for (j = 0; j < NREFLEVS; ++j )
            xtmp[j][n] = sigptr[j][i];
         ytmp[n] = y[i];
         dtmp[n++] = de[i];
      }
   }

   for (i = 0; i < nsiglevs; ++i) {

    /* determine which ref lev corresponds to this sigseries element ... */

      refindx = 0;      
      j = nsig[0];
      while (i > j) {
         if (++refindx == NREFLEVS-1)
             break;
         j += nsig[refindx];
      }

      if ((test_outcrop) && (siglevs[i] < xtmp[refindx][0])) {   
        /* note:  this assumes x (density) increases with depth!! */           
             ysig[i] += -1.;      /* add -1 for outcropping surfaces */
             ++count[i];
      }

        /* see if associated depth exists at this station ... */

      else if ((z = interpolate(siglevs[i], xtmp[refindx], dtmp, n)) > -998.) {

            /* now check for datagaps around this depth...*/
            j = 0;
            while (dtmp[++j] < z) 
                ;
            if ((dtmp[j-1] == z) || (dtmp[j] == z) )
                datagap = 0;
            else if (z < 1001)
                datagap = (dtmp[j] - dtmp[j-1]) > 200;
            else
                datagap = (dtmp[j] - dtmp[j-1]) > 601;
  
            if (!datagap) {
               if ((z = interpolate(siglevs[i], xtmp[refindx], ytmp, n)) > -998.) {
                  ysig[i] += z;
                  ++count[i];
               }
            }
      }
   }

   for (i = 0; i < NREFLEVS; ++i)
      free((char *)xtmp[i]);
   free((char *)ytmp);
   free((char *)dtmp);
   return;
} /* end insert_data() */

/*****************************************************************************/
void insert_derivative(index, ysig, count, lat)
int index;      /* identifies property number in enum property type */
double *ysig;   /* array in which sum of yvals on each surface is stored */
int *count;     /* counts # of observations on each surface */
double lat;     /* latitude (degrees) of this station */

/* Computes the property specified by index at the pressure level corresponding
   to each sigma surface for the current station.  The value of the derived 
   property is added to the appropriate level in the ysig array and the
   counter is incremented. The sigma surfaces are globally specified in
   siglevs; the station data is globally stored in pr, de, te, and sa arrays. */
{
   int i, j, datagap, refindx;
   float pr_int;
   double y, z, p0;
   double interpolate();

/* check that this is an appropriate property ... */

   switch ((enum property) index) {
      case BF:      /* fall through */
      case PV:
               break;
      default:
          fprintf(stderr,"\nUnknown property passed to insert_derivative()!!\n");
          return;
   }  /* end switch */

   for (i = 0; i < nsiglevs; ++i) {

    /* determine which ref lev corresponds to this sigseries element ... */

      refindx = 0;      
      j = nsig[0];
      while (i > j) {
         if (++refindx == NREFLEVS-1)
             break;
         j += nsig[refindx];
      }

      if ((z = interpolate(siglevs[i], sigptr[refindx],de,hdr.nobs)) > -998.) {

            /* now check for datagaps around this depth...*/
            j = 0;
            while (de[++j] < z) 
                ;
            if ((de[j-1] == z) || (de[j] == z) )
                datagap = 0;
            else if (z < 1001)
                datagap = (de[j] - de[j-1]) > 200;
            else
                datagap = (de[j] - de[j-1]) > 601;
  
            if (!datagap) {
               p0 = interpolate(siglevs[i], sigptr[refindx], pr, hdr.nobs);
               switch ((enum property)index) {
                  case BF:
                     pr_int = 10;   /* pressure interval in db.*/
                     y = buoyancy(p0, pr, te, sa, hdr.nobs, pr_int);
                     if (y > -9998.)
                        y = y * 1.0e5;
                     break;
                  case PV:
                     pr_int = 10;   /* pressure interval in db.*/
                     y = buoyancy(p0, pr, te, sa, hdr.nobs, pr_int);
                     if (y > -9998.) 
                        y = potvort((y*y), lat);
                     break;
                  default:
                     y = -9999.0;
                     break;
               }  /* end switch */

               if (y > -9998.) {
                   ysig[i] += y;
                   ++count[i];
               }
             } /* end if !datagap */
       } /* end if */

   }  /* end for */
   return;

}  /* end insert_derivative() */

/*****************************************************************************/
void insert_surf_vals(g, prop_indx, nprops)
struct gridnode *g;    /* ptr to gridnode */
int *prop_indx;       /* contains index to properties being gridded */
int nprops;           /* number of properties being gridded */
   /* defines the depth of a mixed layer at the sea surface, determines 
      the average values of properties within it, and inserts the information
      into a linked list of records sorted by density. The station data are 
      accessed through the global variable: struct HYDRO_DATA sta */
{
   int i, j, n, weight;
   double x, v, vprev, dprev;
   double dens, depth;
   struct surfrec *m, *get_surfrec();

   if (de[0] > 10.0)
       return;

   /* round off surface density to nearest tenth */

   dens =  (double) (NINT(sta.observ[(int)S0][0] * 10.)) / 10.0;

  /* bottom of mixed layer is defined where density = surface density + 0.1 */   
   depth = interpolate(dens+.1, sta.observ[(int)S0], de, hdr.nobs);

   if (depth < -999.)  /* all observations are within .1 of surface density */
       depth = de[hdr.nobs-1];

   m = get_surfrec(g->mix_layer, dens, nprops);

   if (g->mix_layer == NULL ) {   /* empty list */           
      g->mix_layer = m;
   }

   /* add depth info to this record */

   m->depth += depth;
   ++m->n;

  /* compute average properties for mixed layer and add to appropriate fields.
     The average value is computed by summing the observed values weighted by
     the depth between observations.*/

   for (i = 0; i < nprops; ++i) {
      if (sta.observ[prop_indx[i]] != NULL ) {
         j = 0;
         x = sta.observ[prop_indx[i]][0];
         n = 1;               /* weight the observation at sea surface */
         if (x < -8.9 ) {     /* unless it is a missing value */
            x = 0.0;
            n = 0;
         }
         dprev = 0.0;
         vprev = sta.observ[prop_indx[i]][0];
         while ( (j < hdr.nobs) && (de[j] <= depth)) {
            if ( (v = sta.observ[prop_indx[i]][j]) > -8.9) {
               if (vprev < -8.9) 
                   vprev = v;
               weight = (int) (de[j] - dprev);
               x += (v + vprev) * .5 * weight;
               n += weight;
               dprev = de[j];
               vprev = v;
            }
            ++j;
         } 
         if (n > 0) {
            m->prop[i] += (x / (float) n);
            ++m->count[i] ;
         }
      }
   }
   return;

}  /* end insert_surf_vals() */

/*****************************************************************************/
struct surfrec *get_surfrec(rptr, d, nprops)
struct surfrec *rptr;    /* pointer to start of list */
double d;                /* density to key on */
int  nprops;             /* # of properties to allocate space for */
   /* Recursively searches a linked list of surfrecs to:
        a) find an existing record for the specified density;
     or b) create a new record and insert it into the ordered list.

      Returns a pointer to the appropriate record.
   */
{
    struct surfrec *r1ptr, *create_surfrec();
    double *tempd;
    UI *tempi;

    if (rptr == NULL) {         /* empty list */
       r1ptr = create_surfrec(nprops);
       r1ptr->density = d;
       return(r1ptr);
    }

    if (NINT(d * 10) == NINT(rptr->density * 10)) {  /* current rec */
        return (rptr);
    }

    if (d < (rptr->density - .00001)) {   /* insert before the current rec */

       r1ptr = create_surfrec(nprops);
       tempd = r1ptr->prop;
       tempi = r1ptr->count;

         /* copy all fields from rptr into r1ptr */
       r1ptr->density = rptr->density;
       r1ptr->depth = rptr->depth;
       r1ptr->prop = rptr->prop;
       r1ptr->count = rptr->count;
       r1ptr->n = rptr->n;
       r1ptr->next = rptr->next;

        /* initialize the fields of rptr and link it to r1ptr */
       rptr->density = d;
       rptr->depth = 0;
       rptr->prop = tempd;
       rptr->count = tempi;
       rptr->n = 0;
       rptr->next = r1ptr;
       
       return(rptr);
    }

    r1ptr = get_surfrec(rptr->next, d, nprops);  /* search rest of list */
    if (rptr->next == NULL)
          rptr->next = r1ptr;
    return (r1ptr);

}   /* end get_surfrec() */

/*****************************************************************************/
struct surfrec *create_surfrec(nprops)
int nprops;
   /* Allocates memory and initializes the fields of a struct surfrec.
      Returns a pointer to the record */
{
   struct surfrec *r;
   int i;

   r = (struct surfrec *) malloc(sizeof(struct surfrec));
   if (r == NULL) {
      fprintf(stderr,"\nUnable to allocate memory in create_surfrec()\n");
      exit(1);
   }
   r->depth = 0;
   r->density = 0;
   r->n = 0;
   r->next = NULL;
   r->count = (UI *) malloc(nprops * sizeof(UI));
   r->prop = (double *) malloc(nprops * sizeof(double));

   if (r->prop == NULL) {
      fprintf(stderr,"\nUnable to allocate memory in create_surfrec()\n");
      exit(1);
   }

   for (i = 0; i < nprops; ++i) {
      r->prop[i] = 0.0;
      r->count[i] = 0;
   }   

   return (r);
}   /* end create_surfrec() */
/*****************************************************************************/
void compute_avg(sum, nobs, nlevs)
double *sum;   /* array containing summed values */
UI   *nobs;    /* array containing count of values in each sum */
int   nlevs;   /* # of elements in array */
{
   int j;

   for (j = 0; j < nlevs; ++j) {
      sum[j] = (nobs[j] > 0) ?  sum[j] / nobs[j] : MISSINGVAL;
   }
   return;

}  /* end compute_avg() */
/*****************************************************************************/
void define_sigma_transitions(g, nprops)
struct gridnode *g;   /* grid node */
int nprops;           /* number of properties stored at each gridnode */
/*  determines where the averaged depth values cross the depth ranges (zmin
    and zmax) for each sigma reference level and zeros the counters for the 
    accumulated sums of any level outside that range.  */
{
   int i, j, refstart, refstop, iref;

   refstop = 0;
   for (iref = 0; iref < NREFLEVS; ++iref) {
      refstart = refstop;
      refstop = refstart + nsig[iref];

      for (i = refstart; i < refstop; ++i) {
         if (g->nobs[i] > 0) {
            if (g->d[i] < zmin[iref] || g->d[i] > zmax[iref]) {
                g->nobs[i] = 0;
                for (j = 0; j < nprops; ++j) {
                   g->count[j][i] = 0;
                }
            }
         }
      }

   }
   return;

}  /* end define_sigma_transitions() */
/*****************************************************************************/
struct surfrec * define_avg_mixed_layer(listptr, nprops)
struct surfrec *listptr;  /* address of first record in linked list */
int nprops;               /* number of properties being averaged */
   /* Traverses a linked list of surfrecs sorted by density and computes 
      the average density at the sea surface, then searches the linked list
      for that density or 2 density records bracketing the average density and
      returns a pointer to a surfrec containing the property values
      associated with the average density. */ 
{
   struct surfrec *r1, *get_density_rec();
   double x;
   UI n;
   int key;
   
   if (listptr == NULL)                         /* empty list */
       return ((struct surfrec *) NULL);

   r1 = listptr;
   x = 0.0;
   n = 0;
   while (r1 != NULL) {
      x += r1->density ; 
      ++n;   
      r1 = r1->next;    
   }
   key = NINT((x / (double)n) * 10.);     /* average density * 10  */
   
   /* Now search list for the density value ... */

   r1 = get_density_rec(key, listptr, nprops);

   return (r1);

}  /* end define_avg_mixed_layer() */

/***********************************************************************/
struct surfrec *get_density_rec(key, rptr, nprops)
int key;               /* density  being searched for *10   */
struct surfrec *rptr;  /* ptr to element of linked list */
int nprops;            /* # of properties stored in each surfrec */
    /* Recursively searches a linked list sorted by density for the record
       corresponding to key or for two records bracketing that density.
       Returns a record containing averaged property values for that density. */
{
   struct surfrec *r1, *r2, *new, *create_surfrec();
   double x[2], y[2];
   UI n;
   int  key1, key2, i;
   void compute_avg();

   if (rptr == NULL) {           /* YIKES! */
       fprintf(stderr,"\nError in get_density_rec(): ");
       fprintf(stderr,"\nEnd of linked list encountered before finding avg density!");
       fprintf(stderr,"\nError in program logic....exiting.\n");
       exit(1);
   }

   r1 = rptr;
   r2 = rptr->next;

   if (r2 == NULL) {             /* end of list, so r1 must be the density */
      compute_avg(r1->prop, r1->count, nprops);
      r1->depth /= (double)r1->n;
      return (r1);
   }

   key2 = NINT(r2->density * 10);

   if (key > key2)                /* recursive part */
       return( get_density_rec(key, r2, nprops));
   

   if (key == key2) {             /* exact match! */
       compute_avg(r2->prop, r2->count, nprops);
       r2->depth /= (double)r2->n;
       return(r2);
   }
     
   key1 = NINT(r1->density * 10);

   if (key < key1) {              /* YIKES! */
       fprintf(stderr,"\nError in get_density_rec(): ");
       fprintf(stderr,"\nAvg density is less than first density in list!");
       fprintf(stderr,"\nThis is a bug in the program logic....exiting.\n");
       exit(1);
   }
   if (key == key1)  {            /* exact match! */
       compute_avg(r1->prop, r1->count, nprops);
       r1->depth /= (double)r1->n;
       return(r1);
   }

/* if we get this far, then the key must lie between r1 & r2... so 
      construct a new record with values interpolated between r1 and r2 */

   compute_avg(r1->prop, r1->count, nprops);  
   compute_avg(r2->prop, r2->count, nprops);
   x[0] = r1->density;
   x[1] = r2->density;
   
   new = create_surfrec(nprops);
   new->density = (double) key / 10.0;
   new->n = (r1->n + r2->n) >> 1;    /* divide by 2 */
   y[0] = r1->depth / (double) r1->n;
   y[1] = r2->depth / (double) r2->n;
   new->depth = interpolate(new->density, x, y, 2);
   
   for (i = 0; i < nprops; ++i) {
         if (! (r1->count[i] && r2->count[i])) { /* both records must have  */
             new->count[i] = 0;                  /* at least one obs */
             new->prop[i] = MISSINGVAL;
             continue;                         
         }

         new->count[i] = (r1->count[i] + r2->count[i]) >> 1;
         y[0] = r1->prop[i];
         y[1] = r2->prop[i];
         new->prop[i] = interpolate(new->density, x, y, 2);
   }
   return(new);

}  /* end get_density_rec() */

/***********************************************************************/
int do_std_depth(g, npts, prop_id, nprops, ncols, icol, dataptr, countptr, bottom)
struct gridnode *g;        /* address of this gridnode */
int npts;                  /* size of arrays for this gridnode */
int *prop_id;              /* array of property identifiers */
int nprops;                /* # of properties in prop array */
int ncols;                 /* # of cols in data and count arrays */
int icol;                  /* column # associated with this gridpt */
float **dataptr;           /*  ptr to start of data arrays  */
short **countptr;          /*  ptr to start of count arrays */
float *bottom;             /* returned depth of deepest observation */

/*  takes the info for a gridnode and computes the value of each property 
    interpolated onto the std depth levels.  The arrays at dataptr and countptr
    are assumed to have dimensions data[nprops][nstdlevs*ncols].  This
    function deals with one gridpoint at a time, therefore icol must be
    specified to determine the starting position within the array for
    each property; i.e. starting address = data[iprop][nstdlevs * icol].  
    (The arrays have been setup to include an entire row of
    data to improve the efficiency of writing out to the netcdf file).
 >> THE CALLING PROGRAM IS RESPONSIBLE FOR ALLOCATING THE SPACE AT
     *dataptr AND *countptr: 
   
    The # of observations at each standard level and for each property 
    are approximated and returned starting at the address pointed to by 
    countptr.

    The function returns the # of std levels (including the bottom) . 
    std_depth[MAXSTDLEVS] is globally defined and already initialized.
*/
{
   int i, ii, j, jj, k, i_te, i_sa, datagap, room, nlevs;
   double *x, *y, *w, *sigval, *sig, z, r;
   float  *d;
   short  *n;
   int size, start;
   int deepestlev;
   struct surfrec *m, *m2, *define_avg_mixed_layer();
   int  sort_by_depth();
   double interpolate(), svan_();
   extern double std_depth[];

   sigval = (double *) malloc(npts * sizeof(double));
   sig = (double *) malloc(npts * sizeof(double));
   size = sort_by_depth(g->d, g->nobs, sigval, g->prop, g->count, nprops, npts);

   if (size <= 1) {
      for (i = 0; i < nprops; ++i) {
           d = &dataptr[i][icol * NSTDLEVS];
           n = &countptr[i][icol * NSTDLEVS];
           for (j = 0; j < NSTDLEVS; ++j) {
               *(d++) = MISSINGVAL;
               *(n++) = 0;
           }
      }
      *bottom = MISSINGVAL;
      free(sigval);
      return(NSTDLEVS);
   }

/* construct temporary arrays, x, y & w to store depth, y-property and 
   w-# of obs for that property.  Enlarge the size of the arrays 
   to accommodate the mixed layer observations.  */

   room = 10;
   x = (double *) malloc((size+room) * sizeof(double));
   y = (double *) malloc((size+room) * sizeof(double));
   w = (double *) malloc((size+room) * sizeof(double));
   if (w == NULL) {
     fprintf(stderr,"\nUnable to allocate memory in do_std_depth()\n");
     exit(1);
   }

/* find the deepest level for which there is data */

   deepestlev = 0;
   for (k = 0; k < size; ++k) {
      if (g->nobs[k] > 0)
           deepestlev = k;
   }
   *bottom = (float) g->d[deepestlev];

   m = define_avg_mixed_layer(g->mix_layer, nprops);

/* identify index for temp and salt */

   i = i_te = i_sa = -1;
   while (++i < nprops) {
      if (prop_id[i] == (int)TE)
          i_te = i;
      if (prop_id[i] == (int) SA)
          i_sa = i;
   }

/* Define the starting point in the sigma series which is heavier than
    the density at the bottom of the average mixed layer . Ensure that 
    the depth of this sigma level is deeper than the depth of the mixed 
    layer. */

   start = 0;
   if (m != NULL) {
      if (m->depth > g->d[deepestlev])
          m->depth = g->d[deepestlev];

      y[0] = m->density + .1;
       
       while ((sigval[start] < y[0]) && (start < size))
           ++start;

      while ((m->depth > g->d[start]) && (start < size))
         ++start;
   }



/* Eliminate missing values from each y-property. Interpolate the property
   onto std depth surfaces.  Check for vertical datagaps and flag a standard
   depth as missing if the gap is too large.  Also interpolate to approximate 
   the # of observations at each std level.  The interpolation routine returns 
   MISSINGVAL when appropriate, so that each level is assigned a value
   whether or not any data is present.  The value of count at a level with 
   no data is assigned zero. */



   for (j = 0; j < nprops; ++j) {
         npts = 0;
         if (m != NULL) {
            if (m->count[j] > 0) {   
              npts = mixed_layer_vals(j, prop_id[j], m, x, y, w, i_te, i_sa);
              for (i = 0; i < npts; ++i) 
                 sig[i] = m->density;
              sig[npts-1] = m->density + .1;
            }  
         }

         if ((npts - start ) > room)  {
            fprintf(stderr, "\nIncrease the value of 'room' to at least %d", npts-start);
            fprintf(stderr, "\n in do_std_depth()\n");
            exit(1);
         }

         for (k = start; k < size; ++k) {
            if (g->count[j][k] > 0) {
               y[npts] = g->prop[j][k];
               w[npts] = (double) g->count[j][k];
               x[npts] = g->d[k];
               sig[npts] = sigval[k];
               ++npts;
            }  
         } 
         nlevs = nsig[0] + nsig[1];  /* # of sig1 + sig0 levs in global array*/
         d = &dataptr[j][icol * NSTDLEVS];
         n = &countptr[j][icol * NSTDLEVS];
         for (i = 0; i < NSTDLEVS-1; ++i) {   /* interpolate all but bottom */

           z =  interpolate(std_depth[i], x, y, npts);
           k = 0;

           if (z > -998.) {                  /* check for vertical datagaps */
              jj = 0;
              while (x[++jj] < std_depth[i]) 
                   ;

              if (((int)x[jj-1] == (int)std_depth[i]) 
                  || ((int)x[jj] == (int)std_depth[i])) 
                  datagap = 0;

              else if (std_depth[i] < 1001) {
                  if (datagap = ((x[jj] - x[jj-1]) > 200)) {

                   /* try to distinguish between missing data and a pycnostad */

                     r = sig[jj] - sig[jj-1];     
                     if (r > 2) {

                         /* This is probably the sig0/sig1 transition. Assume
                            its OK for now and wait until we're completely in
                             sig1 to check the delta sigma */

                         if(sig[jj] < siglevs[nlevs-1]) {
                            datagap = 0;
                         }
                     }
                     else {
                         ii = 1;
                         while (ii < nlevs && siglevs[ii] < sig[jj-1])
                             ++ii;
                         if (ii < nlevs && 
                            (r <= 2 * (siglevs[ii] - siglevs[ii-1])))
                             datagap = 0;
                     }
                  }
              }
              else
                  datagap = (x[jj] - x[jj-1]) > 601;

              if (datagap) {
                  z = MISSINGVAL;
                  if (j == 0 && report_gaps)
                    fprintf(stderr," datagap:  %.1lf  %.1lf\n", x[jj-1], x[jj]);
              }
              else {
                   r = interpolate(std_depth[i], x, w, npts);
                   if (r < 0)
                       k = 0;
                   else {
                       k = (r - (int)r) > 0 ? (int) r + 1 : (int) r;
                   }
              }
           }

           *(d++) = (float) z;
           *(n++) = (short) k;
         }

/* add the deepest observation */

       *(d++) = (float) g->prop[j][deepestlev];
       *(n++) = (g->prop[j][deepestlev] > -8.) ? (int) g->nobs[deepestlev] : 0;
   }

/* clean up space no longer needed ... */

   free(sigval);
   free(sig);
   free ((char *)x);
   free ((char *)y);
   free ((char *)w);
   m = g->mix_layer;
   while (m != NULL) {
      m2 = m->next;
      free((char *) m->prop);
      free((char *) m->count);
      free((char *)m);
      m = m2;
   }
   g->mix_layer = NULL;

   return(NSTDLEVS);

} /* end do_std_depth() */
/****************************************************************************/

int mixed_layer_vals(iprop, prop_id, m, x, y, w, i_te, i_sa)
int iprop;           /* indicates the ith property of nprops */
int prop_id;         /* indicates the ith property of MAXPROPS */
struct surfrec *m;   /* info defining the mixed layer */
double *x;           /* depth array */
double *y;           /* property array */
double *w;           /* count array */
int i_te, i_sa;      /* index to temperature, salt (or -1) */

   /* Creates arrays of depth, property, and counts at
      100 m intervals for the requested property.  Returns
      the number of points in each array. */
      
{
   int i, npts;
   double  *t, *s, pref;

/* Define top and bottom only unless the mixed layer depth
   exceeds 200 m ... */

   npts = 2;       
   if (m->depth > 199.)
        npts = 1 + (int) m->depth / 100;

/*  Assign depth and count values ... */

   for (i = 0; i < npts-1; ++i) {
         w[i] = m->count[iprop];
         x[i] = i * 100.0;           
   }         
   w[npts-1] = m->count[iprop];
   x[npts-1] = m->depth;

/*  !**! Special cases for individual properties... */

   switch ((enum property) prop_id) {
       case PR :                   
                     for (i = 0; i < npts; ++i) {
                        y[i] = x[i];   
                     }         
                     break;
       case HT:
                     if (i_te < 0 || i_sa < 0) {
                        for (i = 0; i < npts; ++i)
                          y[i] = 0.0;
                        break;
                     }
                     t = (double *) malloc(npts * sizeof(double));
                     s = (double *) malloc(npts * sizeof(double));
                     for (i = 0; i < npts; ++i) {
                        t[i] = m->prop[i_te];
                        s[i] = m->prop[i_sa];
                     }
                     compute_height(npts, x, t, s, y);
                     free(t);
                     free(s);
                     break;
       case S0:
                     if (i_te < 0 || i_sa < 0) {
                        for (i = 0; i < npts; ++i)
                          y[i] = m->prop[iprop];
                        break;
                     }

                        t = (double *) malloc(npts * sizeof(double));
                        s = (double *) malloc(npts * sizeof(double));
                        for (i = 0; i < npts; ++i) {
                           t[i] = m->prop[i_te];
                           s[i] = m->prop[i_sa];
                        }
                        pref = 0.0;
                        compute_sigma(pref, npts, y, x, t, s);
                        free(t);
                        free(s);
                        break;
       case S1:
                     if (i_te < 0 || i_sa < 0) {
                        for (i = 0; i < npts; ++i)
                          y[i] = m->prop[iprop];
                        break;
                     }
                        t = (double *) malloc(npts * sizeof(double));
                        s = (double *) malloc(npts * sizeof(double));
                        for (i = 0; i < npts; ++i) {
                           t[i] = m->prop[i_te];
                           s[i] = m->prop[i_sa];
                        }
                        pref = 1000.0;
                        compute_sigma(pref, npts, y, x, t, s);
                        free(t);
                        free(s);
                        break;
       case S2:
                     if (i_te < 0 || i_sa < 0) {
                        for (i = 0; i < npts; ++i)
                          y[i] = m->prop[iprop];
                        break;
                     }
                        t = (double *) malloc(npts * sizeof(double));
                        s = (double *) malloc(npts * sizeof(double));
                        for (i = 0; i < npts; ++i) {
                           t[i] = m->prop[i_te];
                           s[i] = m->prop[i_sa];
                        }
                        pref = 2000.0;
                        compute_sigma(pref, npts, y, x, t, s);
                        free(t);
                        free(s);
                        break;
       case S3:
                     if (i_te < 0 || i_sa < 0) {
                        for (i = 0; i < npts; ++i)
                          y[i] = m->prop[iprop];
                        break;
                     }
                        t = (double *) malloc(npts * sizeof(double));
                        s = (double *) malloc(npts * sizeof(double));
                        for (i = 0; i < npts; ++i) {
                           t[i] = m->prop[i_te];
                           s[i] = m->prop[i_sa];
                        }
                        pref = 3000.0;
                        compute_sigma(pref, npts, y, x, t, s);
                        free(t);
                        free(s);
                        break;
       case S4:
                     if (i_te < 0 || i_sa < 0) {
                        for (i = 0; i < npts; ++i)
                          y[i] = m->prop[iprop];
                        break;
                     }
                        t = (double *) malloc(npts * sizeof(double));
                        s = (double *) malloc(npts * sizeof(double));
                        for (i = 0; i < npts; ++i) {
                           t[i] = m->prop[i_te];
                           s[i] = m->prop[i_sa];
                        }
                        pref = 4000.0;
                        compute_sigma(pref, npts, y, x, t, s);
                        free(t);
                        free(s);
                        break;
       case S_:
                     if (i_te < 0 || i_sa < 0) {
                        for (i = 0; i < npts; ++i)
                          y[i] = m->prop[iprop];
                        break;
                     }
                        t = (double *) malloc(npts * sizeof(double));
                        s = (double *) malloc(npts * sizeof(double));
                        for (i = 0; i < npts; ++i) {
                           t[i] = m->prop[i_te];
                           s[i] = m->prop[i_sa];
                        }
                        compute_sigma(s_pref, npts, y, x, t, s);
                        free(t);
                        free(s);
                        break;
       case VA:
                     if (i_te < 0 || i_sa < 0) {
                        for (i = 0; i < npts; ++i)
                          y[i] = m->prop[iprop];
                        break;
                     }
                        t = (double *) malloc(npts * sizeof(double));
                        s = (double *) malloc(npts * sizeof(double));
                        for (i = 0; i < npts; ++i) {
                           t[i] = m->prop[i_te];
                           s[i] = m->prop[i_sa];
                        }
                        compute_svan(npts, y, x, t, s);
                        free(t);
                        free(s);
                       break;
       case SV:
                     if (i_te < 0 || i_sa < 0) {
                        for (i = 0; i < npts; ++i)
                          y[i] = m->prop[iprop];
                        break;
                     }
                        t = (double *) malloc(npts * sizeof(double));
                        s = (double *) malloc(npts * sizeof(double));
                        for (i = 0; i < npts; ++i) {
                           t[i] = m->prop[i_te];
                           s[i] = m->prop[i_sa];
                        }
                        compute_sp_vol(npts, y, x, t, s);
                        free(t);
                        free(s);
                       break;
       default:
                        for (i = 0; i < npts; ++i)
                          y[i] = m->prop[iprop];
                        break;
   }  /* end switch */

   return (npts);

}  /* end mixed_layer_vals() */
/****************************************************************************/
int sort_by_depth(d, nobs, sigmas, xx, count, nprops, npts)
double *d;             /* starting addr of depth */
UI  *nobs;             /* starting addr of nobs array */
double *sigmas;        /* array to put sigma values */
double **xx;           /* starting addr of other properties */
UI  **count;          /* starting addr of nobs arrays for other properties */ 
int   nprops;          /* # of rows (other properties) in xx */
int   npts;            /* # of cols (points) in each property array */

/* sorts the depth array into increasing order placing the smallest depth
   level containing data into the d[0] element.  The elements of the
   other arrays (in xx) are swapped to match the depth array.  Returns the 
   # of depth levels containing data. */
     
{
   int i, j, k;
   int *index;
   int size;
   double *temp;
   UI *tmp_cnt;
   void quicksort();

   index = (int *) malloc((UI) (npts * sizeof(int)));
   if (index == NULL) {
     fprintf(stderr, "\nUnable to allocate memory in sort_by_depth()\n");
     exit(1);
   }

/* first, eliminate any depth levels with no observations */
   k = 0;
   for (i = 0; i < npts; ++i) {
      if (nobs[i] > 0) {
         d[k] = d[i];
         nobs[k] = nobs[i];
         sigmas[k] = siglevs[i];
         index[k] = k;             /* create an index array */
         for (j = 0; j < nprops; ++j) {
           xx[j][k] = xx[j][i];
           count[j][k] = count[j][i];
         }
         ++k;
      }
   }
   size = k;
   if (size < 2) {
       free(index);
       return(size);
   }
   quicksort(d, index, size);  

   /* use the index array to swap the elements of the other property arrays
      into the order corresponding to the depth array. */

   temp = (double *) malloc((UI) (k * sizeof(double)));
   if (temp == NULL) {
     fprintf(stderr, "\nUnable to allocate memory in sort_by_depth()\n");
     exit(1);
   }
   tmp_cnt = (UI *) malloc ((UI) (k * sizeof(UI)));
   if (tmp_cnt == NULL) {
     fprintf(stderr, "\nUnable to allocate memory in sort_by_depth()\n");
     exit(1);
   }

   for (i = 0; i < nprops; ++i) {
      for (j = 0; j < size; ++j) {
         temp[j] = xx[i][index[j]];         /* store the correctly ordered  */
         tmp_cnt[j] = count[i][index[j]];   /* elements in temporary arrays */
      }
      for (j = 0; j < size; ++j) {
         xx[i][j] = temp[j];          /* swap temp back into the array */
         count[i][j] = tmp_cnt[j];
      }
   }

   for (i = 0; i < size; ++i) {        /* do the same for the nobs array */
       tmp_cnt[i] = nobs[index[i]];     /* and the sigma array */
       temp [i] = sigmas[index[i]];
   }
   for (i = 0; i < size; ++i) {
        sigmas[i] = temp[i];   
        nobs[i] = tmp_cnt[i];         
   }

   free((char *) index);
   free((char *) temp);
   free((char *) tmp_cnt);

   return (size);
   

} /* end sort_by_depth() */ 

/****************************************************************************/
void quicksort(a, b, nx)
double *a;   /* has dimension [nx] */
int *b;      /* index array has dimension [nx] */
int nx;
{
   int k;
   int find_pivot(), partition();
   double pivot;

   if (find_pivot(a, nx, &pivot) != 0) {
     k = partition(a, b, nx, pivot);
     quicksort(a, b, k);
     quicksort(a+k, b+k, nx-k);
   }
   return;
}  /* end quicksort() */

/****************************************************************************/
int find_pivot(a, n, pivot_ptr)
double *a, *pivot_ptr;
int n;                   /* size of a[] */
/* finds the pivot for quicksort() if one exists */
{
   int i;
   double tmp[3];
   void swap_d();

   if (n > 3) {  /* a carefully chosen inequality to improve efficiency */
      tmp[0] = a[0];
      tmp[1] = a[n/2];
      tmp[2] = a[n-1];
      if (tmp[0] > tmp[1])
         swap_d(&tmp[0], &tmp[1]);
      if (tmp[1] > tmp[2])
         swap_d(&tmp[1], &tmp[2]);
      if (tmp[0] > tmp[1])
         swap_d(&tmp[0], &tmp[1]);

      if (tmp[0] != tmp[2]) {
         *pivot_ptr = (tmp[0] < tmp[1]) ? tmp[1] : tmp[2];
         return(1);
      }
   }
   for (i = 1; i < n; ++i) {
      if (a[0] != a[i]) {
         *pivot_ptr = (a[0] > a[i]) ? a[0] : a[i];
         return(1);
      }
   }
   return (0);      /* all elements have the same value */

}  /* end find_pivot() */

/****************************************************************************/
int partition(a, b, n, pivot)
double *a;
int *b;
int     n;    /* n is the size of a[] and b[] */
double  pivot;
{
   int i = 0, j = n-1;
   void swap_d(), swap_i();

   while (i <= j) {
      while (a[i] < pivot)
         ++i;
      while (a[j] >= pivot)
         --j;
      if (i < j) {
         swap_d(&a[i], &a[j]);
         swap_i(&b[i++], &b[j--]);   /* do the same to b[] */
      }
   }
   return (i);     /* partition break size */

}  /* end of partition() */

/****************************************************************************/
void swap_d(xptr, yptr)
double *xptr, *yptr;
/*  exchanges the values of *xptr, *yptr */
{
   double tmp;

   tmp = *xptr;
   *xptr = *yptr;
   *yptr = tmp;
   return;

} /* end swap_d() */

/****************************************************************************/
void swap_i(xptr, yptr)
int *xptr, *yptr;
/*  exchanges the values of *xptr, *yptr */
{
   int tmp;

   tmp = *xptr;
   *xptr = *yptr;
   *yptr = tmp;
   return;

} /* end swap_i() */
/****************************************************************************/
/* Performs a linear interpolation to find the position of xval in array, x,
   and returns the corresponding value in the array, y.  If xval does not
   appear in array x, the value MISSINGVAL is returned.  This routine assumes 
   that the x array is monotonic and continuous (no missing values); and it
   assumes the y array is continuous.    */

double interpolate(xval, x, y, nypts)
double xval, *x, *y;
int    nypts;
{
   int    k;
   double v1, v2;

   for (k = 0; k < nypts-1; ++k) {

      v1 = xval - x[k];
      v2 = xval - x[k+1];

      if (v1 == 0)             /* x[k] == xval */
          return (y[k]);
      if (v2 == 0)             /* x[k+1] == xval */
          return (y[k+1]);
      if (v1 < 0. && v2 < 0.)  /* xval not between x1 and x2 */  
          continue;
      if (v1 > 0. && v2 > 0.) 
          continue;

      return ( y[k] + (y[k+1] - y[k]) * v1 / (x[k+1] - x[k]) );
   }

   return (MISSINGVAL);

}   /* end interpolate() */

/****************************************************************************/

void free_and_alloc(dptr, n)
double **dptr;   
int n;           /* number of elements to allocate */

   /* Frees up the space pointed to by dptr.  It MUST have been allocated
      using malloc()!!  Then mallocs space for n double values and sets
      dptr pointing to it.  If insufficient memory, a message is written to
      stderr and an exit() is made. */
{

   if (*dptr != NULL)
      free(*dptr);

   *dptr = (double *) malloc(n * sizeof(double));

   if (*dptr == NULL) {
      fprintf(stderr, "\nInsufficient memory for call to malloc()\n");
      exit(1);
   }

   return;
}  /* end free_and_alloc() */
