/*  gridsurf.c

____________________________________________________________________________
  USAGE:  

 gridsurf filename(_roots) [-C<time_bin_index>] -B/west/east/south/north -I<gridspacing>  -S<surface_def_file> -P<property_list> [-D<dirname>] [-E<file_extent>]

  list of filenames MUST be first argument!

 -C : input files are cdf format (default is HydroBase format),
      optionally specify time bin index (default is time_bin[0]);

 -B : specifies grid bounds

 -I : specifies grid increment in degrees;  ex: -I0.5

 -S : file containing surface definitions. If this is not specified
      these values are expected to come from the standard input device.

 -P : list of properties to evaluate; ex: -Ppr/te/th/sa/ox
      -P (by itself) will print a list of the available properties;

 -D : specifies directory for input files (default is current dir)  
      ex: -D/home/pete/data

 -E : specifies input_file extent (default is no extent) ex: -E.sum

____________________________________________________________________________
Gridsurf projects hydrographic properties onto one or more surfaces
and computes mean and standard deviation values of those properties
for each node in a grid. The grid spacing and bounds are specified by 
the user. All points within a square are used to compute the mean and 
are weighted equally. Squares do not overlap.  The centers of each 
square form the x-y loci of elements of the matrix representing each
surface.
.
The surfaces may be defined by some value of any property supported
by HydroBase (depth, pressure, temperature, density, etc.)
The properties at each station are linearly interpolated onto the
surface; and the averaging, which produces the grid, is done on that
surface.  That is, if the surface is a 100 m depth surface, the
interpolated properties will be averaged on that depth surface.  
IT IS STRONGLY RECOMMENDED that properties be averaged on density surfaces
to avoid problems of mixing water masses, especially in frontal zones.
If the surface being gridded with gridsurf is of a type other than
density, we recommend that the data be gridded (averaged) first with
the program grid3d, and use that product as input to gridsurf.

For each surface, an output file is generated containing :
     
     nrows, ncols, grid spacing, grid bounds         (1st line)
     list_of_properties                              (2nd line)
     lat lon   n1 p1 p1dev  n2 p2 p2dev ...  ( 1 line for each gridpt)
      where for each property:      n = # of obs
                                    p = mean 
                                 pdev = std dev    
                                                      
____________________________________________________________________________
*/


#include <stdio.h>
#include <math.h>
#include <fcntl.h>
#include <string.h>
#include <malloc.h>
#include "hydrobase.h"
#include "hydro_cdf.h"

#define   NINT(x)	((int)((x) + 0.5))
#define    ABS(x)       (((x) < 0) ? -(x) : (x))           
#define    BELL         0x07                

/* input file pathnames */

#define    EXTENT   ""
#define    DIR      ""


/* set up data structures to store gridded data */

struct surface {
          double  value;    /* value of property on this surface */
          double  pref;
            int   data_ind;   /* index ID of property */
            char  density_flag ;  /* set if this is a density surface */
          double *prop[MAXPROP]; /* ptrs to grid of prop values */
          double *propsq[MAXPROP];
        unsigned *count[MAXPROP];
            FILE *fptr;    /* output file */
  struct surface *next;    
};

    /* flags to indicate which properties are to be output and which are
       needed for computation */
      
short prop_req[MAXPROP];         /* set of props requested for output */
short prop_needed[MAXPROP];      /* set of props requested || defining a
                                    surface || needed for computation */

   /* global variables to store station ... */

struct HYDRO_DATA station;
struct HYDRO_HDR hdr;
double s_pref;           /* ref lev for computing a non-standard sigma level */

     /* boundaries for grid */

float   xmin, xmax, ymin, ymax, delta_x, delta_y;  
int 	xdateline;   
int     ncols, nrows;
int     tbin;


     /* in prop_subs.c */

extern int   get_prop_indx();               
extern char *get_prop_descrip(), *get_prop_mne();
extern void  print_prop_menu();
extern void  compute_sigma(), compute_height(); 
extern void  compute_theta();
extern double buoyancy(), potvort();

     /* in hydro_utils.c */

extern int open_hydro_file();           
extern int get_station();

   /* in hydro_cdf.c */

extern int cdf_open();
extern void cdf_close();
extern int read_cdf_hdr();
extern int read_cdf_prop();
extern int read_cdf_prop_count();
extern int read_cdf_depths();
extern int read_cdf_bottom_depth();
extern int get_indices();
extern int get_lat_lon();

main (argc, argv)
int   argc;
char *argv[];
{
   short   bopt, iopt, popt;
   short   cdf_flag;
   int     index, nprops = 0;
   int     i;
   int     curfile = 1, nfiles = 0; 
   int     print_msg = 1;       /* print message in file open routines*/
   int     error, prompt;
   unsigned gridsize;
   FILE    *def_file;
   char    *extent, *dir, *st;
   int     infile;
   struct surface *list = NULL, *surf;
   struct surface *add_surf();
   void    alloc_grids();
   void    print_usage();
   void    get_cdf_data();
   void    get_hydro_data();
   void    do_stats();
   int     parse_prop_list();


/* are there command line arguments? */

   if (argc < 2) {
      print_usage(argv[0]);
      exit(1);
   }
 
/*  set these default values */

    dir = DIR;
    extent = EXTENT;
    cdf_flag = 0;
    def_file = (FILE *) stdin;
    tbin = 0;
    prompt = 1;            /* for surface definitions */
    bopt = iopt = popt = 0;
    error = 0;
    s_pref = -1;
    xdateline = 0;

/* initialize these ... */

   for (i = 0; i < MAXPROP; ++i) {
       prop_req[i] = 0;
       prop_needed[i] = 0;
   }



/*  parse the command line arguments */

   for (i = 1; i < argc; i++) {
      if (argv[i][0] == '-') {
            switch (argv[i][1]) {
               case 'C':
                        cdf_flag = 1;
                        st = &argv[i][2];
                        if (*st != 0)
                           error = (sscanf(st,"%d", &tbin) != 1);
                        break;
               case 'D':                   /* get input dir */
                        dir = &argv[i][2];
                        break;
               case 'E':                    /* get file extent */
                        extent = &argv[i][2];
                        break;

               case 'B':                    /* get grid bounds */
                        bopt = 1;
                        st = &argv[i][2];
                           if (*st == '/')
                               ++st;
                        error = (sscanf(st,"%f", &xmin) != 1);
                        while (*(st++) != '/')
                           ;  
                        error += (sscanf(st,"%f", &xmax) != 1);
                        while (*(st++) != '/')
                           ;  
                        error += (sscanf(st,"%f", &ymin) != 1);
                        while (*(st++) != '/')
                           ;  
                        error += (sscanf(st,"%f", &ymax) != 1);
                        
                        if (xmin > 0 && xmax < 0)
                            xmax += 360;
                        if (xmax > 180)
                           xdateline = 1;
                        break;

               case 'I':
                        iopt = 1;
                        error = (sscanf(&argv[i][2],"%f", &delta_x) == 1) ? 0 : 1;
                        delta_y = delta_x;
                        st = strchr(&argv[i][2],'/');
                        if (st != NULL) {
                          sscanf(++st,"%f", &delta_y);
                        }
                        break;


               case 'S':
                        def_file = fopen(&argv[i][2],"r");
                        prompt = 0;              /* turn off prompt flag */
                        if (def_file == NULL) {
                           fprintf(stderr,"\nError opening %s.\n",&argv[i][2]);
                           exit(1);
                        }
                        break;

               case 'P':
                        popt = 1;
                        if ( argv[i][2] == '\0') {
                               print_prop_menu();
                               exit(0);
                        }
                        nprops = parse_prop_list(&argv[i][2]);
                        if (nprops <= 0)
                             error = 1;
                        break;

               default:
                        error = 1;

          }    /* end switch */

          if (error ) {
             fprintf(stderr,"\nError parsing command line args.\n");
             fprintf(stderr,"     in particular: '%s'\n", argv[i]);
             exit(1);
          }

       }  /* end if */

       else  {
           ++nfiles;
       }
   }  /* end for */

   if (!bopt || !iopt || !nfiles || !popt) {
       fprintf(stderr,"\nYou must specify input file(s), bounds, properties and gridspacing!\n");
       exit(1);
   }



   fprintf(stderr,"\n\n  bounds: %.3f %.3f %.3f %.3f\n", ymin, ymax, xmin, xmax );


   /* compute dimensions of matrix formed by grid  */

   nrows = (int) ((ymax - ymin) / delta_y);
   ncols = (int) ((xmax - xmin) /delta_x);
   gridsize = nrows * ncols;

   fprintf(stderr,"\n Grid will be:  %d rows by %d cols\n",  nrows, ncols);


/* get info on each surface,  add surface to the linked list, 
    allocate space for computation, write heading to each output file ... */

   while ( (surf = add_surf(def_file,prompt))  != NULL) {
      surf->next = list;
      list = surf;
      alloc_grids(list, gridsize);
      fprintf(list->fptr,"%d %d %.2f %.2f %.3f %.3f %.3f %.3f %2d\n", 
             nrows, ncols, delta_x, delta_y, xmin, ymin, xmax, ymax, nprops);
      for (i = 0; i < MAXPROP; ++i ) {
         if (prop_req[i]) {
           fprintf(list->fptr, "%2s ", get_prop_mne(i));
         }
      }
      fprintf(list->fptr,"\n");
   }

   fclose(def_file);


/* loop for each input file */

   do {

               /* netCDF file format */

     if (cdf_flag) {                         

       infile = cdf_open(dir, argv[curfile], extent, print_msg);
       if (infile < 0 )
          goto NEXTFILE;
     }
              /* HydroBase file format */
     else {                    

       infile = open_hydro_file(dir, argv[curfile], extent, print_msg);
       if (infile < 0)
          goto NEXTFILE;
     }

     
            /* read each file completely */

     if (cdf_flag) 
         get_cdf_data(infile, list) ;
     else 
         get_hydro_data(infile, list);

NEXTFILE:

     if (cdf_flag) 
         cdf_close(infile);
     else
         close(infile);

   } while (curfile++ < nfiles );


/*  compute means and std deviations and write statistics to outfile ... */


   fprintf(stderr,"\n    writing stats to outfile ...\n");

   do_stats(list);

   fprintf(stderr,"\n\nEnd of gridsurf.\n");
   exit(0);

}   /* end main */

/****************************************************************************/

void print_usage(program)
char *program;
{
   fprintf(stderr,"\nUsage:  %s filename_root(s)  [-C<time_bin_index>] -B/west/east/south/north -I<delta_x[/delta_y]> -P<list_of_properties> -S<surface_def_file> [-D<dirname>] [-E<file_extent>]", program);

   fprintf(stderr,"\n\n  List of filenames MUST be first argument!");
   fprintf(stderr,"\n\n    -C  : input files are cdf format files.");
   fprintf(stderr,"\n             (default is HydroBase format)");
   fprintf(stderr,"\n             specify which time_bin to use (def is 0)");
   fprintf(stderr,"\n    -B  : specifies grid bounds");
   fprintf(stderr,"\n    -I  : specifies grid increment in degrees.  ex: -I0.5");
   fprintf(stderr,"\n          If x and y increments differ, separate with");
   fprintf(stderr,"\n          a /       ex: -I0.5/1.0");
   fprintf(stderr,"\n    -P  : list of properties to project onto surface;");
   fprintf(stderr,"\n          ex:  -Ppr/th/sa/ox/ht");
   fprintf(stderr,"\n               -P (by itself) produces a list of available properties");
   fprintf(stderr,"\n    -S  : file containing surface definitions.");
   fprintf(stderr,"\n          If this is not specified these values are expected");
   fprintf(stderr,"\n          to come from the standard input device.");
   fprintf(stderr,"\n    -D  : specifies dirname for input files (default is current directory) ");
   fprintf(stderr,"\n            ex: -D../data/ ");
   fprintf(stderr,"\n    -E  : specifies input_file extent (default is no extent)");  
   fprintf(stderr,"\n            ex: -E.dat ");

   fprintf(stderr,"\n\n");  
   return;
}

/*****************************************************************************/
int parse_prop_list(st)
char *st;   /* the list with -P stripped off */
    /*  Parses a list of property mnemonics and sets global flags accordingly.
        Returns the number of properties.  An error will cause an exit. */
{
   int index, nprops;
   char prop[4];
   double ref_val;

   nprops = 0;

   do {
      if (*st == '/')
         ++st;
      sscanf(st,"%2s", prop);
      index = get_prop_indx(prop);
      if (index < 0) {
         fprintf(stderr,"\n Unknown property requested: %s\n", prop);
         exit(1);
      }
      prop_req[index] = 1;
      prop_needed[index] = 1;
      ++nprops;
      ++st;
      ++st;

  /* some special cases ... */

   /* s_ */
      if ((enum property)index == S_ ) {
         if (sscanf(st, "%lf", &ref_val) != 1) {
             fprintf(stderr,"\n Specify a ref pressure when requesting the property 's_'");
             fprintf(stderr,"\n   ex: -Ps_1500/th/sa\n");
             exit(1);
         }
         while (!(*st=='/' || *st=='\0' || *st==' '))
            ++st;

         if (s_pref >= 0) {   /* has this already been requested? */
                     if ( NINT( s_pref) != NINT(ref_val)) {
                         fprintf(stderr,"Sorry. You have requested the property s_ with multiple reference levels.\n");
                         fprintf(stderr,"You can only use one of those prefs");
                         fprintf(stderr,"  You will probably have to do multiple runs for each different pref you want to associate with s_\n\n");
                        exit(1);
                     }
          }
          s_pref = ref_val;
      }

   /* sf... stream function gets computed in an external
      program:  just output pr, sv, and ht for now.
   */

      if ((enum property)index == SF ) {
                         
          prop_req[index] = 0;     /* reset these */
          prop_needed[index] = 0;

          index = get_prop_indx("pr");
          prop_req[index] = 1;     
          prop_needed[index] = 1;
          index = get_prop_indx("sv");
          prop_req[index] = 1;     
          prop_needed[index] = 1;
          index = get_prop_indx("ht");
          prop_req[index] = 1;     
          prop_needed[index] = 1;
      }
   /* end of special cases */

  } while (*st == '/');

  return (nprops);
}  /* end parse_prop_list() */
/*****************************************************************************/
struct surface *add_surf(infile, prompt)
FILE *infile;
int  prompt;
/*  -  Allocates storage space for a record of type <struct surface>;
    -  queries the user for information about the surface (via stdin);
    -  sets the fields of the struct accordingly;
    -  opens the output file;
    -  returns a ptr to the struct  or NULL if no struct was added.
*/

{
   static n = 0;
   char id[4];
   int  index, get_prop_indx();
   struct surface *surf;
   char    fname[80];   


   if (!n && prompt) {
   fprintf(stderr,"\nDefine each projection surface ...\n");

   fprintf(stderr,"\n    -----  Surface property options  -------\n");
   print_prop_menu();
   fprintf(stderr,"\nbot:  (bottom observation)");
   fprintf(stderr,"\n\nend:  (end list)");
   fprintf(stderr,"\n    ----------------------------------------"); 

   }

   if (prompt)
         fprintf(stderr,"\nChoose a property type for surface #%d: ", ++n);
   if ( fscanf(infile, "%s", id) != 1) {
         fprintf(stderr,"\nError reading from surface definition file\n");
         exit(1);
   }
 
   if (strncmp(id, "end", 3) == 0) 
               return(NULL);          /* exit function */

   if (strncmp(id, "bot", 3) == 0) {
         surf = (struct surface *) malloc(sizeof(struct surface));
         surf->data_ind = -1;
         if (prompt) 
            fprintf(stderr,"Enter path/name of outfile for this surface: ");

         if (fscanf(infile, "%s", fname) != 1) {
            fprintf(stderr,"\nError reading surf.filename from surface definition file\n");
            exit(1);
         }
   
         if ((surf->fptr = fopen(fname, "w")) == NULL) {
             fprintf(stderr,"\nError opening %s for output.");
             exit(1);
         }

         return (surf);

   }


      if ((index = get_prop_indx(id)) < 0) {       /* is choice appropriate? */
          fprintf(stderr,"\n%s is not an option!\n\n", id);
          exit(1);    
      }  /* end if */ 

      if (index == (int) SF) {
          fprintf(stderr,"\n%s is not an appropriate option!\n\n", id);
          exit(1);    

      }


       surf = (struct surface *) malloc(sizeof(struct surface));

       surf->data_ind = index;

  /*  and reference pressure, if appropriate... */

      if ( index == (int)S_  ){
         if (prompt) {
            fprintf(stderr,"enter reference Pr:  ");
         }
         if (fscanf(infile, "%lf", &surf->pref) != 1) {
           fprintf(stderr,"\nError reading pref for property %s from surface definition file\n", id);
           exit(1);
         }
      }
  /* get value of surface ... */

      if (prompt) {
         fprintf(stderr,"enter %s value", get_prop_descrip(index));
         fprintf(stderr,": ");
      }
      if (fscanf(infile, "%lf", &surf->value) != 1) {
        fprintf(stderr,"\nError reading surf.value from surface definition file\n");
        exit(1);
      }

       
/* !**!  Special cases for individual properties... */

       switch ((enum property) index)  {
          case S_:
                 surf->density_flag = 1;
                 if (s_pref >= 0) {   /* has this already been requested? */
                     if ( NINT( s_pref) != NINT(surf->pref)) {
                         fprintf(stderr,"Sorry. You have requested the property s_ with multiple reference levels.\n");
                         fprintf(stderr,"You can only use one of those prefs");
                         fprintf(stderr,"  You will probably have to do multiple runs for each different pref you want to associate with s_\n\n");
                        exit(1);
                     }
                 }
                 s_pref = surf->pref;
                 prop_needed[index] = 1;     
                 break;
          case S0: 
                 surf->pref = 0.;
                 surf->density_flag = 1;
                 prop_needed[index] = 1;     
                 break;
          case S1: 
                 surf->pref = 1000.;
                 surf->density_flag = 1;
                 prop_needed[index] = 1;     
                 break;
          case S2: 
                 surf->pref = 2000.;
                 surf->density_flag = 1;
                 prop_needed[index] = 1;     
                 break;
          case S3: 
                 surf->pref = 3000;      
                 surf->density_flag = 1;
                 prop_needed[index] = 1;     
                 break;
          case S4: 
                 surf->pref = 4000.;
                 surf->density_flag = 1;
                 prop_needed[index] = 1;     
                 break;

          default:
                 surf->pref = 0.;       
                 surf->density_flag = 0;
                 prop_needed[index] = 1; 
                 break;   

       }  /* end switch */



/* open output file */

   if (prompt) 
      fprintf(stderr,"Enter path/name of outfile for this surface: ");

   if (fscanf(infile, "%s", fname) != 1) {
         fprintf(stderr,"\nError reading surf.filename from surface definition file\n");
         exit(1);
   }
   
   if ((surf->fptr = fopen(fname, "w")) == NULL) {
      fprintf(stderr,"\nError opening %s for output.");
      exit(1);
   }

   return (surf);

}  /* end add_surf() */

/*****************************************************************************/
void alloc_grids(s, n)
struct surface *s;
unsigned  n;
/*  Allocates memory for each appropriate array in the struct * passed as 
.    an argument and initializes all values to 0.
.    The second argument contains the amount of memory to allocate.
.    The count array for PR is always initialized since it is the benchmark
.    which determines whether there are any stations at a gridpt.   */
{
   register i, j;

   for (i = 0; i < MAXPROP; ++i ) {
   
      s->count[i] = NULL;
      s->prop[i] = NULL;
      s->propsq[i] = NULL;

      if (prop_req[i]) {
           if ((s->count[i] = (unsigned *) malloc(n * sizeof(unsigned))) == NULL) {
             fprintf(stderr,"\n\nUnable to allocate memory for s->count");
             exit(1);
           }
           if ((s->prop[i] = (double *) malloc(n * sizeof(double))) == NULL) {
             fprintf(stderr,"\n\nUnable to allocate memory for s->prop");
             exit(1);
           }
           if ((s->propsq[i] = (double *) malloc(n * sizeof(double))) == NULL) {
             fprintf(stderr,"\n\nUnable to allocate memory for s->propsq");
             exit(1);
           }

          /* initialize arrays */

           for (j = 0; j < n; ++j) {
              s->count[i][j] = 0;
              s->prop[i][j] = 0.;
              s->propsq[i][j] = 0.;
           }
       }
       else if ((enum property) i == PR) {
           s->count[i] = (unsigned *) malloc(n * sizeof(unsigned));
           for (j = 0; j < n; ++j) {
              s->count[i][j] = 0;
           }
       }

   } /* end for */

   return;

}  /* end alloc_grids() */

/****************************************************************************/
void get_hydro_data(file, listptr)
int file;
struct surface *listptr;
   /*  Reads each station in a HydroBase file and adds property values
       to the appropriate surfaces.   This module requires that the HydroBase
       file contains a minimum of pr, de, te, sa observations.  */
{
   int error, i;
   int row, col, sq;
   double dlat;
   short derived[MAXPROP];
   void  insertdata();
   struct surface  *surf;
   void    free_and_alloc();

/* initialize this so all derived properties get computed in insertdata() */

   for (i = 0; i < MAXPROP; ++i) 
            derived[i] = 0;


/* read each station in file ... */

    while ((error = get_station(file, &hdr, &station)) == 0) {

    /* compute the offset in the matrix into which this station falls */

       if (xdateline && hdr.lon < 0)
          hdr.lon += 360;
          
       if (hdr.lon < 0) 
          hdr.lon -= .0001;    /* this smidgeon puts stations that fall */
       else                /* right on a border into the proper box */
          hdr.lon += .0001; 

       if (hdr.lat < 0)
          hdr.lat -= .0001;
       else
          hdr.lat += .0001;    

       row = (int) ((hdr.lat - ymin) / delta_y);
       col = (int) ((hdr.lon - xmin) / delta_x);

  /* check for and skip over out of bounds stations   */

       if ((row >= 0) && (row < nrows) && (col >= 0) && (col < ncols))  {
         sq = row * ncols + col;

/* ensure that pr, de, te, and sa are available ... */

       if (!(available(PR, hdr) && available(DE, hdr) && available(TE, hdr) 
           && available(SA, hdr))) {
         fprintf(stderr,"\n>>>>> WARNING!!!  ");
         fprintf(stderr,"Station does not include pr, de, te, and sa.");
         break;
       }


 /* compute appropriate properties at each level in station ... derivative 
    properties (pot.vorticity, buoyancy...) get computed later in 
    insertdata(). */

/* !**! Special cases for individual properties... */

    for (i = 0; i < MAXPROP; ++i) {
       if (prop_needed[i]) {
          switch ((enum property) i) {
             case TH:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_theta(hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA] );
               break;
             case S0:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sigma(0., hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;
             case S1:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sigma(1000., hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;
             case S2:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sigma(2000., hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;
             case S3:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sigma(3000., hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;
             case S4:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sigma(4000., hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;
             case S_:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sigma(s_pref, hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;
             case HT:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_height(hdr.nobs, station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA], station.observ[i]);
               break;
             case SV:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sp_vol( hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;

             case VA:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_svan( hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;

             case PV:   /* fall through */
             case BF:
                  prop_needed[(int)PR] = 1;
                  break;

             default:
               break;
          } /* end switch */
       }

    }
      /* traverse the linked list of surfaces & interpolate data onto each 
         surface */

         dlat = (double) hdr.lat;
         surf = listptr;
         while (surf != NULL) {
            insertdata(surf, sq, hdr.nobs, dlat, derived);
            surf = surf->next;
         }  /* end while */

       } /* end if */

   }  /* end while */

   if (error > 0)
      report_status(error, stderr);

   return;

} /* end get_hydro_data() */

/****************************************************************************/

void free_and_alloc(dptr, n)
double **dptr;   
int n;           /* number of elements to allocate */

   /* Frees up the space pointed to by dptr.  It MUST have been allocated
      using malloc()!!  Then mallocs space for n double values and sets
      dptr pointing to it.  If insufficient memory, a message is written to
      stderr and an exit() is made. */
{

   if (*dptr != NULL) {
      free((char *)*dptr);
      *dptr = NULL;
   }

   *dptr = (double *) malloc(n * sizeof(double));

   if (*dptr == NULL) {
      fprintf(stderr, "\nInsufficient memory for call to malloc()\n");
      exit(1);
   }

   return;
}  /* end free_and_alloc() */
/****************************************************************************/

void insertdata(sptr, sq, nobs, lat, already_deriv)
struct surface *sptr;  /* defines projection surface */
int sq;         /* grid offset for this station */
int nobs;       /* number of observation levels in this station */
double lat;     /* latitude for this station */
short *already_deriv;  /* flags derivative props which are already computed */

   /*   Projects property values onto a surface and adds each value to the 
        appropriate grid element.   The station data arrays are global. */
{
   int i, datagap;
   double p0, d0, val, *d;
   double interpolate();
   double project_deriv_prop();
   double project_prop();


   if (sptr->data_ind == -1) {     /* return bottom observations */
        for (i = 0; i < MAXPROP; ++i ) {

          if (prop_req[i]) {   /* was this property requested? */

/* !**! Special cases for individual properties... */

             switch ((enum property) i) {

               case BF:    /* derivative properties */
               case PV:
                    if (already_deriv[i]) 
                       val = station.observ[i][nobs-1];
                    else
                       val = project_deriv_prop(i, station.observ[(int)PR][nobs-1], nobs, lat);
                    break;

               case OX:    /* optional observed properties */
               case N2:
               case N3:
               case P4:
               case SI:
                    if (!available( (enum property)i, hdr)) {
                       val = -9999.;
                       break;
                    }
                    /* if available fall through to default*/

               default:    /* all other properties */ 
                    val = station.observ[i][nobs-1];
                    break;
             } /* end switch */

             if (val > -9998.) {
               sptr->prop[i][sq] += val;
               sptr->propsq[i][sq] += (val * val);
               ++(sptr->count[i][sq]);
             }
          } 
        }  /* end for */
        return;
   }

/* surface is not the bottom... */

   d = station.observ[(int)DE];  /* set this ptr for easier referencing */

 /* determine if the surface exists at this station
    and check for vertical datagaps.  An unacceptable datagap is defined
    as :       > 200 m for surfaces in the upper 1000 m
               >1100 m for all other surfaces   */

   if ((d0 = interpolate(sptr->value, station.observ[sptr->data_ind], d, nobs)) > -9998.) {

     if (prop_needed[(int)PR])    /* pr is required for derivative props */
        p0 = interpolate(sptr->value, station.observ[sptr->data_ind], d, nobs);

       /* after this loop, d[i-1] and d[i] will bracket the de value
          associated with the projection surface ... */
     i = 0;
     while (d[++i] < d0 ) 
            ;

     if ((d[i-1] == d0) || (d[i] == d0) )
          datagap = 0;
     else if (d0 < 1001)
          datagap = (d[i] - d[i-1]) > 200;
     else
          datagap = (d[i] - d[i-1]) > 600;


      /* Exclude observations which are more than 1500 m above the 
         reference pressure, if one is specified.
         This prevents alot of bogus values from being added, but may cause 
         results to be different than expected if user is unaware of this. */

      datagap = datagap || (d0 < sptr->pref-1500);

     if (!datagap) {


        for (i = 0; i < MAXPROP; ++i ) {

          if (prop_req[i]) {   /* was this property requested? */

/* !**! Special cases for individual properties... */

             switch ((enum property) i) {

               case BF:    /* derivative properties */
               case PV:
                    if (already_deriv[i]) 
                       val = project_prop(station.observ[i], nobs, sptr);
                    else
                       val = project_deriv_prop(i, p0, nobs, lat);
                    break;

               case OX:    /* optional observed properties */
               case N2:
               case N3:
               case P4:
               case SI:
                    if (!available( (enum property)i, hdr)) {
                       val = -9999.;
                       break;
                    }
                    /* if available fall through to default*/

               default:    /* all other properties */ 
                    val = project_prop(station.observ[i], nobs, sptr);
                    break;
             } /* end switch */

             if (val > -9998.) {
               sptr->prop[i][sq] += val;
               sptr->propsq[i][sq] += (val * val);
               ++(sptr->count[i][sq]);
             }
          } 
          else {                          /* prop not requested */
           if ((enum property) i == PR)   /* even if pr is not being stored */
             ++(sptr->count[i][sq]);      /* the counter is always used */
          }
        }  /* end for */
      }   /* end if  !datagap */
       
   }
   else {    /* surface not at this station */

     /* if this is a density surface, check if it outcrops at the sea surface. 
        Outcropping surfaces get a zero added to the pressure and depth
        arrays (operationally their counters just get incremented). 
        First be sure that the station isn't missing too many surface
        observations ... */

     if ((sptr->density_flag) && (d[0] < 21.)) {
       if (sptr->value < station.observ[sptr->data_ind][0]) {
           ++(sptr->count[(int) PR][sq]);    
           if (prop_req[(int) DE])
             ++(sptr->count[(int)DE][sq]); 
       }   
     }

   }

   return;

}  /* end insertdata() */

/****************************************************************************/
double project_prop(obs, nobs, sptr)
double *obs;           /* array of observations for property */
int  nobs;             /* number of observations */
struct surface *sptr;  /* specifies the surface on which to project the prop */

   /*   Projects property values onto a specified surface...Returns the value
        of the property at the pressure level corresponding to this surface,
        or -9999 if the property cannot be projected.  The global arrays 
        pr & de are used.  */
{
   double val;
   double x[2], y[2];
   int    j;
   int   prevdepth, curdepth;



   prevdepth = curdepth = 0;
   j = 0;

   /* find first valid observation level. */

  while ((obs[j] < -8.9)||(station.observ[sptr->data_ind][j] < -8.9)) {
     if (++j == nobs)
          return (-9999.);
  }
  x[0] = station.observ[sptr->data_ind][j];
  y[0] = obs[j];

 /* Now, check successive pairs of datapoints until the 
    surface is found or the last observation is encountered.   */

  while (++j < nobs) {
    if ((obs[j] > -8.9) && (station.observ[sptr->data_ind][j] > -8.9)) {
        x[1] = station.observ[sptr->data_ind][j]; 
        y[1] = obs[j];
        if ((val = interpolate(sptr->value, x, y, 2)) > -9998.) {
           return (val);
        }
        x[0] = x[1];
        y[0] = y[1];
     }
  }  /* end while */

  return (-9999.);
} /* end project_prop() */
/****************************************************************************/
double project_deriv_prop(index, p0, nobs, lat)
int index;       /* identifies property to be projected */
double p0;       /* pressure at surface on which property will be projected */
int nobs;        /* number of p,t,s observation levels at this station */
double lat;      /* latitude of this station */

   /* Computes the value of a derivative property at the specified pressure 
      level from the global station data arrays.  */
{
   float deltap;
   double y;

/*  !**!  Special cases for individual properties... */

   switch ((enum property)index) {
      case BF:
          deltap = 50;   /* pressure interval in db.*/
          y = buoyancy(p0, station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA], nobs, deltap);
          if (y > -9998.)
             y = y * 1.0e5;
          break;
      case PV:
          deltap = 50;   /* pressure interval in db.*/
          y = buoyancy(p0, station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA], nobs, deltap);
          if (y < -9998.0)
              break;
          y = potvort((y*y), lat);
          break;
      default:
          y = -9999.0;
          break;

   }  /* end switch */

   return (y);

}  /* end project_deriv_prop() */
/****************************************************************************/
               
void get_cdf_data(cdfid, listptr)
int cdfid;
struct surface *listptr;
{
   struct CDF_HDR cdf;
   struct surface *surf;
   int startrow, startcol;
   int endrow, endcol;
   float minlat, minlon;
   float maxlat, maxlon;
   float lat, lon;
   double dlat;
   float *z, *x;
   short **count;
   char *mne;
   int error, i, j, k, sq, row, col, npts;
   int index_prop;
   short prop_avail[MAXPROP];     /* set of props in .cdf file */
   short compute_flag[MAXPROP];   /* props NOT in .cdf which can be computed*/
   void  insert_counted_data();
   void   insertdata();


   if (error = read_cdf_hdr(cdfid, &cdf)) {
       if (error == 2) {
         fprintf(stderr, "\n Error in read_cdf_hdr():");
         fprintf(stderr, " Not a HydroBase cdf file!\n");
         exit (1);
       }
       else {
         fprintf(stderr, "\n Unknown Error in read_cdf_hdr():");
         fprintf(stderr, "    error code = %d\n", error);
         exit (1);
       }
   }

/* compare bounds of file to these grid bounds to ensure overlap */

   if (xdateline && cdf.xmin < 0)
      cdf.xmin += 360;
   if (xdateline && cdf.xmax < 0)
      cdf.xmax += 360;
      
   if ((cdf.xmin > xmax) || (cdf.xmax < xmin) 
      || (cdf.ymin > ymax) || (cdf.ymax < ymin)) {
       fprintf(stderr, "\nNo data in cdf file is within grid bounds.\n");
       return;
   }

   if ((tbin < 0) || (tbin >= cdf.nt)) {
       fprintf(stderr, "\nTime bin index requested [%d] is not available in cdf file.\n", tbin);
       return;

   }

   minlon = (cdf.xmin < xmin) ? xmin : cdf.xmin;
   maxlon = (cdf.xmax > xmax) ? xmax : cdf.xmax;
   minlat = (cdf.ymin < ymin) ? ymin : cdf.ymin;
   maxlat = (cdf.ymax > ymax) ? ymax : cdf.ymax;
   
   error = get_indices(cdf, maxlat, minlon, &startrow, &startcol);
   error = get_indices(cdf, minlat, maxlon, &endrow, &endcol);
   
/* determine which properties are available in the cdf file */

   for (i = 0; i < MAXPROP; ++i) {  /* initialize these flags */
      prop_avail[i] = 0;
      compute_flag[i] = 0;
   }

   prop_avail[(int)DE] = 1;        /* depth is an index variable */
   prop_needed[(int)DE] = 1;

   for (i = 0; i < cdf.nprops; ++i)    
      prop_avail[get_prop_indx(cdf.prop_id[i])] = 1;
   
/* determine which properties can and should be computed ...   */

/*  !**! Special cases for individual properties */

   for (i = 0; i < MAXPROP; ++i) {
      if (prop_needed[i] && !prop_avail[i]) {
         switch ((enum property) i) {
            case S0: 
            case S1: 
            case S2:    /* fall through */
            case S3: 
            case S4: 
            case S_:
            case TH:
            case HT:
            case BF:
            case PV:
            case SV:
            case VA:
                compute_flag[i] = prop_avail[(int)PR] && prop_avail[(int) TE] 
                         && prop_avail[(int) SA];

                if (compute_flag[i] == 0) {
                  fprintf(stderr,"\n\n FATAL ERROR!");
                  fprintf(stderr,"\n Unable to compute %.2s for this file.\n", 
                          get_prop_mne(i));
                  fprintf(stderr,"The .cdf file must include pr, te, sa \n"); 
                  exit (0);
                }

                prop_needed[(int)PR] = 1;
                prop_needed[(int)TE] = 1;
                prop_needed[(int)SA] = 1;

                fprintf(stderr,"\n %.2s not available in cdf file, but will be computed from averaged p,t,s values.", get_prop_mne(i));
                break;

            default:
                fprintf(stderr,"\n\n FATAL ERROR!");
                fprintf(stderr,"\n  Property <%.2s> not available in this file.\n", get_prop_mne(i));
                exit(0);

         }  /* end switch */
      }
   }   /* end for */

/* get depth values */

   z = (float *) malloc ((unsigned) cdf.nz * sizeof(float));

   if (read_cdf_depths(cdfid, z) < 0) {
      fprintf(stderr,"\nError attempting to read_cdf_depths().\n"); 
      exit(1);
   }

/* allocate space for property and count arrays; initialize count arrays to 0 */

   for (i = 0; i < MAXPROP; ++i) {
      if (prop_needed[i]) {
         free_and_alloc(&station.observ[i], cdf.nz);
      }
   }

   x = (float *) malloc ((unsigned) cdf.nz * sizeof(float));
   
   if (cdf.counts_included) {
      count = (short **) malloc ((unsigned) MAXPROP * sizeof(short *));
      if (count == NULL) {
          fprintf(stderr, "\nInsufficient memory for call to malloc()");
          fprintf(stderr, "\n variable: (short **)count : get_cdf_data()");
          exit(1);
      }
      for (i = 0; i < MAXPROP; ++i) {
         count[i] = NULL;
         if (prop_needed[i]) {
            count[i] = (short *) malloc((unsigned) cdf.nz * sizeof(short));
            if (count[i] == NULL) {
              fprintf(stderr, "\nInsufficient memory for call to malloc()");
              fprintf(stderr, "\n variable: (short *)count[%d] : get_cdf_data()", i);
              exit(1);
            }
            for (j = 0; j < cdf.nz; ++j)
               count[i][j] = 0;
         }
      }
   }
   
/* visit each appropriate gridpt in the cdf file  */

   for (row = startrow; row <= endrow; ++row) {
       for (col = startcol; col <= endcol; ++col) {

          
          /* convert lat/lon of this data from cdf file to the grid position
             for the surfaces. */

           error = get_lat_lon(cdf, row, col, &lat, &lon);
           if ((lat > ymax) || (lat < ymin) || (lon > xmax) || (lon < xmin))
               continue;

           dlat = (double) lat;   
           sq = (int) ((lat - ymin) / delta_y) * ncols + 
                (int) ((lon - xmin) / delta_x);

           /* get bottom depth for this square ...*/

           read_cdf_bottom_depth(cdfid, &z[cdf.nz-1], row, col, tbin);

           for (i = 0; i < MAXPROP; ++i) {

              /* extract available properties ... */

              if (prop_needed[i] && prop_avail[i]) {

                 mne = get_prop_mne(i);
                 error = read_cdf_prop(cdfid, mne, x, row, col, tbin, 0, cdf.nz);
                 if (error > 0) {
                    fprintf(stderr,"\nError attempting to read %.2s at row,col =  %d,%d from cdf file.", mne, row, col);
                    fprintf(stderr,"\n   error code = %d\n", error);
                    exit(1);
                 }

                 if (cdf.counts_included) {
                    error = read_cdf_prop_count(cdfid, mne, count[i],
                             row, col, tbin, 0, cdf.nz);
                    if (error > 0) {
                       fprintf(stderr,"\nError attempting to read %.2s_cnt at row,col =  %d,%d from cdf file.", mne, row, col);
                       exit(1);
                    }

                 }

            /* place extracted data into appropriate column of station.observ */

                 for (j = 0; j < cdf.nz; ++j)
                    station.observ[i][j] = (double) x[j];
              }

           } /* end for  */

           /* choose the best property to determine whether any data is
              available at a given depth in .cdf file ... */

           if (station.observ[(int)TE] != NULL)
               index_prop = (int)TE;
           else if (station.observ[(int)SA] != NULL)
               index_prop = (int)SA;
           else if (station.observ[(int)PR] != NULL)
               index_prop = (int)PR;
           else 
               index_prop = get_prop_indx(cdf.prop_id[0]);

           /* eliminate levels with no data */

           k = 0;
           for (i = 0; i < cdf.nz; ++i) {
              if (station.observ[index_prop][i] > -8. ) {
                 for (j = 0; j < MAXPROP; ++j) {
                   if (prop_needed[j] && prop_avail[j] && (j != (int)DE)) {
                      station.observ[j][k] = station.observ[j][i];
                      if (cdf.counts_included && prop_avail[j])
                         count[j][k] = count[j][i];
                   }
                 }
                 station.observ[(int)DE][k] = (double) z[i];      
                 ++k;
              }
           }
           if ((npts = k) <= 0)
                continue;

           /* next, compute properties that were flagged. Set count for
              each computed property to 1 to differentiate these (which
              are computed from average property values) from
              properties that were computed from actual property values and
              then averaged.  */
      
/*  !**! Special cases for individual properties */
           for (i = 0; i < MAXPROP; ++i) {
              if (compute_flag[i]) {
                  switch ((enum property) i) {
                     case S0: 
                         compute_sigma(0., npts, station.observ[(int)S0], station.observ[(int)PR], station.observ[(int)TE], station.observ[(int)SA]);
                         break;
                     case S1: 
                         compute_sigma(1000., npts, station.observ[(int)S1], station.observ[(int)PR], station.observ[(int)TE], station.observ[(int)SA]);
                         break;
                     case S2:   
                         compute_sigma(2000., npts, station.observ[(int)S2], station.observ[(int)PR], station.observ[(int)TE], station.observ[(int)SA]);
                         break;
                     case S3: 
                         compute_sigma(3000., npts, station.observ[(int)S3], station.observ[(int)PR], station.observ[(int)TE], station.observ[(int)SA]);
                         break;
                     case S4: 
                         compute_sigma(4000., npts, station.observ[(int)S4], station.observ[(int)PR], station.observ[(int)TE], station.observ[(int)SA]);
                         break;
                     case S_: 
                         compute_sigma(s_pref, npts, station.observ[(int)S_], station.observ[(int)PR], station.observ[(int)TE], station.observ[(int)SA]);
                         break;
                     case TH:
                         compute_theta(npts, station.observ[(int)TH], station.observ[(int)PR], station.observ[(int)TE], station.observ[(int)SA]);
                         break;
                     case HT:
                         compute_height(npts, station.observ[(int)PR], station.observ[(int)TE], station.observ[(int)SA], station.observ[(int)HT]);
                         break;
                     case SV: 
                         compute_sp_vol(npts, station.observ[(int)SV], station.observ[(int)PR], station.observ[(int)TE], station.observ[(int)SA]);
                         break;
                     case VA: 
                         compute_svan( npts, station.observ[(int)VA], station.observ[(int)PR], station.observ[(int)TE], station.observ[(int)SA]);
                         break;
                     default:
                         break;
              
                  }  /* end switch */

                  if (cdf.counts_included) {
                         for (j = 0; j < npts; ++j) {
                            count[i][j] = 1;
                         }
                  }

              }
           }

         /* traverse the linked list of surfaces & interpolate data onto each 
            surface */

           surf = listptr;
           while (surf != NULL) {

              if (cdf.counts_included) 
                 insert_counted_data(surf, sq, npts, count, dlat, prop_avail);
              else 
                 insertdata(surf, sq, npts, dlat, prop_avail);

              surf = surf->next;
           }  /* end while */

       }  /* end for col */
   }  /* end for row */

   free((char *) z);
   free((char *) x);
   for (i = 0; i < MAXPROP; ++i) {
       if (station.observ[i] != NULL) {
          free(station.observ[i]);
          station.observ[i] = NULL;
       }
       if (cdf.counts_included && (count[i] != NULL)) {
          free(count[i]);
          count[i] = NULL;
       }
   }
   if (cdf.counts_included)
      free(count);

   return;
}  /* end get_cdf_data() */
/****************************************************************************/

void insert_counted_data(sptr, sq, nlevs, count, lat, already_deriv)
struct surface *sptr;
int sq, nlevs;
short **count;
double lat;
short *already_deriv;
   /*   Projects property values onto each surface and adds  values to the 
        appropriate grid element. The property arrays are globally defined,
        the address of the count arrays is passed as an argument. */
{
   double d0, p0, val, r;
   double *x, *d;
   double interpolate();
   double project_prop();
   double project_deriv_prop();
   int    nobs, datagap;                   
   int    i, j;

/*  bottom observations were requested... */

   if (sptr->data_ind == -1) {     
        for (i = 0; i < MAXPROP; ++i ) {

          if (prop_req[i]) {   /* was this property requested? */

/* !**! Special cases for individual properties... */

             switch ((enum property) i) {

               case BF:    /* derivative properties */
               case PV:
                    if (already_deriv[i]) 
                       val = station.observ[i][nlevs-1];
                    else
                       val = project_deriv_prop(i, station.observ[(int)PR][nlevs-1], nlevs, lat);
                    break;

               case OX:    /* optional observed properties */
               case N2:
               case N3:
               case P4:
               case SI:
                    if (!available( (enum property)i, hdr)) {
                       val = -9999.;
                       break;
                    }
                    /* if available fall through to default*/

               default:    /* all other properties */ 
                    val = station.observ[i][nlevs-1];
                    break;
             } /* end switch */

             if (val > -9998.) {
               sptr->prop[i][sq] += val;
               sptr->propsq[i][sq] += (val * val);
               ++(sptr->count[i][sq]);
             }
          } 
        }  /* end for */
        return;
   }

/* surface is not the bottom ... */

   d = station.observ[(int)DE];     /* set this pointer for easier references */

 /* determine if the surface exists at this station
    and check for vertical datagaps.  An unacceptable datagap is defined
    as :       > 200 m for surfaces in the upper 1000 m
               > 600 m for all other surfaces   */


   if ((d0 = interpolate(sptr->value, station.observ[sptr->data_ind], d, nlevs)) > -9998.) {

     if (prop_needed[(int)PR])         /* pr is needed to do derivative props */
        p0 = interpolate(sptr->value, station.observ[sptr->data_ind], station.observ[(int)PR], nlevs);

       /* after this loop, d[i-1] and d[i] will bracket the depth value
          associated with the projection surface ... */
     i = 0;
     while (d[++i] < d0 ) 
            ;

     if ((d[i-1] == d0) || (d[i] == d0) )
          datagap = 0;
     else if (d0 < 1001)
          datagap = (d[i] - d[i-1]) > 200;
     else
          datagap = (d[i] - d[i-1]) > 600;


      /* Exclude observations which are more than 1500 m above the 
         reference pressure, if one is specified.
         This prevents alot of bogus values from being added, but may cause 
         results to be different than expected if user is unaware of this. */

      datagap = datagap || (d0 < sptr->pref-1500);

     if (!datagap) {

        x = (double *) malloc(nlevs * sizeof(double));

        for (i = 0; i < MAXPROP; ++i ) {

          if (prop_req[i]) {   /* was this property requested? */
             switch ((enum property) i) {
                case BF:
                case PV:
                    if (already_deriv[i]) 
                       val = project_prop(station.observ[i], nlevs, sptr);
                    else
                       val = project_deriv_prop(i, p0, nlevs, lat);
                    break;
                default:
                    val = project_prop(station.observ[i], nlevs, sptr);
                    break;
             } /* end switch */

             if (val > -9998.) {
                 /* determine number of obs which contributed to mean value */
                  for (j = 0; j < nlevs; ++j) {
                      x[j] = (double) count[i][j];
                  }
            /* the ceil() function doesn't seem to work so do it the long way*/
                  r = interpolate(d0, d, x, nlevs);
                  nobs = (r - (int) r) > 0 ? (int) r + 1: (int) r;

                  sptr->prop[i][sq] += val * nobs;
                  sptr->propsq[i][sq] += sptr->prop[i][sq] * val;
                  sptr->count[i][sq]+= nobs;
             }

          } 
          else {                          /* prop not requested */
           if ((enum property) i == PR)   /* even if pr is not being stored */
             ++(sptr->count[i][sq]);      /* the counter is always used */
          }
        }  /* end for */

        free(x);
     } /* end if !datagap */
       
   }
   else {    /* surface not at this station */

     /* if this is a density surface, check if it outcrops at the sea surface.  
        Outcropping surfaces get a zero added to the pressure and depth arrays.
        First be sure that it isn't missing too many surface observations ... */

     if ((sptr->density_flag) && (d[0] < 21.)) {
       if (sptr->value < station.observ[sptr->data_ind][0]) {
           ++(sptr->count[(int) PR][sq]);    
           if (prop_req[(int) DE])
             ++(sptr->count[(int)DE][sq]);   
       }   
     }

   }

   return;

}  /* end insert_counted_data() */

/****************************************************************************/

/****************************************************************************/
/* Performs a linear interpolation to find the position of xval in array, x,
   and returns the corresponding value in the array, y.  If xval does not
   appear in array x, the value -9999 is returned.  This routine assumes that
   the x array is monotonic and continuous (no missing values); and it
   assumes the y array is continuous.    */

double interpolate(xval, x, y, nypts)
double xval, *x, *y;
int    nypts;
{
   int    k;
   double v1, v2;

   for (k = 0; k < nypts-1; ++k) {

      v1 = xval - x[k];
      v2 = xval - x[k+1];

      if (v1 == 0)             /* x[k] == xval */
          return (y[k]);
      if (v2 == 0)             /* x[k+1] == xval */
          return (y[k+1]);
      if (v1 < 0. && v2 < 0.)  /* xval not between x1 and x2 */  
          continue;
      if (v1 > 0. && v2 > 0.) 
          continue;

      return ( y[k] + (y[k+1] - y[k]) * v1 / (x[k+1] - x[k]) );
   }

   return (-9999.0);

}   /* end interpolate() */


/****************************************************************************/
void do_stats(listptr)
struct surface *listptr;
{
   struct surface *surf;
   int    sq, row, col, i, n;
   float  lat, lon;
   float  mean, stddev;
   double var;

   surf = listptr;
   while (surf != NULL) {     /* traverse linked list of surfaces */
        sq = -1;      /* initialize, then simply visit each square*/

        for (row = 0; row < nrows; ++row) {
          for (col = 0; col < ncols; ++col) {
             lat = (row  + .5) * delta_y + ymin ;
             lon = (col + .5) * delta_x + xmin;
             ++sq;

             if (surf->count[(int)PR][sq] > 0) {
               fprintf(surf->fptr,"%7.3f %8.3f", lat, lon);
               for (i = 0; i < MAXPROP; ++i ) {
                 if (prop_req[i]) {
                   mean = surf->prop[i][sq];
                   stddev = 0.;
                   if ((n = surf->count[i][sq]) > 1) {
                      var = (surf->propsq[i][sq] - surf->prop[i][sq] * surf->prop[i][sq] / (double) n) / (double) (n-1);
                      stddev = (float) sqrt (ABS(var));
                      mean = (float) surf->prop[i][sq] / (float) n;
                   }
                   fprintf(surf->fptr,"  %8.3f %8.3f %4d", mean, stddev, n);
                 }
               } /* end for */
               fprintf(surf->fptr,"\n");
             }

           }   /* end for */
        } /* end for */

       fclose(surf->fptr);
       surf = surf->next;

   }  /* end while */

   return;
}  /* end do_stats() */

