/*  hydro_utils.c
.
.  A set of routines to facilitate reading, writing files in HydroBase
.  format.
................................................................................
................................................................................
 int  open_hydro_file(char *dir, char *root, char *extent, int print_msg):
       Opens an existing hydro file for reading.
 int  get_station(int file, struct HYDRO_HDR *h_addr, struct HYDRO_DATA *d_ptr):
       Reads a complete station.
 int  read_hydro_hdr(int file, struct HYDRO_HDR *haddr):    
       Gets station header info.
 void report_status(int status, file *file)
       Reports any errors reported by read_hydro_hdr().
 int  get_data_scan(int file, double *scan, int n, int *prop_order) :  
       Reads the next scan in an open hydro file
 int  get_separator(int file) :  
       Advances file beyond the station separator.
 int  create_hydro_file(char *fname, int mode):  
       Opens a HydroBase file for writing with specified mode.
 int  write_hydro_station(int file, struct HYDRO_HDR h, struct HYDRO_DATA data):
       Writes an entire station to an open hydro file .  
 int  write_hydro_hdr(int file, struct HYDRO_HDR h) : 
       Writes a header to an open hydro file 
 int  write_hydro_scan(int file, double *scan, int n, int *prop_order) : 
       Writes an individual data scan 
 int  write_separator(int file) : 
       Writes a separator line.
 int  ms10(float lat, lon, int *ms1_ptr) : 
       Computes the 10-degree & 1-degree Marsden Square # for a lat,lon pair.
 int  available(enum property p, struct HYDRO_HDR hdr) :
       Returns a 1 if the specified property is available for a station. 
................................................................................
................................................................................
*/
#include <stdio.h>
#include <fcntl.h>
#include <malloc.h>
#include "hydrobase.h"


#define LF  0x0a       /* ASCII code for linefeed */

extern char *get_prop_mne();

/**********************************************************************/

int open_hydro_file(dir, root, extent, print_mess)
char *dir;      /* name of directory   or "" */
char *root;     /* root of filename    or full path to file */
char *extent;   /* extent for file     or "" */
int  print_mess; /* 0 to suppress printing messages on stderr device */

/* opens an existing hydro file for READING only. Returns a file
   descriptor if successful or -1 if an error occurs. */  

{ 
   char fname[80];
   int  file, i;

    strcpy(fname, dir);
     if ((i = strlen(dir)) != 0) {
        if (dir[i-1] != '/')
           strncat(fname,"/",1);
     }
     strcat(fname, root);
     strcat(fname, extent);

     file = open(fname, O_RDONLY);

     if (print_mess) {
        if (file < 0) {
          fprintf (stderr,"\nunable to open %s for input\n\n", fname);
        }
        else {
          fprintf(stderr,"\nOpened %s ...\n", fname);
        }
     }

     return (file);
   
}  /* end open_hydro_file() */


/**********************************************************************/

int get_station(file, h_addr, d_ptr)
int   file;                /* already opened file handle */
struct HYDRO_HDR *h_addr;  /* station header info (already allocated) */
struct HYDRO_DATA *d_ptr;  /* pointer to station data (already allocated)*/

  /* retrieves an entire station and places the data in  struct HYDRO_DATA. 
     It advances beyond the station
     separator at the end of the current station. Returns 0 for a successful
     read, -1 for end_of_file,  and an error code if an error occurs.

      error code =  1 :  error parsing header.
                    2 :  error parsing datascan.
                    3 :  error reading station separator.
                    4 :  unexpected end of file occurred.
  */
{
   int i, j, iprop;
   int read_hydro_hdr(), get_data_scan();
   char line[NBSEP];
   double *scan;
 
   if ((i = read_hydro_hdr(file, h_addr)) < 0) 
      return (-1);
   else if (i > 0)
      return (1);
   else
      ;

/* assign values at d_ptr and allocate memory for data scans */

   d_ptr->nobs = h_addr->nobs;
   d_ptr->nprops = h_addr->nprops;
   for (i = 0; i < d_ptr->nprops; ++i) {
      iprop = h_addr->prop_id[i]; 
      if (d_ptr->observ[iprop] != NULL) {
           free(d_ptr->observ[iprop]);
           d_ptr->observ[iprop] = NULL;
      }
      d_ptr->observ[iprop] = (double *) malloc(h_addr->nobs * sizeof(double));
      if (d_ptr->observ[iprop] == NULL) {
         fprintf(stderr,"\nUnable to allocate memory in get_station()\n");
         exit(1);
      }
   }

   scan = (double *) malloc(h_addr->nprops * sizeof(double));

/* read each scan from file */
   
   for (i = 0; i < h_addr->nobs; ++i) {
      if (get_data_scan(file, scan, h_addr->nprops, h_addr->prop_id) > 0) {
           free(scan);
           return (2);
      }
      for (j = 0; j < d_ptr->nprops; ++j) {
         iprop = h_addr->prop_id[j]; 
         d_ptr->observ[iprop][i] = scan[j];
      }
   }  /* end for */

   free(scan);

   if (read(file, line, NBSEP) != NBSEP) {   /* move past station separator */
       return (3);
   }

   return (0);
}  /* end get_station() */

/**********************************************************************/

int read_hydro_hdr(file, haddr)
int file;
struct HYDRO_HDR *haddr;

/*   reads a header from an already opened hydro file, parses the
    information and stores it at haddr.  Returns 0 for a successful
    read, -1 for end-of-file, or a code > 0 to indicate an error.

          error codes returned :
                1 : error parsing header.
                2 : error parsing property codes
*/
{
    char *s, line[NBHEAD];
    int i, nbytes;

    if ( read(file, line, NBHEAD) != NBHEAD)
         return (-1);
    if (sscanf(&line[0],"%s", &haddr->country[0]) != 1)
         return (1);
    if (sscanf(&line[3],"%s", &haddr->ship[0]) != 1)
         return (1);
    if (sscanf(&line[6],"%d", &haddr->cruise) != 1)
         return (1);
    if (sscanf(&line[11],"%d", &haddr->station) != 1)
         return (1);
    if (sscanf(&line[17],"%d %d %d", &haddr->year, &haddr->month, &haddr->day) != 3)
         return (1);
    if (sscanf(&line[30],"%f %f", &haddr->lat, &haddr->lon) != 2)
         return (1);
    if (sscanf(&line[49],"%d %d %d", &haddr->pdr, &haddr->nobs, &haddr->nprops) != 3)
         return (1);
    if (sscanf(&line[67],"%d %d", &haddr->ms10, &haddr->ms1) != 2)
         return (1);

    nbytes = 3 * haddr->nprops;  /* 3 bytes per prop incl trailing whitespace */
    if (read(file, line, nbytes) != nbytes)
        return (2);

    if ( line[nbytes-1] != LF )  /* check that last char read was a LF */
        return (2);

/* allocate space to cross-reference properties for this station */

    if (haddr->prop_id != NULL)
         free(haddr->prop_id);

    haddr->prop_id = (int *) malloc(haddr->nprops * sizeof(int));

/* now get the properties */
    s = line;
    for (i = 0; i < haddr->nprops; ++i) {
        if ((haddr->prop_id[i] = get_prop_indx(s)) < 0) {
           fprintf(stderr,"\n Unknown property listed in file: %.2s\n", s);
           return (2);
        }
        s += 3;   /* advance to next property */
    }
    return(0);

} /* end read_hydro_hdr() */

/**********************************************************************/

void report_status(status, file)
int status;
FILE *file;
{
switch (status) {
       case -1:
            break;
       case 0:
            break;
       case 1:
            fprintf(file,"\nError parsing header in get_station(). \n");
            break;
       case 2:
            fprintf(file,"\nError parsing datascan in get_station(). \n");
            break;
       case 3:
            fprintf(file,"\nError reading station separator in get_station(). \n");
            break;
       case 4:
            fprintf(file,"\nUnexpected eof encountered in get_station(). \n");
            break;
       default:
            fprintf(file,"\n Unknown error code returned by get_station() : %d\n", status);

} /* end switch */
return;
}  /* end report_status() */

/**********************************************************************/

int get_data_scan(file, scan, n, prop_order)
int   file;                /* already opened file */
double *scan;              /* array to store data values */
int   n;                   /* number of properties per scan */
int *prop_order;           /* index to enumerated property list */

  /* Reads and parses one data scan.  Returns 0 for a successful
     read  or an error code if an error occurs.

      error code =  2 :  error parsing datascan.
                    4 :  unexpected end of file occurred.
  */
{
   char *s, *line;
   int nbytes, i;

/* determine number of bytes in a scan allowing for the LF at the end... */

   nbytes = 0;
   for (i = 0; i < n; ++i) {             
     nbytes += get_field_width(prop_order[i]) + 1;  /* add a space to field */
   }
   ++nbytes;                                        /* allow for LF */
   line = (char *) malloc(nbytes * sizeof(char));
 
   if (read(file, line, nbytes) != nbytes ) {
          free(line);
          return(4);  
   }

   if (line[nbytes-1] != LF) {
     free(line);
     return(2);
   }

/* parse the line into property fields... */
   s = line;
   for (i = 0; i < n; ++i) {             
      if (sscanf(s,"%lf", &scan[i] ) != 1) {
          free(line);
          return(2);
      }
      s += get_field_width(prop_order[i]) + 1;
   }

   free(line);
   return (0);

} /* get_data_scan() */
/**********************************************************************/

int get_separator(file)
int  file;       /* file descriptor for already opened file */

   /* Advances a file beyond the station separator.  Returns 0 for
      a successful read or 1 if an error occurs.  */
{
   char line[NBSEP];

   if (read(file, line, NBSEP) != NBSEP) {   /* move past station separator */
       return (1);
   }

   return (0);

}  /* end get_separator() */

/**********************************************************************/
int create_hydro_file(fname, mode)
char *fname;     /* string containing name of file to be opened */
int  mode;  /* 0=overwrite if file exists, 1=noclobber, 2=append */

/*  opens a file for writing in specified mode. An error will occur
    if noclobber is specified and the file already exists. Returns a file  
    descriptor for a successful open,  or -1 for an error */
{
   int file;

   switch (mode) {
       case APPEND:
               if ((file = open(fname, 1)) >= 0) {
                  lseek(file,0L,2);  /* position pointer at end of file */
                  return(file);
               }
               /* else fall through */
       case OVERWRITE:
          if ((file = creat(fname, 0666)) < 0) { 
              return(-1);
          }
          return(file);

       case NOCLOBBER:  /* fall through */
       default:
          if ((file = open(fname, 1)) >= 0) {  /* check if file exists already*/
              return (-1);                    /* return an error if it does */
          }

          if ((file = creat(fname, 0666)) < 0) { /* create if it doesn't exist*/
              return(-1);
          }
          return(file);

   } /* end switch */
   
}  /* end create_hydro_file() */
/**********************************************************************/
int write_hydro_station(file, h, data)
int file;              /* file descriptor of already open file */
struct HYDRO_HDR h;    /* header info */
struct HYDRO_DATA data; 
   /* writes a complete station to an open file.  Returns 0 for
      a successful write or a number > 0 if an error occurs.
   */
{
   int i, j;
   double *scan;
   int write_hydro_hdr(), write_hydro_scan();
   int write_separator();

   if (write_hydro_hdr(file, h) > 0)
      return (1);

   scan = (double *) malloc(h.nprops * sizeof(double));

   for (i = 0; i < h.nobs; ++i) {
      for (j = 0; j < h.nprops; ++j) {
         scan[j] = data.observ[h.prop_id[j]][i];
      }
      if (write_hydro_scan(file, scan, h.nprops, h.prop_id) > 0) {
            free(scan);
            return(2);
      }
   }
   free(scan);

   write_separator(file);

   return(0);

} /* write_hydro_station() */
 
/**********************************************************************/

int write_hydro_hdr(file, h)
int file;             /* file descriptor for already open file */
struct HYDRO_HDR h;   /* header info */

   /* writes a station header. Returns 0 for a successful operation or
      number > 0 if an error occurs. 
               error code : 2 :  error writing line to file. */
{

   char line[NBHEAD], *s;
   int nbytes, i;

/* compose first header line and write to file ...*/

    sprintf(line,"%.2s %.2s %4d %4d %5d %2d %2d %9.3f %9.3f %5d %5d %5d %5d %3d", h.country, h.ship, h.cruise, h.station, h.year, h.month, h.day, h.lat, h.lon, h.pdr, h.nobs, h.nprops, h.ms10, h.ms1); 

   line[NBHEAD-1] = LF;
   if (write(file, line, NBHEAD) != NBHEAD)
     return (2);

/*  compose line of property descriptors and write to file ... */

   nbytes = h.nprops * 3;     /* # of bytes needed to write prop descriptors */
   s = line;
   for (i = 0; i < h.nprops; ++i) {
      sprintf(s,"%.2s ", get_prop_mne(h.prop_id[i]));
      s += 3;
   }
   line[nbytes-1] = LF;

   if (write(file, line, nbytes) != nbytes)
      return (2);

   return(0);
} /* end write_hydro_hdr() */

/**********************************************************************/

int write_hydro_scan(file, scan, n, prop_id)
int file;          /* file descriptor of file opened for writing */
double *scan;      /* array of property values in appropriate order */
int n;             /* number of properties */
int *prop_id;      /* index to enum property for each value in scan */

   /* writes a single data scan to an open hydro file.  Returns 0 for
      a successful write or a number > 0 if an error occurs.
           error code =   2 :  error writing to output file.
   */
{

   char *line, *s, str[10];
   int nbytes, i, jj, kk, k;

/* determine number of bytes needed for scan */

   nbytes = 0;
   for (i = 0; i < n; ++i) {
      nbytes += get_field_width(prop_id[i]) + 1;
   }
   ++nbytes;         /* allow for LF */
   line = (char *) malloc(nbytes * sizeof(char));
   s = line;

/* now write value in format appropriate to each property */

   for (i = 0; i < n; ++i ) {
      if (prop_id[i] < 0 || prop_id[i] >= MAXPROP) {
          fprintf(stderr,"FATAL ERROR: prop_id index passed to write_hydro_scan() is out of range!!");
          exit(1);
      }
      sprintf(str," %c%d.%dlf", '%', get_field_width(prop_id[i]), 
              get_field_precis(prop_id[i]));
              
      /* some  values are too large to fit in the field width: */
      if (scan[i] < -998)
           scan[i] = -9.0;
           
      kk = get_field_width(prop_id[i])- get_field_precis(prop_id[i]) - 1;
      jj = 1;
      for (k = 2; k < kk; ++k) {
           jj *= 10;
      }
      if (scan[i] > (9.999 * jj)) {
        scan[i] = -9.9;
      }
        
      sprintf(s, str, scan[i]);
      s += get_field_width(prop_id[i]) + 1;
   } /* end for */

   line[nbytes-1] = LF;
   if (write(file, line, nbytes) != nbytes) {
        free(line);
        return (2);
   }

   free(line);
   return (0);

}  /* end write_hydro_scan() */

/**********************************************************************/
int write_separator(file)
int file;       /* file descriptor of file open for writing. */

   /* writes a separator line to a hydro file.  Returns 0 for a successful
      write or a code > 0 if an error occurs.
   */
{
   char line[NBSEP];

   line[0] = '*';
   line[1] = '*';
   line[2] = LF;
   if (write(file, line, NBSEP) != NBSEP)
       return (2);

   return (0);     

}  /* end write_separator() */

/**********************************************************************/
int ms10(lat, lon, ms1_ptr)
float lat, lon;
int *ms1_ptr;
   /*  Computes the 10-degree and 1-degree Marsden Square designations for a 
       particular lat/lon combination. */
{
   static int  quadrant[2][2] = { 7000, 1000,
                                  5000, 3000 };
   int  k, kk;
   int  ilat, ilon;


   k  = (lat < 0.) ? 1 : 0;   /* determine earth quadrant */
   kk = (lon < 0.) ? 0 : 1;
   if (lon > 180.) kk = 0;
   
   ilat = (int) (lat + .00001);  /* this smidgeon handles borderline */
   ilon = (int) (lon + .00001);  /*  cases correctly in all hemispheres */

   if (ilat < 0) ilat = -ilat;
   if (ilon < 0) ilon = -ilon;

   if (ilat >= 90) ilat = 89;    /* special case at the poles */

   *ms1_ptr = (ilat % 10) * 10 + (ilon % 10);

   return (quadrant[k][kk] + (ilat / 10) * 100 + (ilon / 10));

}  /* end ms10() */

/**********************************************************************/
int available(prop, hdr)
enum property prop;
struct HYDRO_HDR hdr;
   /*  Determines if the specified property is available at a particular
       station.  Returns 0 if not, 1 if available.  */
{
   int i, found;

   found = 0;
   i = -1;
   while (++i < hdr.nprops) {
         if (hdr.prop_id[i] == (int) prop)
             return (1);
   }
   return (0);

}  /* end available() */
