*
*   algorithms for oceanographic computations
*            n fofonoff & r millard
*
*   sal78(cond, t, p, iflag)  ! salt from cond ratio or vice versa
*   svan(s, tref, pref, sigma)! specific volume anomaly  
*   depth(p, lat)             ! depth from pressure
*   tf(s,p)                   ! freezing point of seawater
*   cpsw(s, t, p0)            ! specific heat of seawater
*   atg(s, t, p)              ! adiabatic temp gradient
*   theta(s, t, p, pref)      ! potential temperature referenced to pref
*   svel(s, t, p0)            ! speed of sound in seawater
*   bvfrq(s, t, p, n, pav, e) ! brunt-vaisala frequency from sp.vol.anom.
*   bvgrdts(p0, s, t, p, n, e)! brunt-vaisala frequency at a specified pressure from t, s gradients
*   grady(y,p,nobs,pav,ybar)  ! least squares slope of y vs. p
*   p80( dpth, xlat)          ! pressure from depth using eos80
*   gravity(xlat)             ! grav acceleration at latitude (cm/sec**2)
*   coriol(xlat)              ! coriolus parameter from latitude
*
*
****************************************************************************
****************************************************************************
      real*8 function sal78(x,t,p,iflag) 
*...........................................................................
* function to convert conductivity ratio to salinity (iflag = 0)
*                  OR salinity to conductivity ratio (iflag = 1)
*     references:   also located in unesco report # 37 1981
*        practical salinity scale 1978: e.l. lewis ieee ocean eng. jan. 1980
*...........................................................................
*   args:  
*       x      :  conductivity ratio (iflag = 0) OR salinity (iflag = 1)
*       p      :  pressure ( decibars)
*       t      :  temperature  (deg celsius) 
*       iflag  :  =0 (to convert cond ratio to salinity)
*                 =1 (to convert salinity to cond ratio)
*   returns salinity (iflag = 0)
*        or conductivity ratio (iflag = 1)
*...........................................................................
*   internal functions defined...
*
*     sal(xr,xt) : practical salinity scale (1978 definition w/ temp correction)
*                  xt = t - 15.0  
*                  xr = sqrt(rt)
*    dsal(xr,xt) : returns derivative of sal(xr,xt) with respect to xr.
*
*      rt35(xt)  : c(35,t,0) / c(35,15,0) variation with temperature
*
      implicit real*8 (a-h,o-z)
*
      sal(xr,xt) =((((2.7081*xr-7.0261)*xr+14.0941)*xr+25.3851)*xr
     &            -0.1692)*  xr+0.0080 
     &            +(xt/(1.0+0.0162*xt))*(((((-0.0144*xr+
     &            0.0636)*xr-0.0375)*xr-0.0066)*xr-0.0056)*xr+0.0005)
*
      dsal(xr,xt) =((((13.5405*xr-28.1044)*xr+42.2823)*xr+50.7702)*xr 
     &   -0.1692)+(xt/(1.0+0.0162*xt))*((((-0.0720*xr+0.2544)*xr
     &   -0.1125)*xr-0.0132)*xr-0.0056) 
*
      rt35(xt) = (((1.0031e-9*xt-6.9698e-7)*xt+1.104259e-4)*xt
     &   + 2.00564e-2)*xt + 0.6766097 
* ...........................................................................
*
* polynomials of rp: c(s,t,p)/c(s,t,0) variation with pressure 
*  c(xp) polynomial corresponds to a1-a3 constants: lewis 1980
*  a(xt) polynomial corresponds to b3 and b4 constants: lewis 1980
*
      c(xp) = ((3.989e-15*xp-6.370e-10)*xp+2.070e-5)*xp 
      b(xt) = (4.464e-4*xt+3.426e-2)*xt + 1.0 
      a(xt) = -3.107e-3*xt + 0.4215 
* ...........................................................................
*  check for values too small...
*      returns zero for cond ratio < 0.0005     (iflag = 0)
*                or for salinity < 0.02         (iflag = 1)
*
      sal78 = 0.0
      if((iflag.eq.0).and.(x .le. 5e-4)) return
      if((iflag.eq.1).and.(x .le. 0.02)) return
*
      dt = t - 15.0
* 
* ......................................................
*  convert conductivity ratio to salinity ...
* ......................................................
*
      if(iflag .eq. 0) then
         r = x
         rt = r/(rt35(t)*(1.0 + c(p)/(b(t) + a(t)*r))) 
         rt = sqrt(abs(rt)) 
         sal78 = sal(rt,dt)
         return  
      end if
*
* ......................................................
*        salinity to conductivity ratio ...
* ......................................................
* invert salinity to conductivity by the
*  newton-raphson iterative method.
*
      salt = x
*
*  set initial values ...
*
      rt = sqrt(salt/35.0)
      si = sal(rt,dt)
      n = 0   
      dels = 1.
*
*  iteratively refine salinity inversion ... 
*
      do while ((dels.gt.1.0e-4).and.(n.lt.10))
         rt = rt + (salt - si)/dsal(rt,dt)
         si = sal(rt,dt) 
         n = n + 1 
         dels = abs(si - salt)
      end do
*
* compute conductivity ratio ... 
*
      rtt = rt35(t) * rt * rt 
      at = a(t) 
      bt = b(t) 
      cp = c(p) 
      cp = rtt * (cp + bt)
      bt = bt - rtt * at
*             
* solve quadratic equation for r: r = rt35 * rt * (1+c/ar+b) 
*
      r = sqrt(abs(bt*bt + 4.0*at*cp)) - bt 
*
* return conductivity
      sal78 = 0.5 * r / at
      return  
      end   
*
****************************************************************************
****************************************************************************
      real*8 function svan(s,t,p0,sigma)
*
* Specific Volume Anomaly (steric anomaly) based on 1980 equation
* of state for seawater and 1978 practical salinity scale.
* References:
*        millero, et al (1980) deep-sea res.,27a,255-264
*        millero and poisson 1981,deep-sea res.,28a pp 625-629.
*   (both  references are also found in unesco report 38 -- 1981)
* ................................................................
* units:      
*       p0  :    pressure  [or ref pressure] (decibars)
*        t  :    temperature  [or pot temp]  (deg C)
*        s  :    salinity (ipss-78)
*     svan  :    spec. vol. anom.  (m**3/kg *1.0e-8)
*     sigma :    density anomaly   (kg/m**3)
* ................................................................
* check value: svan=981.3021 e-8 m**3/kg.  for s = 40 (ipss-78) ,
* t = 40 deg c, p0= 10000 decibars.
* check value: sigma = 59.82037  kg/m**3 for s = 40 (ipss-78) ,
* t = 40 deg c, p0= 10000 decibars.
* ................................................................
      implicit real*8 (a-h,o-z)

      real*8 p,t,s,sig,sr,r1,r2,r3,r4
      real*8 a,b,c,d,e,a1,b1,aw,bw,k,k0,kw,k35
*
      equivalence (e,d,b1), (bw,b,r3), (c,a1,r2) 
      equivalence (aw,a,r1),(kw,k0,k)
*
      data r3500 /1028.1063/
      data r4 /4.8314e-4/    ! referred to as C in millero and poisson 1981
      data dr350 /28.106331/
*
*
* convert pressure to bars and take square root of salinity.
*
      p = p0 / 10.
      sr = sqrt(abs(s)) 
*
* pure water density at atmospheric pressure
*   bigg p.h.,(1967) br. j. applied physics 8 pp 521-537.
*
      r1 = ((((6.536332e-9*t-1.120083e-6)*t+1.001685e-4)*t 
     &      -9.095290e-3)*t+6.793952e-2)*t-28.263737
*
* seawater density at atmospheric press. 
*    coefficients involving salinity:
*      r2 = A   in notation of millero and poisson 1981
*      r3 = B  
*
      r2 = (((5.3875e-9*t-8.2467e-7)*t+7.6438e-5)*t-4.0899e-3)*t
     &     +8.24493e-1 
*
      r3 = (-1.6546e-6*t+1.0227e-4)*t-5.72466e-3
*
*  international one-atmosphere equation of state of seawater
*
      sig = (r4*s + r3*sr + r2)*s + r1 
*
*  specific volume at atmospheric pressure
*
      v350p = 1.0 / r3500
      sva = -sig * v350p / (r3500+sig)
      sigma = sig + dr350
*
*  scale specific vol. anomaly to normally reported units
*
      svan = sva * 1.0e+8
      if (p.eq.0.) return
*
* ............................................................
c      high pressure equation of state for seawater
c        millero, et al , 1980 dsr 27a, pp 255-264
c               constant notation follows article
* ............................................................
* compute compression terms ...
*
      e = (9.1697e-10*t+2.0816e-8)*t-9.9348e-7
      bw = (5.2787e-8*t-6.12293e-6)*t+3.47718e-5
      b = bw + e * s
*             
      d = 1.91075e-4
      c = (-1.6078e-6*t-1.0981e-5)*t+2.2838e-3
      aw = ((-5.77905e-7*t+1.16092e-4)*t+1.43713e-3)*t 
     &      -0.1194975
      a = (d * sr + c) * s + aw 
*             
      b1 = (-5.3009e-4*t+1.6483e-2)*t+7.944e-2
      a1 = ((-6.1670e-5*t+1.09987e-2)*t-0.603459)*t+54.6746 
      kw = (((-5.155288e-5*t+1.360477e-2)*t-2.327105)*t 
     &      +148.4206)*t-1930.06
      k0 = (b1*sr + a1)*s + kw
*
* evaluate pressure polynomial 
* ............................................................
c   k equals the secant bulk modulus of seawater
c   dk = k(s,t,p) - k(35,0,p)
c   k35 = k(35,0,p)
* ............................................................
      dk = (b * p + a) * p + k0
      k35 = (5.03217e-5*p+3.359406)*p+21582.27
      gam = p / k35
      pk = 1.0 - gam
      sva = sva*pk + (v350p+sva)*p*dk/(k35*(k35+dk))
*
*  scale specific vol. anamoly to normally reported units...
*
      svan = sva*1.0e+8
      v350p = v350p * pk
* ....................................................................
* compute density anamoly with respect to 1000.0 kg/m**3
*  1) dr350: density anomaly at 35 (ipss-78), 0 deg. c and 0 decibars
*  2) dr35p: density anomaly 35 (ipss-78), 0 deg. c ,  pres. variation
*  3) dvan : density anomaly variations involving specfic vol. anamoly
* ....................................................................
      dr35p = gam / v350p
      dvan = sva / (v350p * (v350p+sva))
      sigma = dr350 + dr35p - dvan
*
      return  
      end   
* 
**************************************************************************
**************************************************************************
      real*8 function depth(p,lat)
*
*   depth in meters from pressure in decibars using
*   saunders and fofonoff's method.
*       deep-sea res., 1976,23,109-111.
*   formula refitted for 1980 equation of state
* 
*       pressure        p        decibars
*       latitude        lat      degrees
*       depth           depth    meters
*
c checkvalue: depth = 9712.653 m for p=10000 decibars, latitude=30 deg
c     above for standard ocean: t=0 deg. celsius ; s=35 (ipss-78)
* ....................................................................
*
      implicit real*8 (a-h,o-z)
      real*8 lat
*
      x = sin(lat/57.29578)
*
      x = x * x
*
* gr = gravity variation with latitude: anon (1970) bulletin geodesique
*
      gr = 9.780318*(1.0+(5.2788e-3+2.36e-5*x)*x) + 1.092e-6*p
      depth = (((-1.82e-15*p+2.279e-10)*p-2.2512e-5)*p+9.72659)*p
      depth = depth / gr
      return
      end
**************************************************************************
**************************************************************************
      real*8 function tf(s,p)
*
c   function to compute the freezing point of seawater
c   
c   reference: unesco tech. papers in the marine science no. 28. 1978
c   eighth report jpots
c   annex 6 freezing point of seawater f.j. millero pp.29-35.
c
c  units:
c         pressure      p          decibars
c         salinity      s          pss-78
c         temperature   tf         degrees celsius
c         freezing pt.
* ..................................................................
c  checkvalue: tf= -2.588567 deg. c for s=40.0, p=500. decibars 
      implicit real*8 (a-h,o-z)
      tf=(-.0575+1.710523e-3*sqrt(abs(s))-2.154996e-4*s)*s-7.53e-4*p
      return
      end
      real*8 function cpsw(s,t,p0)
* ..................................................................
c       pressure        p0       decibars
c       temperature     t        deg celsius (ipts-68)
c       salinity        s        (ipss-78)
c       specific heat   cpsw     j/(kg deg c)
* ..................................................................
c ref: millero et al,1973,jgr,78,4499-4507
c       millero et al, unesco report no. 38 1981 pp. 99-188.
c pressure variation from least squares polynomial
c developed by fofonoff 1980.
c check value: cpsw = 3849.500 j/(kg deg. c) for s = 40 (ipss-78),
c t = 40 deg c, p0= 10000 decibars
c   scale pressure to bars
      implicit real*8 (a-h,o-z)
      p=p0/10.
* ..................................................................
c sqrt salinity for fractional terms
      sr = sqrt(abs(s))
c specific heat cp0 for p=0 (millero et al ,unesco 1981)
      a = (-1.38385e-3*t+0.1072763)*t-7.643575
      b = (5.148e-5*t-4.07718e-3)*t+0.1770383
      c = (((2.093236e-5*t-2.654387e-3)*t+0.1412855)*t
     x    -3.720283)*t+4217.4
      cp0 = (b*sr + a)*s + c
c cp1 pressure and temperature terms for s = 0
      a = (((1.7168e-8*t+2.0357e-6)*t-3.13885e-4)*t+1.45747e-2)*t
     x   -0.49592
      b = (((2.2956e-11*t-4.0027e-9)*t+2.87533e-7)*t-1.08645e-5)*t
     x   +2.4931e-4
      c = ((6.136e-13*t-6.5637e-11)*t+2.6380e-9)*t-5.422e-8
      cp1 = ((c*p+b)*p+a)*p
c cp2 pressure and temperature terms for s > 0
      a = (((-2.9179e-10*t+2.5941e-8)*t+9.802e-7)*t-1.28315e-4)*t
     x   +4.9247e-3
      b = (3.122e-8*t-1.517e-6)*t-1.2331e-4
      a = (a+b*sr)*s
      b = ((1.8448e-11*t-2.3905e-9)*t+1.17054e-7)*t-2.9558e-6
      b = (b+9.971e-8*sr)*s
      c = (3.513e-13*t-1.7682e-11)*t+5.540e-10
      c = (c-1.4300e-12*t*sr)*s
      cp2 = ((c*p+b)*p+a)*p
c specific heat return
      cpsw = cp0 + cp1 + cp2
      return
      end
**************************************************************************
**************************************************************************
      real*8 function atg(s,t,p) 
*
c adiabatic temperature gradient deg c per decibar
c ref: bryden,h.,1973,deep-sea res.,20,401-408
c units:      
c       pressure        p        decibars
c       temperature     t        deg celsius (ipts-68)
c       salinity        s        (ipss-78)
c       adiabatic       atg      deg. c/decibar
c checkvalue: atg=3.255976e-4 c/dbar for s=40 (ipss-78),
c t=40 deg c,p0=10000 decibars
      implicit real*8 (a-h,o-z)
      ds = s - 35.0 
      atg = (((-2.1687e-16*t+1.8676e-14)*t-4.6206e-13)*p
     x+((2.7759e-12*t-1.1351e-10)*ds+((-5.4481e-14*t
     x+8.733e-12)*t-6.7795e-10)*t+1.8741e-8))*p 
     x+(-4.2393e-8*t+1.8932e-6)*ds
     x+((6.6228e-10*t-6.836e-8)*t+8.5258e-6)*t+3.5803e-5
      return
      end
**************************************************************************
**************************************************************************
      real*8 function theta(s,t0,p0,pr)
*
c to compute local potential temperature at pr
c using bryden 1973 polynomial for adiabatic lapse rate
c and runge-kutta 4-th order integration algorithm.
c ref: bryden,h.,1973,deep-sea res.,20,401-408
c fofonoff,n.,1977,deep-sea res.,24,489-491
*
c       p0      (in situ) pressure     [db]
c       t0      (in situ) temperature  [deg C]
c       s        salinity              [ipss-78]
c       pr       reference pressure    [db]
*  returns:
*       theta    potential temperature [deg C]
* 
c checkvalue: theta= 36.89073 c,s=40 (ipss-78),t0=40 deg c,
c p0=10000 decibars,pr=0 decibars
c             
      implicit real*8 (a-h,o-z)
*
      p = p0
      t = t0
*
      h = pr - p
      xk = h*atg(s,t,p) 
      t = t + 0.5*xk
      q = xk  
      p = p + 0.5*h 
      xk = h*atg(s,t,p) 
      t = t + 0.29289322*(xk-q) 
      q = 0.58578644*xk + 0.121320344*q 
      xk = h*atg(s,t,p) 
      t = t + 1.707106781*(xk-q)
      q = 3.414213562*xk - 4.121320344*q
      p = p + 0.5*h 
      xk = h*atg(s,t,p) 
      theta = t + (xk-2.0*q)/6.0
      return  
      end  
*
**************************************************************************
**************************************************************************
      real*8 function svel(s,t,p0)
*
c sound velocity in seawater:  chen and millero 1977,jasa,62,1129-1135
*
c       pressure        p0       decibars
c       temperature     t        deg celsius (ipts-68)
c       salinity        s        (ipss-78)
c       sound speed     svel     meters/second
*
c checkvalue: svel=1731.995 m/s, s=40 (ipss-78),t=40 deg c,p=10000 dbar
c
      implicit real*8 (a-h,o-z)
      equivalence (a0,b0,c0),(a1,b1,c1),(a2,c2),(a3,c3)
c
c   scale pressure to bars
      p=p0/10.
*
      sr = sqrt(abs(s))
c s**2 term
      d = 1.727e-3 - 7.9836e-6*p
c s**3/2 term
      b1 = 7.3637e-5 +1.7945e-7*t
      b0 = -1.922e-2 -4.42e-5*t
      b = b0 + b1*p
c s**1 term
      a3 = (-3.389e-13*t+6.649e-12)*t+1.100e-10
      a2 = ((7.988e-12*t-1.6002e-10)*t+9.1041e-9)*t-3.9064e-7
      a1 = (((-2.0122e-10*t+1.0507e-8)*t-6.4885e-8)*t-1.2580e-5)*t
     x     +9.4742e-5
      a0 = (((-3.21e-8*t+2.006e-6)*t+7.164e-5)*t-1.262e-2)*t
     x     +1.389
      a = ((a3*p+a2)*p+a1)*p+a0
c s**0 term
      c3 = (-2.3643e-12*t+3.8504e-10)*t-9.7729e-9
      c2 = (((1.0405e-12*t-2.5335e-10)*t+2.5974e-8)*t-1.7107e-6)*t
     x     +3.1260e-5
      c1 = (((-6.1185e-10*t+1.3621e-7)*t-8.1788e-6)*t+6.8982e-4)*t
     x     +0.153563
      c0 = ((((3.1464e-9*t-1.47800e-6)*t+3.3420e-4)*t-5.80852e-2)*t
     x     +5.03711)*t+1402.388
      c = ((c3*p+c2)*p+c1)*p+c0
c sound speed return
      svel = c + (a+b*sr+d*s)*s
*
      return
      end
*
**************************************************************************
**************************************************************************
      real*8 function bvfrq(s,t,p,n,pav,e)
*
*  brunt-vaisala frequency  (uses eos80)
*  after formulation of breck owen's & n.p. fofonoff
*
c       p      pressure     [dbars]
c       t      temperature  [deg C]
c       s      salinity     [ipss-78]
*       n      # of pts
*       pav    mean press over entire interval  [dbars] (returned)
c       e      n**2         [radians/sec]   (returned)
c       bvfrq  bouyancy freq  [cph]  (returned) 
* .............................................................
c checkvalue: bvfrq=14.57836 cph e=6.4739928e-4 rad/sec.
c            s(1)=35.0, t(1)=5.0, p(1)=1000.0
c            s(2)=35.0, t(2)=4.0, p(2)=1002.0
c  >>> note result centered at pav=1001.0 dbars <<<
*................................................................
      implicit real*8 (a-h,o-z)
*
      data rps2cph /572.9578/
*
      real*8 p(1),t(1),s(1),data(500)
*
c ! convert gravity to m/sec
      gr=9.80655
      gr2=gr*gr*1.e-4
*
c !  db to pascal conversion = 10^-4
c    get center pressure for interval
      tgrd=grady(t(1),p(1),n,pav,tav)
      do 35 k=1,n
      data(k)= svan(s(k),theta(s(k),t(k),p(k),pav),pav,sig)*1.0e-8
   35 continue
*
*  get v(35,0,pav)
      v350p = (1./(sig+1000.))-data(n)
*
c  compute potential density anomaly gradient
      dvdp=grady(data(1),p(1),n,pav,vbar)
      vbar=v350p+vbar
c  
      e   = -gr2*dvdp/(vbar)**2
      bvfrq = rps2cph * sign1(sqrt(abs(e)),e)
      return
      end
************************************************************************
************************************************************************
      real*8 function bvgrdts(p0,s,t,p,n,e)
c ************************************
c arguments:      
c   p0:    pressure level at which bvgrdts is evaluated (if p0 < 0, 
c          (p(1)+p(n))/2 is used and returned in this argument ). 
c    p:    observed pressures  ( decibars)
c    t:    observed temperatures   (deg celsius (ipts-68))
c    s:    observed salinities  (ipss-78)
c    n:    number of observations   (size of p,t,s arrays)
c    e:    stability parameter  (radians/second^2)
c   bvgrdts:    bouyancy frequency (cph)
c ************************************
c checkvalue: bvfrq=14.5xx cph e=6.47xxxe-4 rad/sec^2.
c            s(1)=35.0, t(1)=5.0, p(1)=1000.0
c            s(2)=35.0, t(2)=4.0, p(2)=1002.0
c     (note result centered at pav=1001.0 dbars)
c ************************************
c  Brunt-Vaisala frequency calculation with eos80 gradients of temp and salt.
c  The derivative quantities are computed in function eos8d() and returned in
c  the array drv.
c
c  see The Oceans - Sverdrup, Johnson & Fleming p. 417-418
c  also Hesselberg & Sverdrup (1915)  ref. page 430 The Oceans
c .........................................
      IMPLICIT REAL*8 (a-h,o-z)
      REAL*8 drv(3,8),s(1),t(1),p(1)
c
c  
      DATA rps2cph /572.9578/    ! 3600/2*pi changes rad/sec to cph
      DATA gp /9.80655/          ! Gravitational Constant in m/sec^2
c
c  ! 10^-4:pressure db to pascals (newton/m^2)
c
      gp2 = gp * gp * 1.e-4
c
c  compute gradients and values of t & s at pav ...
      tgrd = grady(t(1),p(1),n,pav,tav)
      sgrd = grady(s(1),p(1),n,pav,sav)
c
      if (p0 .ge. 0) then
         pav = p0
         tav = yinterp(p0, p(1), t(1), n)
         sav = yinterp(p0, p(1), s(1), n)
      end if

      z = eos8d(sav,tav,pav,drv)
c
c  derivatives of specific vol. with respect to temp and salt...
      gt = -drv(2,1)
      gs = -drv(3,8)
c 
      v = (1 / (1000.+ drv(1,3)))
      v2 = v * v
      e = (gp2/v2*(gt*(tgrd-atg(sav,tav,pav))+gs*sgrd))
      bvgrdts = rps2cph * SIGN1(SQRT(ABS(e)),e)
      p0 = pav
      return
      end

************************************************************************
************************************************************************
      function grady(y,p,nobs,pav,ybar)
*
c  Returns least squares slope 'grady' of y versus p.
c  The gradient is representive of the interval centered at pav;
*  ybar is the arithmetic average of y values over the entire interval.
*
      implicit real*8 (a-h,o-z)
*
      real*8 p(1),y(1)
*
      grady = 0.0
      a0 = 0.0
      cxx = 0.0
      cx = 0.0
      cxy = 0.0
      cy = 0.0
      if(nobs.le.1) go to 30
*
      do 20 k=1,nobs
  20     cx =cx+p(k)
*
      pav = cx / nobs
      do 35 k=1,nobs
         cxy=cxy+y(k)*(p(k)-pav)
         cy =cy+y(k)
         cxx=cxx+(p(k)-pav)**2
   35 continue

      if(cxx.eq.0.0) return
      a0 = cxy / cxx
      ybar = cy / nobs
   30 continue
      grady = a0
*
      return
      end
*
****************************************************************************
****************************************************************************
      real*8 function p80(dpth,xlat)
c
c pressure from depth from saunder's formula with eos80.
c reference: saunders,peter m., practical conversion of pressure
c            to depth., j.p.o. , april 1981.
c r millard
c march 9, 1983
c check value: p80 = 7500.004 dbars
*              for lat = 30 deg, depth = 7321.45 meters
* .....................................................................
      implicit real*8 (a-h,o-z)
      parameter  (pi = 3.141592654)
      plat = abs(xlat*pi/180.)
      d = sin(plat)
      c1 = 5.92e-3 + d**2 * 5.25e-3
      p80 = ((1-c1)-sqrt(((1-c1)**2)-(8.84e-6*dpth)))/4.42e-6
      return
      end
****************************************************************************
****************************************************************************
      real*8 function gravity(xlat)
*
*  acceleration due to gravity in cm/sec^2 as a function of latitude.
*
      implicit real*8 (a-h,o-z)
*
      parameter (pi = 3.141592654)
*
      plat = abs(xlat*pi/180.)
      gravity = 978.0318 *(1.0+5.3024e-3 * sin(plat)**2 
     &         - 5.9e-6 * (sin(2.*plat))**2)
      return
      end
****************************************************************************
****************************************************************************
      real*8 function coriol(xlat)
*
*  Coriolus parameter as a function of latitude in m/sec^2.
*
      implicit real*8 (a-h,o-z)
*
      parameter  (pi = 3.141592654)
*
*
      plat = abs(xlat*pi/180.)
      coriol = 14.5842e-5 * sin(plat)
      return
      end

*************************************************************
*************************************************************
      REAL*8 FUNCTION yinterp(xval, x, y, npts)
*
*   linearly interpolates to find the position of xval in array x, and returns
*   the corresponding value in the array, y.  If xval does not appear in array
*   x, the value of "flag" is returned.  This routine assumes that the x array
*   is monotonic and continuous (no missing values); and assumes the y array
*   is continuous.
*
      REAL*8 x(npts), y(npts), xval, flag
*
      DATA  flag / -99999./
*
************************************************************
*
      do k = 1, npts-1
         v1 = xval - x(k)
         v2 = xval - x(k+1)
*
         if (v1 .lt. 0. .AND. v2 .lt. 0.)then
            continue
         else if (v1 .gt. 0. .AND. v2 .gt. 0.) then
            continue
         else if (v1 .eq. v2 .AND. v1 .ne. 0.) then
            continue
         else if (v1 .eq. 0. .AND. v2 .eq. 0.) then       
            yinterp = y(k)
            return
         else
            yinterp = (y(k) + (y(k+1) - y(k)) * v1 / (x(k+1) - x(k)) )
            return
         end if
*
      end do
*
*  if execution falls through to this point, xval was not found in x array;
*     return a flag value ....
*
      yinterp = flag
      return
*
      END
c
c
c
      real function sign1(a,b)
c
      real a,b
c
      if(b.ge.0)then
         sign1=abs(a)
      else
         sign1=-abs(a)
      endif
c
      return
      end
