/* preblank.c
_______________________________________________________________________________

   USAGE:   -I<input_filename> -O<output_filename>  -P<property>   
            -R<search_radius>
_______________________________________________________________________________
.  Reads a file created by gridsurf.
.
.  Applies an algorithm to a matrix of points to determine which 
.  grid nodes will contain no data after smoothing the specified property.  
.  The purpose is to create
.  a blanking file representing coastlines, outcropping bathymetry, etc.
.  for use in smoothing and plotting.  The output file contains a list of
.  lat/lon pairs corresponding to gridnodes which contain no data.
.
.  The smoothing algorithm prevents 
.  oversmoothing at the edges by evaluating the distribution of points
.  in each layer (defined by the search-radius) surrounding a gridnode.    
.  There must be at least one point on the left AND right side of the layer in
.  order to add both the left and right sides to the gridnode; the same applies
.  to the top and bottom sides.  
.
.  The search radius specified should correspond to the search radius which will
.  be used to smooth the gridded dataset later. 
_______________________________________________________________________________
*/
#include <stdio.h>
#include <string.h>
#include "hydrobase.h"


#define IMASK  0x01     /* masks for option flags */
#define OMASK  0x02
#define PMASK  0x04
#define RMASK  0x08
#define ALL    0x0f     /* bitwise OR of all above masks */

extern int    get_prop_indx(); 
extern char  *get_prop_mne();
extern void   print_prop_menu();


void main(argc, argv)
int   argc;
char *argv[];
{
   int    row, col, n, i, j, sq;
   int    gridsize;
   int    rad, maxrad, minobs;
   int    ncols, nrows;
   int    error, index, nprops, prop_avail[MAXPROP];
   int    *np, *np1;      
   short  optflag = 0;
   float  lat, lon;
   float  mean, var;
   float  xmin, xmax, ymin, ymax, xspacing, yspacing;
   FILE  *infile, *outfile;
   char  *fname;
   char  *s, id[3];
   void  print_usage();
   void  add_wghted_layer();

/* check for command line arguments */

   if (argc < 2) {
     print_usage(argv[0]);
     exit(1);
   }

   minobs = 1;

/*  parse the command line arguments */

   for (i = 1; i < argc; i++) {
      error =  (argv[i][0] == '-') ? 0 : 1;
      if (!error) {
            switch (argv[i][1]) {
               case 'I':                   /* get input file */
                        optflag = optflag | IMASK;
                        fname = &argv[i][2];
                        if ((infile = fopen(fname, "r")) == NULL) {
                          fprintf(stderr, "\n Unable to open %s for input\n\n",fname);
                          exit(1);
                        }
                        fprintf(stderr, "\nOpened %s ... \n", fname);
                        break;
               case 'O':                    /* get output file name */
                        optflag = optflag | OMASK;
                        fname = &argv[i][2];
                        if ((outfile = fopen(fname,"w")) == NULL) {
                          fprintf(stderr, "\nUnable to open %s for output \n\n", fname);
                          exit(1);
                        }
                        break;

               case 'R':
                       optflag = optflag | RMASK;
                       error = (sscanf(&argv[i][2],"%d", &maxrad) == 1) ? 0 : 1;
                       break;

               case 'P':
                       optflag = optflag | PMASK;
                        s = &argv[i][2];
                        if (*s == '\0') {
                               print_prop_menu();
                               exit(0);
                        }
                         if (*s == '/')
                               ++s;
                         sscanf(s,"%2s", id);
                         index = get_prop_indx(id);
                         error = (index < 0) ? 1 : 0;
                         break;

               default:
                       error = 1;

          }    /* end switch */
       }  /* end if */


       if (error ) {
             fprintf(stderr,"\nError parsing command line args.\n");
             fprintf(stderr,"     in particular: '%s'\n", argv[i]);
             print_usage(argv[0]);
             exit(1);
       }


   }  /* end for */

   if (optflag != ALL) {
       print_usage(argv[0]);
       fprintf(stderr,"\nYou must specify all options!\n");
       exit(1);
   }

/*   get grid information from input file */

   if (fscanf(infile,"%d%d%f%f%f%f%f%f%d", &nrows, &ncols, &xspacing, &yspacing, &xmin,
      &ymin, &xmax, &ymax, &nprops )  != 9) {
        fprintf(stderr,"\nError reading heading from infile\n\n");
        exit(1);
   }
   fprintf(stderr, "\nInput grid is %d rows by %d cols", nrows, ncols);

   fprintf(stderr,"\n Properties available: ");
   for (i = 0; i < nprops; ++i) {
       fscanf(infile,"%2s", id);
       prop_avail[i] = get_prop_indx(id);
       if (index == prop_avail[i]){
           index = i + 400;
       }
       fprintf(stderr," %2s", id);
   }
   fprintf(stderr,"\n");
   if ((index = index - 400) < 0)  {
      fprintf(stderr,"\nRequested property: %2s not available.", get_prop_mne(i+400));
      exit(1);
   }

/* allocate space for matrices and initialize all values */

    gridsize = nrows * ncols;

    np = (int *)malloc(gridsize * sizeof(int));
    np1 = (int *) malloc(gridsize * sizeof(int));

    for (i = 0; i < gridsize; ++i) {
        np[i] = 0;
        np1[i] = 0;
    }
 
/*  read data from infile and store # of obs in matrix by row, col */ 
 
   fprintf(stderr,"\nReading input file...\n");

   while ( fscanf(infile, "%f%f",  &lat, &lon) != EOF) {

     row = (int) (.0001 + (lat - ymin) / yspacing);
     col = (int) (.0001 + (lon - xmin) / xspacing);

      for (i = 0; i < nprops; ++i) {
        if (fscanf( infile,"%f%f%d", &mean, &var, &n) != 3) {
           fprintf(stderr,"\nError reading input file at ");
           fprintf(stderr,"values of (row, col) = ( %d , %d)\n", row, col);
        }
        if (i == index) {
           np[row * ncols + col] = n;
        }
      } /* end for */

   }  /* end while */

/*  for each pt in the new grid, sum the values from surrounding squares
    in old grid, weighted by distance. Move successively outward until 
    either minobs is achieved or max search radius is reached.  */

   fprintf(stderr,"\nComputing new matrix ...\n");
    sq = -1;
   for ( row = 0; row < nrows; ++row) {
      fprintf(stderr,"*");
      for (col = 0; col < ncols; ++col) {
         ++sq;
         rad = 0;    /* initialize radius to 0  for each pt */
         np1[sq] = np[sq];
         while ((np1[sq] < minobs) && (rad < maxrad)) {
              ++rad;
              add_wghted_layer(rad, row, col, nrows, ncols, np, &np1[sq]);
         }

     
         if ( np1[sq] == 0) {       /* list in blanking file */
             lat = ymin + (row + .5) * yspacing;
             lon = xmin + (col + .5) * xspacing;
             fprintf(outfile,"\n%.3f %.3f", lon, lat);
         }
      }  /* end for */
   }  /* end for */

   fprintf(stderr,"\nEnd of preblank.\n");
   exit(0);
} /* end main */
/****************************************************************************/

void print_usage(program)
char *program;
{
   fprintf(stderr,"\nUsage:  %s -I<input_filename> -O<output_filename> -R<search_radius> -P<property_mnemonic>", program);
   fprintf(stderr,"\n\n    -I  : specifies input file_name");
   fprintf(stderr,"\n    -O  : specifies filename for output ");
   fprintf(stderr,"\n    -R  : maximum search radius (# of squares away) to go");
   fprintf(stderr,"\n          to find minobs to compute each gridpt: ");
   fprintf(stderr,"\n            ex: -R3 ");
   fprintf(stderr,"\n    -P  : specifies property to examine");
   fprintf(stderr,"\n     ex: -Ppr");
   fprintf(stderr,"\n         -P (by itself) will print a menu of available properties");

   fprintf(stderr,"\n\n");  
   return;
}
/****************************************************************************/

void add_wghted_layer(radius, i, j, nrows, ncols, old, countptr)
int radius;         /* search radius  (in # of gridpts )*/
int i, j;           /* indices of gridpt  being computed in new matrix  */
int nrows, ncols;   /* dimensions of old matrix */
int *old;           /* ptr to old  matrix */
int *countptr;      /* ptr to counter for the new matrix at this point */
{
   int top, bot, left, right;
   int row, col, layer, side;
   int np_w, np_e, np_n, np_s;
   int sq;


   if (radius <= 0) 
         return ;


/* initialize these values ... */

   np_w = np_e = np_s = np_n = 0;

/* define row,col of lower left corner of first layer */

   bot = i ;   
   left = j ;
   layer = 1;

/* move to lower left corner of layer corresponding to radius */

   while (layer++ < radius) {   
     --bot;
     --left;
   }  /* end while */

/* side of square defined by locus of pts in layer is 2 * radius in length */

   side = 2 * radius + 1; 
 
/* define corners of square */

   top = bot + side - 1;
   right = left + side - 1;

/* check left/right for data symmetry */
/* move up left side of square */

   row = bot;
   col = left;

   if ((col >= 0) && (col < ncols)) {     /* within column bounds of matrix? */
      while ( row < top) {
         if ((row >= 0) && (row < nrows)) {      /* within row bounds? */
           sq = row * ncols + col;
           if (old[sq] > 0) {             /* any obs at this pt? */
             np_w += old[sq];
           }
         }
         ++row;
      } /* end while */
   }  /* end if */

/* if the left edge has data, check the right edge ...*/

   if (np_w) {
      row = top;
      col = right;

/*    move down right side of square */

     if ((col >= 0) && (col < ncols)) {     /* within col bounds? */
         while ( row > bot) {
            if ((row >= 0) && (row < nrows)) {      /* within row bounds? */
              sq = row * ncols + col;
              if (old[sq] > 0) {             /* any obs at this pt? */
                np_e += old[sq];
              }
            }
            --row;
         } /* end while */
      }  /* end if */

/*    if both left/right have data : add them to new grid ... */  

      if (np_e) {
             *countptr += np_e + np_w;
      }  /* end if */

   }  /* end if np_w */

/* check the north/south edges for data symmetry */

   row = top;
   col = left;

/*    move across top of square */

   if ((row >= 0) && (row < nrows)) {     /* within row bounds? */
      while ( col < right) {
         if ((col >= 0) && (col < ncols)) {      /* within column bounds? */
           sq = row * ncols + col;
           if (old[sq] > 0) {             /* any obs at this pt? */
                np_n += old[sq];
           }
         }
         ++col;
      } /* end while */
   }  /* end if */


/* if north edge has data, check south edge ... */

   if (np_n) {
      row = bot;
      col = right;

/*    move across bottom of square */

      if ((row >= 0) && (row < nrows)) {     /* within row bounds? */
         while ( col > left) {
           if ((col >= 0) && (col < ncols)) {      /* within col bounds? */
              sq = row * ncols + col;
              if (old[sq] > 0) {             /* any obs at this pt? */
                np_s += old[sq];
              }
           }
           --col;
        } /* end while */
      }  /* end if */

/*   if south edge has data, add buffers to global arrays */

      if (np_s) {
             *countptr += np_s + np_n;
      }

   } /* end if np_n */

   return;

}  /* end add_wghted_layer() */

