/*  timeprop.c

____________________________________________________________________________
  USAGE:  

 timeprop filename(_roots) -P<property_list> [-B/west/east/south/north] [-S<surface_def_file>]  [-D<dirname>] [-E<file_extent>]

  list of filenames MUST be first argument!

 -P : list of properties to evaluate; ex: -Ppr/te/th/sa/ox
      -P (by itself) will print a list of the available properties;

 [-B] : specifies geographic bounds

 [-S] : file containing surface definitions. If this is not specified
      these values are expected to come from the standard input device.

 [-D] : specifies directory for input files (default is current dir)  
      ex: -D/home/pete/data

 [-E] : specifies input_file extent (default is no extent) ex: -E.sum

____________________________________________________________________________
timeprop projects hydrographic properties onto one or more surfaces
and outputs the values of those properties together with year, month,
lat, and lon for each station found.

The surfaces may be defined by some value of any property supported
by HydroBase (depth, pressure, temperature, density, etc.)
The properties at each station are linearly interpolated onto the
surface;

For each surface, an output file is generated containing :
     
     
     n, list_of_properties                           (1st line)
     year month lat lon   p1 ... pn  ( 1 line for each station)
                                                      
____________________________________________________________________________
*/


#include <stdio.h>
#include <math.h>
#include <fcntl.h>
#include <string.h>
#include <malloc.h>
#include "hydrobase.h"
#include "hydro_cdf.h"


#define    ABS(x)       (((x) < 0) ? -(x) : (x))           
#define    NINT(x)      ((int)((x) + 0.5))

/* input file pathnames */

#define    EXTENT   ""
#define    DIR      ""



/* set up data structure to store info about each surface */

struct surface {
          double  value;    /* value of property on this surface */
          double  pref;
            int   data_ind;   /* index ID of property */
            char  density_flag ;  /* set if this is a density surface */
            FILE *fptr;    /* output file */
  struct surface *next;    
};
    /* flags to indicate which properties are to be output and which are
       needed for computation */
      
short prop_req[MAXPROP];         /* set of props requested for output */
short prop_needed[MAXPROP];      /* set of props requested || defining a
                                    surface || needed for computation */

   /* global variables to store station ... */

struct HYDRO_DATA station;
struct HYDRO_HDR hdr;
double s_pref;           /* ref lev for computing a non-standard sigma level */

     /* boundaries  */

float   xmin, xmax, ymin, ymax;     
int  xdateline;


     /* in prop_subs.c */

extern int   get_prop_indx();               
extern char *get_prop_descrip(), *get_prop_mne();
extern void  print_prop_menu();
extern void  compute_sigma(), compute_height(); 
extern void  compute_theta();
extern double buoyancy(), potvort();

     /* in hydro_utils.c */

extern int open_hydro_file();           
extern int get_station();


main (argc, argv)
int   argc;
char *argv[];
{
   short   bopt, popt;
   int     index, nprops = 0;
   int     i;
   int     curfile = 1, nfiles = 0; 
   int     print_msg = 1;       /* print message in file open routines*/
   int     error, prompt;
   FILE    *def_file;
   char    *extent, *dir, *st;
   int     infile;
   struct surface *list = NULL, *surf;
   struct surface *add_surf();
   void    print_usage();
   void    get_hydro_data();
   void    outputdata();
   int     parse_prop_list();


/* are there command line arguments? */

   if (argc < 2) {
      print_usage(argv[0]);
      exit(1);
   }
 
/*  set these default values */

    dir = DIR;
    extent = EXTENT;
    def_file = (FILE *) stdin;
    prompt = 1;            /* for surface definitions */
    bopt  = popt = 0;
    error = 0;
    xmin = -360;
    xmax = 360;
    ymin = -90;
    ymax = 90;
    s_pref = -1;
    xdateline = 0;


/* initialize these ... */

   for (i = 0; i < MAXPROP; ++i) {
       prop_req[i] = 0;
       prop_needed[i] = 0;
   }



/*  parse the command line arguments */

   for (i = 1; i < argc; i++) {
      if (argv[i][0] == '-') {
            switch (argv[i][1]) {
               case 'D':                   /* get input dir */
                        dir = &argv[i][2];
                        break;
               case 'E':                    /* get file extent */
                        extent = &argv[i][2];
                        break;

               case 'B':                    /* get grid bounds */
                        bopt = 1;
                        st = &argv[i][2];
                           if (*st == '/')
                               ++st;
                        error = (sscanf(st,"%f", &xmin) != 1);
                        while (*(st++) != '/')
                           ;  
                        error += (sscanf(st,"%f", &xmax) != 1);
                        while (*(st++) != '/')
                           ;  
                        error += (sscanf(st,"%f", &ymin) != 1);
                        while (*(st++) != '/')
                           ;  
                        error += (sscanf(st,"%f", &ymax) != 1);
                        
                        if (xmin > 0 && xmax < 0)
                           xmax += 360;
                           
                        if (xmax > 180)
                           xdateline = 1;
                        break;


               case 'S':
                        def_file = fopen(&argv[i][2],"r");
                        prompt = 0;              /* turn off prompt flag */
                        if (def_file == NULL) {
                           fprintf(stderr,"\nError opening %s.\n",&argv[i][2]);
                           exit(1);
                        }
                        break;

               case 'P':
                        popt = 1;
                        if ( argv[i][2] == '\0') {
                               print_prop_menu();
                               exit(0);
                        }
                        nprops = parse_prop_list(&argv[i][2]);
                        if (nprops <= 0)
                             error = 1;
                        break;

               default:
                        error = 1;

          }    /* end switch */

          if (error ) {
             fprintf(stderr,"\nError parsing command line args.\n");
             fprintf(stderr,"     in particular: '%s'\n", argv[i]);
             exit(1);
          }

       }  /* end if */

       else  {
           ++nfiles;
       }
   }  /* end for */

   if ( !nfiles || !popt) {
       fprintf(stderr,"\nYou must specify input file(s) and properties!\n");
       exit(1);
   }



   fprintf(stderr,"\n\n  bounds: %.3f %.3f %.3f %.3f\n", ymin, ymax, xmin, xmax );

/* get info on each surface,  add surface to the linked list, 
    allocate space for computation, write heading to each output file ... */

   while ( (surf = add_surf(def_file,prompt))  != NULL) {
      surf->next = list;
      list = surf;
   }

   fclose(def_file);


/* loop for each input file */

   do {


       infile = open_hydro_file(dir, argv[curfile], extent, print_msg);

       if (infile >= 0)
          get_hydro_data(infile, list);

       close(infile);

   } while (curfile++ < nfiles );

   fprintf(stderr,"\n\nEnd of timeprop.\n");
   exit(0);

}   /* end main */

/****************************************************************************/

void print_usage(program)
char *program;
{
   fprintf(stderr,"\nUsage:  %s filename_root(s) -P<list_of_properties> [-B/west/east/south/north] [-S<surface_def_file>] [-D<dirname>] [-E<file_extent>]", program);

   fprintf(stderr,"\n\n  List of filenames MUST be first argument!");
   fprintf(stderr,"\n    -P  : list of properties to project onto surface;");
   fprintf(stderr,"\n          ex:  -Ppr/th/sa/ox/ht");
   fprintf(stderr,"\n               -P (by itself) produces a list of available properties");
   fprintf(stderr,"\n   [-B] : specifies grid bounds");
   fprintf(stderr,"\n   [-S] : file containing surface definitions.");
   fprintf(stderr,"\n          If this is not specified these values are expected");
   fprintf(stderr,"\n          to come from the standard input device.");
   fprintf(stderr,"\n   [-D] : specifies dirname for input files (default is current directory) ");
   fprintf(stderr,"\n            ex: -D../data/ ");
   fprintf(stderr,"\n   [-E] : specifies input_file extent (default is no extent)");  
   fprintf(stderr,"\n            ex: -E.dat ");

   fprintf(stderr,"\n\n");  
   return;
}

/*****************************************************************************/
int parse_prop_list(st)
char *st;   /* the list with -P stripped off */
    /*  Parses a list of property mnemonics and sets global flags accordingly.
        Returns the number of properties.  An error will cause an exit. */
{
   int index, nprops;
   char prop[4];
   double ref_val;

   nprops = 0;

   do {
      if (*st == '/')
         ++st;
      sscanf(st,"%2s", prop);
      index = get_prop_indx(prop);
      if (index < 0) {
         fprintf(stderr,"\n Unknown property requested: %s\n", prop);
         exit(1);
      }
      prop_req[index] = 1;
      prop_needed[index] = 1;
      ++nprops;
      ++st;
      ++st;

  /* some special cases ... */

   /* s_ */
      if ((enum property)index == S_ ) {

         if (sscanf(st, "%lf", &ref_val) != 1) {
             fprintf(stderr,"\n Specify a ref pressure when requesting the property 's?'");
             fprintf(stderr,"\n   ex: -Ps_1500/th/sa\n");
             exit(1);
         }
         while (!(*st=='/' || *st=='\0' || *st==' '))
            ++st;

         if (s_pref >= 0) {   /* has this already been requested? */
                     if ( NINT( s_pref) != NINT(ref_val)) {
                         fprintf(stderr,"Sorry. You have requested the property s_ with multiple reference levels.\n");
                         fprintf(stderr,"You can only use one of those prefs");
                         fprintf(stderr,"  You will probably have to do multiple runs for each different pref you want to associate with s_\n\n");
                        exit(1);
                     }
          }
          s_pref = ref_val;
      }

   /* sf... stream function gets computed in an external
      program:  just output pr, sv, and ht for now.
   */

      if ((enum property)index == SF ) {
                         
          prop_req[index] = 0;     /* reset these */
          prop_needed[index] = 0;
          fprintf(stderr,"\nStream Function is not supported in this module.\n");

      }
   /* end of special cases */

  } while (*st == '/');

  return (nprops);
}  /* end parse_prop_list() */
/*****************************************************************************/
struct surface *add_surf(infile, prompt)
FILE *infile;
int  prompt;
/*  -  Allocates storage space for a record of type <struct surface>;
    -  queries the user for information about the surface (via stdin);
    -  sets the fields of the struct accordingly;
    -  opens the output file;
    -  returns a ptr to the struct  or NULL if no struct was added.
*/

{
   static n = 0;
   char id[4];
   int  index, get_prop_indx();
   struct surface *surf;
   char    fname[80];   


   if (!n && prompt) {
   fprintf(stderr,"\nDefine each projection surface ...\n");

   fprintf(stderr,"\n    -----  Surface property options  -------\n");
   print_prop_menu();
   fprintf(stderr,"\nbot:  (bottom observation)");
   fprintf(stderr,"\n\nend:  (end list)");
   fprintf(stderr,"\n    ----------------------------------------"); 

   }

   if (prompt)
         fprintf(stderr,"\nChoose a property type for surface #%d: ", ++n);
   if ( fscanf(infile, "%s", id) != 1) {
         fprintf(stderr,"\nError reading from surface definition file\n");
         exit(1);
   }
 
   if (strncmp(id, "end", 3) == 0) 
               return(NULL);          /* exit function */

   if (strncmp(id, "bot", 3) == 0) {
         surf = (struct surface *) malloc(sizeof(struct surface));
         surf->data_ind = -1;
         if (prompt) 
            fprintf(stderr,"Enter path/name of outfile for this surface: ");

         if (fscanf(infile, "%s", fname) != 1) {
            fprintf(stderr,"\nError reading surf.filename from surface definition file\n");
            exit(1);
         }
   
         if ((surf->fptr = fopen(fname, "w")) == NULL) {
             fprintf(stderr,"\nError opening %s for output.");
             exit(1);
         }

         return (surf);
   }

      if ((index = get_prop_indx(id)) < 0) {       /* is choice appropriate? */
          fprintf(stderr,"\n%s is not an option!\n\n", id);
          exit(1);    
      }  /* end if */ 

      if (index == (int) SF) {
          fprintf(stderr,"\n%s is not an appropriate option!\n\n", id);
          exit(1);    

      }

       surf = (struct surface *) malloc(sizeof(struct surface));

       surf->data_ind = index;


  /*  and reference pressure, if appropriate... */

      if ( (index== (int)S_ ) || (index == (int)VA) ){
         if (prompt) {
            fprintf(stderr,"enter reference Pr:  ");
         }
         if (fscanf(infile, "%lf", &surf->pref) != 1) {
           fprintf(stderr,"\nError reading pref for property %s from surface definition file\n", id);
           exit(1);
         }
      }
  /* get value of surface ... */

      if (prompt) {
         fprintf(stderr,"enter %s value", get_prop_mne(index));
         fprintf(stderr,": ");
      }
      if (fscanf(infile, "%lf", &surf->value) != 1) {
        fprintf(stderr,"\nError reading surf.value from surface definition file\n");
        exit(1);
      }

       
/* !**!  Special cases for individual properties... */

       switch ((enum property) index)  {
          case S_:
                 surf->density_flag = 1;
                 if (s_pref >= 0) {   /* has this already been requested? */
                     if ( NINT( s_pref) != NINT(surf->pref)) {
                         fprintf(stderr,"Sorry. You have requested the property s_ with multiple reference levels.\n");
                         fprintf(stderr,"You can only use one of those prefs");
                         fprintf(stderr,"  You will probably have to do multiple runs for each different pref you want to associate with s_\n\n");
                        exit(1);
                     }
                 }
                 s_pref = surf->pref;
                 prop_needed[index] = 1;     
                 break;
          case S0: 
                 surf->pref = 0.;
                 surf->density_flag = 1;
                 prop_needed[index] = 1;     
                 break;
          case S1: 
                 surf->pref = 1000.;
                 surf->density_flag = 1;
                 prop_needed[index] = 1;     
                 break;
          case S2: 
                 surf->pref = 2000.;
                 surf->density_flag = 1;
                 prop_needed[index] = 1;     
                 break;
          case S3: 
                 surf->pref = 3000;      
                 surf->density_flag = 1;
                 prop_needed[index] = 1;     
                 break;
          case S4: 
                 surf->pref = 4000.;
                 surf->density_flag = 1;
                 prop_needed[index] = 1;     
                 break;

          case SF:
                 fprintf(stderr,"Stream Function is not an option for a projection surface.\n");
                 free (surf);
                 return(NULL);
          default:
                 surf->pref = 0.;       
                 surf->density_flag = 0;
                 prop_needed[index] = 1; 
                 break;   

       }  /* end switch */



/* open output file */

   if (prompt) 
      fprintf(stderr,"Enter path/name of outfile for this surface: ");

   if (fscanf(infile, "%s", fname) != 1) {
         fprintf(stderr,"\nError reading surf.filename from surface definition file\n");
         exit(1);
   }
   
   if ((surf->fptr = fopen(fname, "w")) == NULL) {
      fprintf(stderr,"\nError opening %s for output.");
      exit(1);
   }

   return (surf);

}  /* end add_surf() */
/*****************************************************************************/

void get_hydro_data(file, listptr)
int file;
struct surface *listptr;
   /*  Reads each station in a HydroBase file and adds property values
       to the appropriate surfaces.   This module requires that the HydroBase
       file contains a minimum of pr, de, te, sa observations.  */
{
   int error, i;
   double dlat;
   short derived[MAXPROP];
   void  insertdata();
   struct surface  *surf;
   void    free_and_alloc();

/* initialize this so all derived properties get computed in insertdata() */

   for (i = 0; i < MAXPROP; ++i) 
            derived[i] = 0;


/* read each station in file ... */

    while ((error = get_station(file, &hdr, &station)) == 0) {

  /* check for and skip over out of bounds stations   */
  
       if (xdateline && hdr.lon < 0)
          hdr.lon += 360;

       if ((hdr.lat > ymax) || (hdr.lat < ymin) 
        || (hdr.lon < xmin) || (hdr.lon > xmax))  
             continue;

/* ensure that pr, de, te, and sa are available ... */

       if (!(available(PR, hdr) && available(DE, hdr) && available(TE, hdr) 
              && available(SA, hdr))) {
            fprintf(stderr,"\n>>>>> WARNING!!!  ");
            fprintf(stderr,"Station does not include pr, de, te, and sa.");
            break;
       }


 /* compute appropriate properties at each level in station ... derivative 
    properties (pot.vorticity, buoyancy...) get computed later in 
    insertdata(). */

/* !**! Special cases for individual properties... */

    for (i = 0; i < MAXPROP; ++i) {
       if (prop_needed[i]) {
          switch ((enum property) i) {
             case TH:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_theta(hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA] );
               break;
             case S0:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sigma(0., hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;
             case S1:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sigma(1000., hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;
             case S2:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sigma(2000., hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;
             case S3:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sigma(3000., hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;
             case S4:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sigma(4000., hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;
             case S_:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sigma(s_pref, hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;
             case HT:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_height(hdr.nobs, station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA], station.observ[i]);
               break;
             case SV:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sp_vol( hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;

             case VA:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_svan( hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;

             case PV:   /* fall through */
             case BF:
                  prop_needed[(int)PR] = 1;
                  break;

             default:
               break;
          } /* end switch */
       }
    }

      /* traverse the linked list of surfaces & interpolate data onto each 
         surface */

         surf = listptr;
         while (surf != NULL) {
            outputdata(surf, derived);
            surf = surf->next;
         }  /* end while */


   }  /* end while */

   if (error > 0)
      report_status(error, stderr);

   return;

} /* end get_hydro_data() */

/****************************************************************************/

void free_and_alloc(dptr, n)
double **dptr;   
int n;           /* number of elements to allocate */

   /* Frees up the space pointed to by dptr.  It MUST have been allocated
      using malloc()!!  Then mallocs space for n double values and sets
      dptr pointing to it.  If insufficient memory, a message is written to
      stderr and an exit() is made. */
{

   if (*dptr != NULL) {
      free((char *)*dptr);
      *dptr = NULL;
   }

   *dptr = (double *) malloc(n * sizeof(double));

   if (*dptr == NULL) {
      fprintf(stderr, "\nInsufficient memory for call to malloc()\n");
      exit(1);
   }

   return;
}  /* end free_and_alloc() */
/****************************************************************************/

void outputdata(sptr, already_deriv)
struct surface *sptr;  /* defines projection surface */
short *already_deriv;  /* flags derivative props which are already computed */

   /*   Projects property values onto a surface and writes values to output 
        file.   The station data arrays are global. */
{
   int i, datagap;
   char field_string[20];
   double p0, d0, val, dlat, *d;
   double interpolate();
   double project_deriv_prop();
   double project_prop();

   if (sptr->data_ind == -1) {     /* return bottom observations */
        fprintf(sptr->fptr,"%5d %2d %8.3f %8.3f ", hdr.year, hdr.month, hdr.lat,
                hdr.lon);

        for (i = 0; i < MAXPROP; ++i ) {

          if (prop_req[i]) {   /* was this property requested? */

/* !**! Special cases for individual properties... */

             switch ((enum property) i) {

               case BF:    /* derivative properties */
               case PV:
                    if (already_deriv[i]) 
                       val = station.observ[i][hdr.nobs-1];
                    else
                       val = project_deriv_prop(i, station.observ[(int)PR][hdr.nobs-1], hdr.nobs, hdr.lat);
                    break;

               case OX:    /* optional observed properties */
               case N2:
               case N3:
               case P4:
               case SI:
                    if (!available( (enum property)i, hdr)) {
                       val = -9999.;
                       break;
                    }
                    /* if available fall through to default*/

               default:    /* all other properties */ 
                    val = station.observ[i][hdr.nobs-1];
                    break;
             } /* end switch */

              sprintf(field_string," %c%d.%dlf", '%', get_field_width(i), get_field_precis(i));
             fprintf(sptr->fptr, field_string, val);
          } 
        }  /* end for */
        fprintf(sptr->fptr, "\n");
        return;
   }

/* surface is not the bottom... */

   d = station.observ[(int)DE];  /* set this ptr for easier referencing */

 /* determine if the surface exists at this station
    and check for vertical datagaps.  An unacceptable datagap is defined
    as :       > 200 m for surfaces in the upper 1000 m
               > 600 m for all other surfaces   */

   if ((d0 = interpolate(sptr->value, station.observ[sptr->data_ind], d, hdr.nobs)) > -9998.) {

     if (prop_needed[(int)PR])    /* pr is required for derivative props */
        p0 = interpolate(sptr->value, station.observ[sptr->data_ind], d, hdr.nobs);

       /* after this loop, d[i-1] and d[i] will bracket the de value
          associated with the projection surface ... */
     i = 0;
     while (d[++i] < d0 ) 
            ;

     if ((d[i-1] == d0) || (d[i] == d0) )
          datagap = 0;
     else if (d0 < 1001)
          datagap = (d[i] - d[i-1]) > 200;
     else
          datagap = (d[i] - d[i-1]) > 600;


      /* Exclude observations which are more than 1500 m above the 
         reference pressure, if one is specified.
         This prevents alot of bogus values from being added, but may cause 
         results to be different than expected if user is unaware of this. */

      datagap = datagap || (d0 < sptr->pref-1500);

     if (!datagap) {
        fprintf(sptr->fptr,"%5d %2d %8.3f %8.3f ", hdr.year, hdr.month, hdr.lat,
                hdr.lon);

        for (i = 0; i < MAXPROP; ++i ) {

          if (prop_req[i]) {   /* was this property requested? */

/* !**! Special cases for individual properties... */

             switch ((enum property) i) {

               case BF:    /* derivative properties */
               case PV:
                    if (already_deriv[i]) 
                       val = project_prop(station.observ[i], hdr.nobs, sptr);
                    else {
                       dlat = (double) hdr.lat;
                       val = project_deriv_prop(i, p0, hdr.nobs, dlat);
                    }
                    break;

               case OX:    /* optional observed properties */
               case N2:
               case N3:
               case P4:
               case SI:
                    if (!available( (enum property)i, hdr)) {
                       val = -9999.;
                       break;
                    }
                    /* if available fall through to default*/

               default:    /* all other properties */ 
                    val = project_prop(station.observ[i], hdr.nobs, sptr);
                    break;
             } /* end switch */

              sprintf(field_string," %c%d.%dlf", '%', get_field_width(i), get_field_precis(i));
             fprintf(sptr->fptr, field_string, val);
          } 
        }  /* end for */
        fprintf(sptr->fptr, "\n");

      }   /* end if  !datagap */
       
   }

   return;

}  /* end outputdata() */

/****************************************************************************/
double project_prop(obs, nobs, sptr)
double *obs;           /* array of observations for property */
int  nobs;             /* number of observations */
struct surface *sptr;  /* specifies the surface on which to project the prop */

   /*   Projects property values onto a specified surface...Returns the value
        of the property at the pressure level corresponding to this surface,
        or -9999 if the property cannot be projected.  The global arrays 
        pr & de are used.  */
{
   double val;
   double x[2], y[2];
   int    j;
   int   prevdepth, curdepth;

   prevdepth = curdepth = 0;
   j = 0;

   /* find first valid observation level. */

  while ((obs[j] < -8.9)||(station.observ[sptr->data_ind][j] < -8.9)) {
     if (++j == nobs)
          return (-9999.);
  }
  x[0] = station.observ[sptr->data_ind][j];
  y[0] = obs[j];

 /* Now, check successive pairs of datapoints until the 
    surface is found or the last observation is encountered.   */

  while (++j < nobs) {
    if ((obs[j] > -8.9) && (station.observ[sptr->data_ind][j] > -8.9)) {
        x[1] = station.observ[sptr->data_ind][j]; 
        y[1] = obs[j];
        if ((val = interpolate(sptr->value, x, y, 2)) > -9998.) {
           return (val);
        }
        x[0] = x[1];
        y[0] = y[1];
     }
  }  /* end while */

  return (-9999.);
} /* end project_prop() */
/****************************************************************************/
double project_deriv_prop(index, p0, nobs, lat)
int index;       /* identifies property to be projected */
double p0;       /* pressure at surface on which property will be projected */
int nobs;        /* number of p,t,s observation levels at this station */
double lat;      /* latitude of this station */

   /* Computes the value of a derivative property at the specified pressure 
      level from the global station data arrays.  */
{
   float deltap;
   double y;

/*  !**!  Special cases for individual properties... */

   switch ((enum property)index) {
      case BF:
          deltap = 50;   /* pressure interval in db.*/
          y = buoyancy(p0, station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA], nobs, deltap);
          if (y > -9998.)
             y = y * 1.0e5;
          break;
      case PV:
          deltap = 50;   /* pressure interval in db.*/
          y = buoyancy(p0, station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA], nobs, deltap);
          if (y < -9998.0)
              break;
          y = potvort((y*y), lat);
          break;
      default:
          y = -9999.0;
          break;

   }  /* end switch */

   return (y);

}  /* end project_deriv_prop() */

/****************************************************************************/
/* Performs a linear interpolation to find the position of xval in array, x,
   and returns the corresponding value in the array, y.  If xval does not
   appear in array x, the value -9999 is returned.  This routine assumes that
   the x array is monotonic and continuous (no missing values); and it
   assumes the y array is continuous.    */

double interpolate(xval, x, y, nypts)
double xval, *x, *y;
int    nypts;
{
   int    k;
   double v1, v2;

   for (k = 0; k < nypts-1; ++k) {

      v1 = xval - x[k];
      v2 = xval - x[k+1];

      if (v1 == 0)             /* x[k] == xval */
          return (y[k]);
      if (v2 == 0)             /* x[k+1] == xval */
          return (y[k+1]);
      if (v1 < 0. && v2 < 0.)  /* xval not between x1 and x2 */  
          continue;
      if (v1 > 0. && v2 > 0.) 
          continue;

      return ( y[k] + (y[k+1] - y[k]) * v1 / (x[k+1] - x[k]) );
   }

   return (-9999.0);

}   /* end interpolate() */

