/* xy4gmt.c
................................................................................
.   creates a list of  property pairs for each observation level in
.   a file of hydro stations.   Separates stations with a '>' to permit
.   points to be connected in GMT...
.
.  If -I option NOT specified, reads from STDIN; writes to STDOUT
.                          
................................................................................
................................................................................
*/

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "hydrobase.h"

#define   STDIN       0  

  /* functions in prop_subs.c... */
  
extern int get_prop_indx();   
extern int get_field_width();         
extern int get_field_precis(); 

  /* functions in hydro_utils.c ...*/

extern int open_hydro_file();
extern int get_station();
extern int available();        

void main (argc, argv)
int   argc;
char *argv[];
{
   double  d;
   struct HYDRO_HDR hdr;
   struct HYDRO_DATA data;
   int  i, n, error;
   short dflag, xflag, yflag, mflag;
   int mindepth, maxdepth;
   int scan_ok, any_scans, status;
   int infile;
   int xindex, yindex;
   enum property X, Y;
   char *s, str_format[30];
   void report_status();
   void print_usage();


/* check for command line arguments */

   if (argc < 2) {
     print_usage(argv[0]);
     exit(1);
   }

/* initialize these */

   xflag = yflag = dflag = mflag = 0;
   infile = STDIN;

/*  parse the command line arguments */

   for (i = 1; i < argc; i++) {
      error =  (argv[i][0] == '-') ? 0 : 1;
      if (!error) {
            switch (argv[i][1]) {
               case 'D':
                        dflag = 1;
                        s = &argv[i][2];  
                        error += (sscanf(s,"%d", &mindepth) != 1);
                        s = strchr(s,'/');  /* check for another delimiter*/
                        error = (s == NULL);
                        if (s != NULL) {
                          ++s;  /* move past delimiter */
                          error += (sscanf(s,"%d", &maxdepth) != 1);
                        }
                        if (error) break;
                        if (maxdepth < mindepth) {
                           fprintf(stderr,"\nmindepth exceeds maxdepth\n");
                           exit(1);
                        }
                        break;

               case 'I':                   /* get input file */
                        infile = open_hydro_file("", &argv[i][2],"", 1);
                        if (infile < 0) {
                           fprintf(stderr,"\n Unable to open %s \n", &argv[i][2]);
                           exit(1);
                        }  
                        break;
               case 'M':                   /* specify multiple segment option */
                        mflag = 1;
                        break;
               case 'X':                    /* get x-property */
                        xflag = 1;
                        xindex = get_prop_indx(&argv[i][2]);
                        if (xindex < 0) {
                           fprintf(stderr,"\nUnknown X property requested: %s\n", &argv[i][2]);
                           exit(1); 
                        }
                        X = (enum property) xindex;
                        break;
               case 'Y':                    /* get y-property */
                        yflag = 1;
                        yindex = get_prop_indx(&argv[i][2]);
                        if (yindex < 0) {
                           fprintf(stderr,"\nUnknown Y property requested: %s\n", &argv[i][2]);
                           exit(1); 
                        }
                        Y = (enum property) yindex;
                        break;
               default:
                       error = 1;

          }    /* end switch */
       }  /* end if */


       if (error ) {
             fprintf(stderr,"\nError parsing command line args.\n");
             fprintf(stderr,"     in particular: '%s'\n", argv[i]);
             exit(1);
       }


   }  /* end for */

   if (! (xflag && yflag) ) {
       fprintf(stderr,"\nYou must specify -X and -Y properties ...\n");
       exit(1);
   }
   
   sprintf(str_format," %c%d.%dlf  %c%d.%dlf \n", '%',get_field_width(xindex),
       get_field_precis(xindex), '%', get_field_width(yindex),
       get_field_precis(yindex));
   
      /* loop to read each station */

   while ((status = get_station(infile, &hdr, &data)) == 0) {  
       any_scans = 0;
       if (available(X,hdr) && available(Y,hdr)) {
          for (n = 0; n < hdr.nobs; ++n) {
          
            scan_ok = 1;
            if (dflag) {
               d = data.observ[(int)DE][n];
               scan_ok = (d >= (double)mindepth) && (d <= (double)maxdepth);
            }
            
            if (scan_ok) {
               ++any_scans;
               fprintf(stdout,str_format, data.observ[xindex][n],
                         data.observ[yindex][n]);
            }              

          }  /* end for */
          
          if (any_scans && mflag) {           /* station separator */
             fprintf(stdout,"> \n" ); 
          }
                    
      } /* end if */ 
   }  /*end while */ 
 
   report_status(status, stderr);

   exit(0);

}  /* end main */

/****************************************************************************/

void print_usage(program)
char *program;
{
   fprintf(stderr,"\nUsage:  %s -X<x_property> -Y<y_property> [-I<input_filename>] [-Dmindepth/maxdepth]  ", program);
   fprintf(stderr,"\n\n    -X  : 2-char property mnemonic to specify X");
   fprintf(stderr,"\n    -Y  : 2-char property mnemonic to specify Y");
   fprintf(stderr,"\n   [-I] : optional input file_name");
   fprintf(stderr,"\n   [-D] : optional depth range for observations");
   fprintf(stderr,"\n   [-M] : (optional) separate stations with > symbol");
   fprintf(stderr,"\n\n");  
   return;
}

