/*  cdf2asc.c

                *****************************
                      H Y D R O B A S E
                *****************************
.
.  USAGE:
.  cdf2asc filename [-Ttime_bin_index]  > outfile
.
.  Reads a .cdf file created by HydroBase grid3d and creates an ascii file
.  in HydroBase station format with each gridnode in the cdf file forming
.  a pseudo-station.
.
*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include "hydro_cdf.h"
#include "hydrobase.h"

#define STDOUT 1

  /* functions found in hydro_cdf.c */
extern int cdf_open();
extern int read_cdf_hdr();
extern int read_cdf_prop();
extern int read_cdf_depths();
extern int read_cdf_bottom_depth();
extern int get_lat_lon();
extern void cdf_close();

  /* functions found in hydro_utils.c */
extern int write_hydro_station(); 
extern int ms10();          

  /* functions found in prop_subs.c */
extern int get_prop_indx();

main (argc, argv)
int argc;
char *argv[];
{
   int cdfid, n, i, j, index, col, row, tbin;
   int error, print_mess ;
   int index_prop, no_values;
   struct CDF_HDR cdf;
   struct HYDRO_DATA station;
   struct HYDRO_HDR hdr;
   float *z, *x;
   double *p, *t, *s;
   void print_usage();

   if (argc < 2) {
      print_usage(argv[0]);
      exit(1);
   }

/*  set these default values */

   tbin = 0;
   print_mess = 1;
   error = 0;
   cdfid = -1;

/* parse the command line arguments */

   for (i = 1; i < argc; i++) {
      if (argv[i][0] == '-') {
            switch (argv[i][1]) {

               case 'T':
                        error = (sscanf(&argv[i][2],"%d", &tbin) != 1);
                        break;
               default:
                        error = 1;

          }    /* end switch */

          if (error ) {
             fprintf(stderr,"\nError parsing command line args.\n");
             fprintf(stderr,"     in particular: '%s'\n", argv[i]);
             fprintf(stderr,"\nFor usage instructions, type %s\n\n", argv[0]);
             exit(1);
          }

       }  /* end if */

       else {
           if ((cdfid = cdf_open("", argv[i], "", print_mess)) < 0)
               exit(1);
       }

   }  /* end for */

   if (cdfid < 0) {
       fprintf(stderr,"\nYou must specify an input file!\n");
       fprintf(stderr,"\nFor usage instructions, type %s\n\n", argv[0]);
       exit(1);
   }

/* read cdf file ... */

   if (error = read_cdf_hdr(cdfid, &cdf)) {
       if (error == 2) {
         fprintf(stderr, "\n Error in read_cdf_hdr():");
         fprintf(stderr, " Not a HydroBase cdf file!\n");
         exit (1);
       }
       else {
         fprintf(stderr, "\n Unknown Error in read_cdf_hdr():");
         fprintf(stderr, "    error code = %d\n", error);
         exit (1);
       }
   }

/* set up header for creating HydroBase stations... */

   strncpy(hdr.country, "XX", 3);
   strncpy(hdr.ship, "XX", 3);
   hdr.cruise = 999;
   hdr.station = 999;
   hdr.year = 999;
   hdr.month = 99;
   hdr.day = 99;
   hdr.nprops = station.nprops = cdf.nprops + 1;   /* add depth as a property */

   hdr.prop_id = (int *) malloc(hdr.nprops * sizeof(int));

/* get property IDs, insert depth at top of list ...
   allocate memory to store the station data ... */

   hdr.prop_id[0] = (int) DE;
   station.observ[(int)DE] = (double *) malloc(cdf.nz * sizeof(double));
   for (i = 1; i < hdr.nprops; ++i ) {
      index = hdr.prop_id[i] = get_prop_indx(cdf.prop_id[i-1]);
      station.observ[index] = (double *) malloc(cdf.nz * sizeof(double));
   }
   x = (float *) malloc(cdf.nz * sizeof(float));
   z = (float *) malloc(cdf.nz * sizeof(float));

/* get the standard depths used in the cdf file */

   n = read_cdf_depths(cdfid, z);

/* determine the best property to determine whether any data is available
   at a given depth in .cdf file */

   if (station.observ[(int)TE] != NULL)
       index_prop = (int)TE;
   else if (station.observ[(int)TH] != NULL)
       index_prop = (int)TH;
   else if (station.observ[(int)SA] != NULL)
       index_prop = (int)SA;
   else if (station.observ[(int)PR] != NULL)
       index_prop = (int)PR;
   else
       index_prop = hdr.prop_id[1];


/* visit each gridpoint in the cdf file and create a station profile */

   for (row = 0; row < cdf.ny; ++row) {
      for (col = 0; col < cdf.nx; ++col) {

/* first get the various properties, and convert from float to double ... */

         no_values = 1;
         for (i = 0; i < cdf.nprops; ++i) {
            error = read_cdf_prop(cdfid, cdf.prop_id[i], x, row, col,
                     tbin, 0, cdf.nz);
            for (n = 0; n < cdf.nz; ++n) {
               station.observ[hdr.prop_id[i+1]][n] = (double) x[n];
               no_values = no_values && (x[n] < -9.);
            }
         }

         if (no_values)   
            continue;

/*  now convert depth from float to double.  If p, t, s are present, eliminate
    scans which do not have a value for the index property. */
         n = 0;
         for (j = 0; j < cdf.nz; ++j) {
            station.observ[(int)DE][j] = (double) z[j];

            if (station.observ[index_prop][j] > -9.) {
                 for (i = 0; i < hdr.nprops; ++i) {
                    index = hdr.prop_id[i];
                    station.observ[index][n] = station.observ[index][j];
                 }
                 ++n;
            }

         }  /* end for j */

         station.nobs = hdr.nobs =  n;

         if (hdr.nobs == 0)
            continue;

/* now get bottom depth associated with this gridpoint and insert it into
   the depth array ... */

         error = read_cdf_bottom_depth(cdfid, &x[0], row, col, tbin);
         station.observ[(int)DE][hdr.nobs-1] = (double) x[0];

/*  fill in header elements specific to this gridnode ... */

         hdr.pdr = (int) x[0] + 1;
         error = get_lat_lon(cdf, row, col, &hdr.lat, &hdr.lon);
         hdr.ms10 = ms10(hdr.lat, hdr.lon, &hdr.ms1);

         error = write_hydro_station(STDOUT, hdr, station);

      }  /* end for col */
   }  /* end for row */

   cdf_close(cdfid);
   fprintf(stderr, "\n End of cdf2asc.\n");

   exit(0);


} /* end main */


void print_usage(program)
char * program;
{
   fprintf(stderr,"\nUsage: %s cdf_file_name [-Ttime_bin_index] \n", program);
   return;
} /* end print_usage() */
