/* hydro_cdf.c
.
.  A set of routines to facilitate the creation and reading of netCDF files 
.  in conjunction with HydroBase software.  These functions utilize the NCAR
.  netCDF C-interface routines, which implement XDR.
.
.  int cdf_init :           Opens a new CDF file for writing.   
.  int cdf_open :           Open a cdf file for reading.
.  void cdf_close :         Close a cdf file.
.  int cdf_define :         Defines all variables and attributes for cdf file.
.  int write_prop_cdf :     Writes values of a property at a lat/lon node.
.  int write_prop_count_cdf: Writes # of obs for a property at a lat/lon node.
.  int write_std_depths_cdf: Writes standard depths to de variable.
.  int write_time_bins_cdf : Writes min/max values defining time bins.
.  int write_bottom_depth_cdf:Writes the bottom depth values to bottom variable.
.  int read_cdf_hdr :       Reads header info from a cdf file.
.  int read_cdf_prop :      Reads all values of a property at a lat/lon node.
.  int read_cdf_prop_count: Reads # of obs for a property at a lat/lon node.
.  int read_cdf_depths :    Reads values of depth variable from cdf file.
.  int read_cdf_bottom_depth :  Reads values of bottom depth from cdf file.
.  int get_indices :        Converts lat/lon to row/col for a cdf file.
.  int get_lat_lon :        Converts row/col to lat/lon.
.  int std_depth_init() :     initializes globally defined std_depth array.
.  int d2stdlev(depth)    :   returns stddepth index associated with depth
.  double stdlev2depth(index): returns depth associated with stddepth index.
  
.
*/
#include <stdio.h>
#include <string.h>
#include "netcdf.h"
#include "hydro_cdf.h"
#include "hydrobase.h"

#define PROP_DIM 4        /* # of dimensions describing each property */


#define NDEF_DEPTHS  41  /* dimension of def_depth array */

double def_depth[NDEF_DEPTHS] = { 0,10, 20,  30,  50,  75, 100, 125, 150, 200,
                             250, 300, 400, 500, 600, 700, 800, 900,1000,1100,
                            1200,1300,1400,1500,1750,2000,2500,3000,3500,4000,
                            4500,5000,5500,6000,6500,7000,7500,8000,8500,9000,
                            -99
 };

/*********************************************************************/ 

int cdf_init(filename)
char *filename;
/* Creates a new netCDF file and returns the netCDF id for that file */
{
   int cdfid;

   cdfid = nccreate(filename, NC_CLOBBER);
   if (cdfid < 0) {
      fprintf(stderr, "\nUnable to create cdf_file: %s.", filename);
      fprintf(stderr, "\n    error = %d", ncerr);
      exit(1);
   }
   return (cdfid);
   
}  /* end cdf_init() */

/***************************************************************************/

int cdf_open(dir, file_root, extent, print_mess)
char *dir;          /* directory of file  OR "" */
char *file_root;    /* root of file name  OR entire file_path */
char *extent;       /* file extent        OR "" */
int  print_mess;    /* 1/0 to print/suppress message on stderr */

   /* opens an existing cdf file for reading and returns the file descriptor 
      to be used in future read operations. If file does not exist, a message
      is printed on stderr and -1 is returned. */
{
   int cdfid;
   int i;
   char fname[80];

/* construct filename from arguments */
        strcpy(fname, dir);
        if ((i = strlen(dir)) != 0) {
           if (dir[i-1] != '/')
              strncat(fname,"/",1);
        }

        strcat(fname, file_root);

        if ((strlen(extent) > 0) && (extent[0] != '.'))
           strncat(fname,".",1);
        strcat(fname, extent);

   if (print_mess) {
      if ((cdfid = ncopen(fname, NC_NOWRITE)) < 0) {
         fprintf(stderr,"\n Unable to open %s.\n", fname);
      }
      else {
          fprintf(stderr,"\nOpened %s ...", fname);
      }
   }

   return (cdfid);

} /* end cdf_open() */

/***************************************************************************/

void cdf_close(cdfid)
int cdfid;
{
   ncclose(cdfid);
   return;

} /* end cdf_close() */

/*********************************************************************/ 

int cdf_define(cdfid, h, prefill, add_count_vars)
int cdfid;           /*  id of already open cdf file */
struct CDF_HDR h;    /* header defined in cdf.h */
int prefill;         /* 0 = NOT prefill; or 1 = prefill */
int add_count_vars;  /* 1/0 = include/not_include variables counting # obs for each prop */

/* Defines all variables and attributes for the CDF file based on the info
   passed in struct CDF_HDR. If the prefill flag is set, each variable will
   be prefilled with the fill-value assigned in the header info.  By not
   prefilling, the write operations will be more efficent, however each point
   must be filled by the program calling the writing routines.  If 
   add_count_vars is set, variables will be defined to store the number of
   observations which determined the value of each property at each point.
   By convention, these will be named by appending "_cnt" to the property
   mnemonic.  
   
   After writing this info to the cdf_file, it takes the open file out of
   define mode.  Returns -1 if an error occurs.
*/ 
{
   int lat_dim, lon_dim, depth_dim, time_dim;
   int depth_id, bottom_id;
   int minyear_id, maxyear_id;
   int prop_id[MAXPROP * 2];             /* accommodate all props + counts */
   int dims[PROP_DIM], nchars, error;
   char *descrip;
   char varname[10];
   int i, nvars;
   short e_code;
   short short_val = 0;                   /* fill value for count variables */
   extern int get_prop_indx();            /* functions in prop_subs.c */
   extern char *get_prop_descrip();

/* define dimensions */

   lat_dim = ncdimdef(cdfid, "lat", (long) h.ny);
   if (lat_dim < 0) {
      return (-1);
   }
   lon_dim = ncdimdef(cdfid, "lon", (long) h.nx);
   depth_dim = ncdimdef(cdfid, "de", (long) h.nz);
   time_dim = ncdimdef(cdfid, "time", (long) h.nt);

/* define variables */

   dims[0] = depth_dim;
   depth_id = ncvardef(cdfid, "de", NC_FLOAT, 1, dims);
   ncattput(cdfid, depth_id, "units", NC_CHAR, 6, "meters");
   ncattput(cdfid, depth_id, "long_name", NC_CHAR, 10, "DEPTH  (M)");
   ncattput(cdfid, depth_id, "generic_name", NC_CHAR, 5, "depth");
   e_code = 3;
   ncattput(cdfid, depth_id, "epic_code", NC_SHORT, 1, (void *)&e_code);

   dims[0] = time_dim;
   minyear_id = ncvardef(cdfid, "minyear", NC_LONG, 1, dims);
   ncattput(cdfid, minyear_id, "units", NC_CHAR, 5, "years");

   maxyear_id = ncvardef(cdfid, "maxyear", NC_LONG, 1, dims);
   ncattput(cdfid, maxyear_id, "units", NC_CHAR, 5, "years");

   dims[0] = time_dim;
   dims[1] = lat_dim;
   dims[2] = lon_dim;

   e_code = 0;
   bottom_id = ncvardef(cdfid,"bottom", NC_FLOAT, 3, dims);
   ncattput(cdfid, bottom_id, "units", NC_CHAR, 6, "meters");
   ncattput(cdfid, bottom_id, "long_name", NC_CHAR, 12, "Bottom Depth");
   ncattput(cdfid, depth_id, "epic_code", NC_SHORT, 1, (void *)&e_code);

   dims[0] = time_dim;
   dims[1] = lat_dim;
   dims[2] = lon_dim;
   dims[3] = depth_dim;
   for (i = 0; i < h.nprops; ++i) {
      prop_id[i] = ncvardef(cdfid, h.prop_id[i], NC_FLOAT, 4, dims);
      nchars = strlen(h.prop_units[i]);
      ncattput(cdfid, prop_id[i], "units", NC_CHAR, nchars, (void *) h.prop_units[i]);
      ncattput(cdfid, prop_id[i], "_FillValue", NC_FLOAT, 1, (void *) &h.fill_value);
      descrip = get_prop_descrip(get_prop_indx(h.prop_id[i]));
      nchars = strlen(descrip);
      ncattput(cdfid, prop_id[i], "long_name", NC_CHAR, nchars, (void *) descrip);
   }
   nvars = i;
   if (add_count_vars) {   /* add a count var for each prop + one for depth */
      for (i= 0; i < h.nprops; ++i) {
         strncpy(varname, h.prop_id[i], 3);
         strcat (varname, COUNT_VAR_SUFFIX);
         prop_id[nvars] = ncvardef(cdfid, varname, NC_SHORT, 4, dims);
         ncattput(cdfid, prop_id[nvars], "_FillValue", NC_SHORT, 1, (void *) &short_val);
         ++nvars;
      }
         prop_id[nvars] = ncvardef(cdfid, "de_cnt", NC_SHORT, 4, dims);
         ncattput(cdfid, prop_id[nvars], "_FillValue", NC_SHORT, 1, (void *) &short_val);
   }

/* assign global attributes */ 

   ncattput(cdfid, NC_GLOBAL, "latmin", NC_FLOAT, 1, (void *) &h.ymin);
   ncattput(cdfid, NC_GLOBAL, "latmax", NC_FLOAT, 1, (void *) &h.ymax);
   ncattput(cdfid, NC_GLOBAL, "latincr", NC_FLOAT, 1, (void *) &h.yincr);
   ncattput(cdfid, NC_GLOBAL, "lonmin", NC_FLOAT, 1, (void *) &h.xmin);
   ncattput(cdfid, NC_GLOBAL, "lonmax", NC_FLOAT, 1, (void *) &h.xmax);
   ncattput(cdfid, NC_GLOBAL, "lonincr", NC_FLOAT, 1, (void *) &h.xincr);
   ncattput(cdfid, NC_GLOBAL, "node_offset", NC_LONG, 1, (void *) &h.node_offset);
   ncattput(cdfid, NC_GLOBAL, "nprops", NC_LONG, 1, (void *) &h.nprops);
   ncattput(cdfid, NC_GLOBAL, "counts_included", NC_LONG, 1, (void *) &add_count_vars);
   nchars = strlen(h.title);
   ncattput(cdfid, NC_GLOBAL, "title", NC_CHAR, nchars, (void *) h.title);
   nchars = strlen(h.command);
   ncattput(cdfid, NC_GLOBAL, "command", NC_CHAR, nchars, (void *) h.command);
   ncattput(cdfid, NC_GLOBAL, "compliance", NC_CHAR, 31, (void *) "HydroBase  PMEL/EPIC  WHOI/OARS");

   if (!prefill) {
      ncsetfill(cdfid, NC_NOFILL);
   }
   ncendef(cdfid);       /* leave define mode */

   return 0;

}  /* end cdf_define() */

/*********************************************************************/ 


int write_prop_cdf(cdfid, data_ptr, prop_mne, row, col, tbin, zlev, nrows, ncols, ntbins, nlevs)
int cdfid;            /* cdf file identifier */
float *data_ptr;      /* start of data array */
char *prop_mne;       /* 2-char property mnemonic */
int row, col, tbin, zlev;   /* grid position to start writing data */
int nrows, ncols;     /* # of contiguous rows and cols stored in data array */
int ntbins, nlevs;    /* # of contiguous time bins and zlevs to output */

/* Writes the data representing a single property at a single gridnode for 
   n time bins and all depths to the already opened netcdf file.  Returns 0  
   for a successful write or prints an error message and exits in case of an 
   error.
*/
{
   long start[PROP_DIM], count[PROP_DIM];
   int error, varid;
   
   start[0] = tbin;      
   start[1] = row;
   start[2] = col;
   start[3] = zlev;

   count[0] = ntbins;
   count[1] = nrows;
   count[2] = ncols;
   count[3] = nlevs;

   varid = ncvarid(cdfid, prop_mne);
   error = ncvarput(cdfid, varid, start, count, (void *) data_ptr);
   if (error == -1) {
      fprintf(stderr,"\nError writing property %.2s to cdf_file.\n", prop_mne);
      fprintf(stderr,"       write_prop_cdf() exiting.\n\n");
   }
   return 0;

}  /* end write_prop_cdf() */

/*********************************************************************/ 


int write_prop_count_cdf(cdfid, cnt_ptr, prop_mne, row, col,tbin, zlev, nrows, ncols, ntbins, nlevs)
int cdfid;            /* cdf file identifier */
short *cnt_ptr;       /* start of count array */
char *prop_mne;       /* 2-char property mnemonic */
int row, col, zlev;   /* grid position to start writing data */
int tbin;             /*  "                              "   */
int nrows, ncols;     /* # of contiguous rows and cols stored in count array */
int ntbins;           /* # of contiguous time bins in count array */
int nlevs;            /* # of zlevs to output */

/* Writes the data representing the number of observations at each depth for a  
   property at a  lat/lon position to an already opened netcdf file.
   Returns 0 for a successful write or prints an error message
   and exits in case of an error.
*/
{
   long start[PROP_DIM], count[PROP_DIM];
   char varname[20];
   int error, varid;
   
   start[0] = tbin;      
   start[1] = row;
   start[2] = col;
   start[3] = zlev;

   count[0] = ntbins;
   count[1] = nrows;
   count[2] = ncols;
   count[3] = nlevs;

   strncpy(varname, prop_mne, 3);
   strcat(varname, COUNT_VAR_SUFFIX);
   varid = ncvarid(cdfid, varname);
   error = ncvarput(cdfid, varid, start, count, (void *) cnt_ptr);
   if (error == -1) {
      fprintf(stderr,"\nError writing property %.2s to cdf_file.\n", prop_mne);
      fprintf(stderr,"       write_prop_cdf() exiting.\n\n");
   }
   return 0;

}  /* end write_prop_cdf() */

/***************************************************************************/
int write_std_depths_cdf(cdfid, h)
int cdfid;
struct CDF_HDR h;
/* Write the standard depth values contained in global variable std_depth[]
    to an already opened cdf file. */
{
   int error, i;
   long start[1], count[1];
   int depth_id;
   float *depth_ptr;
   extern double std_depth[];   /* in properties.h */
   extern int std_depth_initialized;


   start[0] = 0;
   count[0] = h.nz;

   if (!std_depth_initialized) {
       fprintf(stderr, "\n standard depths not initialized! : cdf_define()\n");
       exit(1);
   }

 /*  They must be of type float; so first create an array of appropriate type*/

   depth_ptr = (float *) malloc (h.nz * sizeof(float));
   for (i = 0; i < h.nz; ++i)
      depth_ptr[i] = (float) std_depth[i];

   depth_id = ncvarid(cdfid, "de");

   error = ncvarput(cdfid, depth_id, start, count, (void *) depth_ptr);
   free(depth_ptr);
   if (error == -1) {
       fprintf(stderr, "\nError writing standard depths to cdf file.\n");
       fprintf(stderr, "    function ncvarput()  in : cdf_define()\n");
       exit(1);
   }
   return(0);
} /* end write_std_depths_cdf() */
/***************************************************************************/
int write_bottom_depth_cdf(cdfid, row, col, tbin, nrows, ncols, ntbins, data_ptr)
int cdfid;                  /* cdf file identifier */
int row, col, tbin;         /* grid position to start writing data */
int nrows, ncols, ntbins;   /* # of datapts in each dimension to output */
float *data_ptr;            /* start of data array */
{
   int  bottom_id, error;
   long start[3], count[3];

   start[0] = tbin;
   start[1] = row;
   start[2] = col;

   count[0] = ntbins;
   count[1] = nrows;
   count[2] = ncols;

   bottom_id = ncvarid(cdfid, "bottom");
   error = ncvarput(cdfid, bottom_id, start, count, (void *) data_ptr);
   return (error);

} /* end write_bottom_depth_cdf() */
/***************************************************************************/
int write_time_bins_cdf(cdfid, hptr)
int cdfid;
struct CDF_HDR *hptr;
/* Write the min/max values defining timebins to an already opened cdf file. */
{
   int error;
   long start[1], count[1];
   int minyear_id, maxyear_id;


   start[0] = 0;
   count[0] = hptr->nt;


   minyear_id = ncvarid(cdfid, "minyear");
   maxyear_id = ncvarid(cdfid, "maxyear");

   error = ncvarput(cdfid, minyear_id, start, count, (void *) hptr->tmin);
   error = ncvarput(cdfid, maxyear_id, start, count, (void *) hptr->tmax);
   if (error == -1) {
       fprintf(stderr, "\nError defining time bins in cdf file.\n");
       exit(1);
   }
   return(0);
} /* end write_std_depths_cdf() */
/***************************************************************************/


int read_cdf_hdr(cdfid, haddr)
int cdfid;
struct CDF_HDR *haddr;

   /* Reads an already opened cdf file written with the HydroBase utilities.
      Returns 0 for a successful read.  In case of an error, returns a 
      number > 0.   
               error codes:   1 :  file does not exist;
                              2 :  not a HydroBase cdf file;
   */
{
   int ndims, nvars, ngatts, recdim, natts, len;
   int lat_dim, lon_dim, depth_dim, time_dim;
   int  dim[MAX_VAR_DIMS];
   char varname[MAX_NC_NAME];
   int error, i, varid, np, n, found;
   nc_type datatype;
   char *mne;
   extern char *get_prop_mne();
   float *fval;
   int *lval;
   long start[1], count[1];

   fval = (float *) malloc(nctypelen(NC_FLOAT));
   lval = (int *) malloc(nctypelen(NC_LONG));

   /* get # of properties stored in file */ 

   ncattget(cdfid, NC_GLOBAL, "nprops", (void *) lval);
   np = *lval;

   if (ncinquire(cdfid, &ndims, &nvars, &ngatts, &recdim) < 0)
       return (1);

   if ((lat_dim = ncdimid (cdfid, "lat")) < 0 )
       return (2);
   error = ncdiminq(cdfid, lat_dim, (char *) 0, &(haddr->ny) );

   if ((lon_dim = ncdimid (cdfid, "lon")) < 0 )
       return (2);
   error = ncdiminq(cdfid, lon_dim, (char *) 0, &(haddr->nx) );

   if ((depth_dim = ncdimid (cdfid, "de")) < 0 )
       return (2);
   error = ncdiminq(cdfid, depth_dim, (char *) 0, &(haddr->nz) );

   if ((time_dim = ncdimid (cdfid, "time")) < 0 )
       return (2);
   error = ncdiminq(cdfid, time_dim, (char *) 0, &(haddr->nt) );

   /* Check each cdf variable .
      Get mnemonic and units for each property, and ranges for time bins.  */
   
   haddr->prop_id = (char **) malloc(np * sizeof(char *));
   haddr->prop_units = (char **) malloc(np * sizeof(char *));
   haddr->tmin = (int *) malloc(haddr->nt * sizeof(int));
   haddr->tmax = (int *) malloc(haddr->nt * sizeof(int));

   n = 0;
   for (varid = 0; varid < nvars; ++varid) {

      error = ncvarinq(cdfid, varid, varname, &datatype, &ndims, dim, &natts);

      found = 0;
      i = 0;
      while ((!found) && (i < MAXPROP)) {   /* Is it a property? */
          mne = get_prop_mne(i++);
          found = ! (strncmp(varname, mne, 3));  
      }

      if (found) {           

         if (strncmp(mne, "de", 2) == 0) {       /* for depth, just get units */
            ncattget(cdfid, varid, "units", (void *) haddr->z_units);
         }
         else {
            haddr->prop_id[n] = (char *) malloc(3);
            strncpy(haddr->prop_id[n], mne, 3);
            ncattinq(cdfid, varid, "units", &datatype, &len);
            haddr->prop_units[n] = (char *) malloc(len);
            ncattget(cdfid, varid, "units", (void *) haddr->prop_units[n]);
            ++n;
         }
      }
      else {
        start[0] = 0;
        count[0] = haddr->nt;
        if (strncmp(varname, "minyear", 7) == 0) {  /* get the time bin mins */
           ncvarget(cdfid, varid, start, count, (void *)haddr->tmin);
        }
        if (strncmp(varname, "maxyear", 7) == 0) {  /* get the time bin max */
           ncvarget(cdfid, varid, start, count, (void *)haddr->tmax);
        }
      }

   }

   if (n != np) {
     fprintf(stderr,"\nWARNING!");
     fprintf(stderr,"\n # of properties found in file [%d] does not match", n);
     fprintf(stderr,"\n  the # of props in file attribute information [%d]\n", np);
   }
   haddr->nprops = n; 



   strncpy(haddr->x_units, "degrees", 8);    /* longitude */
   strncpy(haddr->y_units, "degrees", 8);    /* latitude */
   strncpy(haddr->t_units, "years", 6);      /* time */


   /* get the fill value by checking the first variable's attribute */
       
   varid = ncvarid(cdfid, haddr->prop_id[0]);
   ncattget(cdfid, varid, "_FillValue", (void *) fval );
   haddr->fill_value = *fval;

   /* get global attributes */

   ncattget(cdfid, NC_GLOBAL, "latmin", (void *) fval);
   haddr->ymin = *fval;
   ncattget(cdfid, NC_GLOBAL, "latmax", (void *) fval);
   haddr->ymax = *fval;
   ncattget(cdfid, NC_GLOBAL, "latincr", (void *) fval);
   haddr->yincr = *fval;
   ncattget(cdfid, NC_GLOBAL, "lonmin", (void *) fval);
   haddr->xmin = *fval;
   ncattget(cdfid, NC_GLOBAL, "lonmax", (void *) fval);
   haddr->xmax = *fval;
   ncattget(cdfid, NC_GLOBAL, "lonincr", (void *) fval);
   haddr->xincr = *fval;
   ncattget(cdfid, NC_GLOBAL, "title", (void *) haddr->title);
   ncattget(cdfid, NC_GLOBAL, "node_offset", (void *) lval);
   haddr->node_offset = *lval;
   ncattget(cdfid, NC_GLOBAL, "counts_included", (void *) lval);
   haddr->counts_included = *lval;
   ncattget(cdfid, NC_GLOBAL, "command", (void *) &haddr->command[0]);

   return(0);
}  /* end read_cdf_hdr() */

/***************************************************************************/


int read_cdf_depths(cdfid, d_ptr)
int cdfid;         /* id of cdf file already opened for reading */
float *d_ptr;      /* address to return depth values */

   /* Retrieves the standard depth values used as a coordinate variable. 
      Returns the number of depth values or -1 in case of an error.
   */
{
   int depth_dim, varid;
   long  start[1], count[1];

   if ((depth_dim = ncdimid(cdfid, "de")) < 0)
       return (-1);

   if (ncdiminq(cdfid, depth_dim, (char *) 0, count) < 0)
       return (-1);

   if ( (varid = ncvarid(cdfid, "de")) < 0)
       return (-1);

   start[0] = 0;
   if ((ncvarget(cdfid, varid, start, count, (void *)d_ptr)) < 0)
       return (-1);

   return ((int) count[0]);

} /* end read_cdf_depths() */
      
/***************************************************************************/
int read_cdf_bottom_depth(cdfid, d_ptr, row, col, tbin)
int cdfid;               /* id of cdf file already opened for reading */
float *d_ptr;            /* address to return depth value */
int row, col, tbin;      /* offset into grid */

   /* Retrieves a single bottom depth value at row, col from a cdf file. 
      Returns 1 for a successful read, or -1 in case of an error.   */
{
   int varid;
   long start[3], count[3];

   start[0] = tbin;
   start[1] = row;
   start[2] = col;

   count[0] = 1;
   count[1] = 1;
   count[2] = 1;

   if ((varid = ncvarid(cdfid, "bottom")) < 0)
       return (-1);

   if (ncvarget(cdfid, varid, start, count, (void *)d_ptr) < 0)
       return (-1);

   return(1);

}  /* read_cdf_bottom_depth() */
/***************************************************************************/

int read_cdf_prop(cdfid, varname, dataptr, row, col, tbin, zlev, npts)
int cdfid;       /* id of cdf file already opened for reading */
char *varname;   /* property mnemonic */
float *dataptr;  /* address of array into which values will be read */
int row;         /* row corresponding to lat of gridnode to read */
int col;         /* column corresponding to lon of gridnode to read */
int tbin;        /* index to time bin */
int zlev;        /* depth level at which to begin */
int npts;        /* # of values to read */

   /* Reads an already opened cdf file written with the HydroBase utilities.
      Dataptr is the address of an array with enough space to hold npts
      amount of data. 
      Returns 0 for a successful read.  In case of an error, returns a 
      number > 0.   
               error codes:   1 :  file not open;
   */
{
   int varid, error;
   long start[PROP_DIM], count[PROP_DIM];

   varid = ncvarid(cdfid, varname);
   if (varid < 0 )
     return(1);

   start[0] = tbin;     
   start[1] = row;
   start[2] = col;
   start[3] = zlev;

   count[0] =  1;
   count[1] =  1;
   count[2] =  1;
   count[3] =  npts;

   error = ncvarget(cdfid, varid, start, count, (void *) dataptr);
   if (error < 0)
      return (1);

   return 0;

} /* read_cdf_prop() */

/***************************************************************************/


int read_cdf_prop_count(cdfid, prop_mne, cnt_ptr, row, col, tbin, zlev, npts)
int cdfid;       /* id of cdf file already opened for reading */
char *prop_mne;  /* 2-character property mnemonic */
short *cnt_ptr;  /* address of array into which values will be read */
int row;         /* lat of gridnode to read */
int col;         /* lon of gridnode to read */
int tbin;        /* index of time bin to read */
int zlev;        /* depth level at which to begin */
int npts;        /* # of values to read */

   /* Reads an already opened cdf file written with the HydroBase utilities.
      cnt_ptr is the address of an array with enough space to hold npts
      amount of data. 
      Returns 0 for a successful read.  In case of an error, returns a 
      number > 0.   
               error codes:   1 :  file not open;
   */
{
   int varid, error;
   char varname[10];
   long start[PROP_DIM], count[PROP_DIM];

   strncpy(varname, prop_mne, 3);
   strcat(varname, COUNT_VAR_SUFFIX);
   varid = ncvarid(cdfid, varname);
   if (varid < 0 )
     return(1);

   start[0] = tbin;    
   start[1] = row;
   start[2] = col;
   start[3] = zlev;

   count[0] =  1;
   count[1] =  1;
   count[2] =  1;
   count[3] =  npts;

   error = ncvarget(cdfid, varid, start, count, (void *) cnt_ptr);
   if (error < 0)
      return (1);

   return 0;

} /* read_cdf_prop_count() */

/***************************************************************************/

int get_indices(h, lat, lon, row_ptr, col_ptr)
struct CDF_HDR h;
float lat, lon; 
int *row_ptr, *col_ptr;

   /* Translates a latitude, longitude into a corresponding (row,col)
      from the info stored in the CDF_HDR. The lat/lon does
      not necessarily have to fall exactly on a grid node: i.e. each
      grid node has an associated area = yincr * xincr, whose definition
      depends on whether it is a node grid or a pixel grid. 
      Returns 0 for successful conversion within bounds; or a -1 if the
      row, col does not fall within the grid bounds.
      
   */
{  
   if (h.node_offset == 0)  {                         /* for node grids */ 
      if ((lat > h.ymax + h.yincr) || (lat < h.ymin - h.yincr))
         return(-1);
      if ((lon > h.xmax + h.xincr) || (lon < h.xmin - h.xincr))
         return(-1);

   /* the extra smidgeon puts stations that fall right on a border into the 
      proper box */


     *row_ptr = (int) (((h.ymax - (lat + .0001)) / h.yincr) - .5);
     *col_ptr = (int) (((lon - (h.xmin + .0001)) / h.xincr) - .5);
   }

   else {                                             /* for pixel grids */
      if ((lat > h.ymax) || (lat < h.ymin))
         return(-1);
      if ((lon > h.xmax) || (lon < h.xmin))
         return(-1);

   /* the extra smidgeon puts stations that fall right on a border into the 
      proper box */

     *row_ptr = (int) ((h.ymax - (lat + .0001)) / h.yincr);
     *col_ptr = (int) ((lon - (h.xmin + .0001)) / h.xincr);
   }


   return 0;
 
} /* end get_indices() */

/***************************************************************************/

int get_lat_lon(h, row, col, lat_ptr, lon_ptr)
struct CDF_HDR h;
int row, col;
float *lat_ptr, *lon_ptr;

    /* Translates the row,col into the lat,lon of its associated gridnode.
       Returns 0 for a successful conversion or -1 if the point is out
       of the gridbounds.
    */ 
{
   if ((row >= h.ny) || (row < 0) || (col >= h.nx) || (col < 0))
      return (-1);


   if (h.node_offset == 0) {                          /* for node grids */
      *lat_ptr = h.ymax - row * h.yincr;
      *lon_ptr = h.xmin + col * h.xincr;
   }
   else {                                             /* for pixel grids */
      *lat_ptr =  h.ymax - (row + .5) * h.yincr;
      *lon_ptr =  h.xmin + (col + .5) * h.xincr;
   }

  return (0);

}  /* end get_lat_lon() */
/**********************************************************************/
int std_depth_init(depth_file)
FILE *depth_file;
   /* Initializes the global variable std_depth and NSTDLEVS using values in 
      depth_file or the default values if depth_file points to NULL. Returns 
      the number of std depths */
{
   int i;
   extern double std_depth[];

/*  use default standard depths ... */
   if (depth_file == NULL) {
      for (i = 0; i < NDEF_DEPTHS; ++i)
         std_depth[i] = def_depth[i];

      std_depth_initialized = 1;
      NSTDLEVS = NDEF_DEPTHS;

      return(NDEF_DEPTHS);
   }

/*  get standard depths from file ... */
   i = 0;
   while (fscanf(depth_file,"%lf", &std_depth[i] ) == 1) {
     if (++i >= MAXSTDLEVS) {
        fprintf(stderr,"\nOnly able to use %d standard depths from file.\n", i);
        break;
     }
  
   }
   fclose (depth_file);
   NSTDLEVS = i;
   std_depth_initialized = 1;
   return (i);

}  /* end std_depth_init() */

/**********************************************************************/
int d2stdlev(depth)
double depth;

   /* Returns the standard level index corresponding to depth.  Or a
      negative number if some error occurs:
                 -1 :  not a standard depth 
                 -2 :  standard depths have not been defined yet.
                 -99:  depth exceeds greatest standard depth.
      The global variables std_depth[] and NSTDLEVS are necessary and 
      should have been initialized prior to calling this function.
   */
{

   int i, d, sd;
   
/*  make sure standard depths have been defined ... */

   if (!std_depth_initialized)
       return (-2);

/*  check each standard depth and compare integer values to avoid problems 
    of floating point storage ... */

   d = (int) (depth +.00001);
   i = -1;
   while (++i < NSTDLEVS) {

      if (d < (sd = (int) (std_depth[i] +.00001)))
          return (-1);

      if (d == sd)
          return(i);
   }

/* if this statement is reached, the depth exceeds greatest standard depth 
     defined */
   return (-99);

}
/**********************************************************************/
double stdlev2depth(index)
int index;

   /* Returns the depth corresponding to stddepth index .  Or a
      -99.0 if the index is not a stddepth index.
   */
{
   if ((index < 0) || (index >= NSTDLEVS))
       return (-99.0);

   return (std_depth[index]);

} /* end of stdlev2depth */
