/* ms5sort.c
............................................................................
.   Divides HydroBase stations into 5 x 5 deg files.  The naming
.   convention for output files is    ####_*.ext   where #### = 10-deg MSq.
.                                                            * = {0,1,2,3}
.   The 5-deg square is determined by the 1-deg Marsden Square.
.
..............................................................................
*/
#include <stdio.h>
#include <string.h>
#include "hydrobase.h"

#define   MAXF     50     /* max # of output files ... cannot exceed 60 */
#define   NFOUT    4    /* # of 5 degree squares in ten degrees */
#define   DIR     ""                 
#define   EXTENT   ""
#define   PRINT_MSG  1    /* 0 or 1 */


int msq1[100] =  { 0, 0, 0, 0, 0, 1, 1, 1, 1, 1,  /* converts ms_1 to ms_5 */
                   0, 0, 0, 0, 0, 1, 1, 1, 1, 1,
                   0, 0, 0, 0, 0, 1, 1, 1, 1, 1,
                   0, 0, 0, 0, 0, 1, 1, 1, 1, 1,
                   0, 0, 0, 0, 0, 1, 1, 1, 1, 1,
                   2, 2, 2, 2, 2, 3, 3, 3, 3, 3,
                   2, 2, 2, 2, 2, 3, 3, 3, 3, 3,
                   2, 2, 2, 2, 2, 3, 3, 3, 3, 3,
                   2, 2, 2, 2, 2, 3, 3, 3, 3, 3,
                   2, 2, 2, 2, 2, 3, 3, 3, 3, 3  };


char *fext[NFOUT] = { "_0", "_1", "_2", "_3"};



main (argc, argv)
int   argc;
char *argv[];
{
register int j;
int  infile, mode;
int  msq5[MAXF + 1], msq10[MAXF + 1], count[MAXF + 1];
int  nfiles, i;
int  curfile = 1, nextfile = 0, status;
int  outfile[MAXF + 1];              /* file descriptors returned by open()*/
char nflag;
char *fname, name[80];
char *outpath, *extin, *extout, *dir;
struct HYDRO_HDR hdr;
struct HYDRO_DATA data;
void print_usage(), report_status();

/* set output path */

if (argc < 2) {
   print_usage(argv[0]);
   exit(1);
}

/*  set these default values */

    dir = DIR;
    outpath = DIR;
    extin = EXTENT;
    extout = EXTENT;
    mode = NOCLOBBER;
    nfiles = 0;
    nflag = 0;

/* parse command line arguments... */

   for (i = 1; i < argc; i++) {
      if (argv[i][0] == '-') {
            switch (argv[i][1]) {
               case 'D':
                        dir = &argv[i][2];
                        break;

               case 'E':
                        extin = &argv[i][2];
                        break;

               case 'O':
                        outpath = &argv[i][2];
                        break;
               case 'N':
                        nflag = 1;
                        extout = &argv[i][2];
                        break;
               case 'A':
                        mode = APPEND;
                        break;
               case 'T':
                        mode = OVERWRITE;
                        break;
               default :
                        print_usage(argv[0]);
                        fprintf(stderr,"\nError parsing command line");
                        fprintf(stderr,"\n in particular: %s\n", argv[i]);
                        exit(1);
            }  /* end switch */
       }  /* end if */
       else  {
           ++nfiles;
       }
   }  /* end for */

   if ( !nfiles || !nflag) {
       print_usage(argv[0]);
       fprintf(stderr,"\n\nYou must specify input file(s) and extent for output files!\n");
       exit(1);
   }
      
/* initialize the following */

for (j=0; j < MAXF; ++j){
       count[j] = 0;
       msq5[j] = msq10[j] = 0;
       outfile[j] = NULL; 
}
msq5[MAXF] = -1;
msq10[MAXF] = -1;
count[MAXF] = 0;

/* open the scratch file ... */

strcpy(name, outpath);
j = strlen(name);              /* check for easily forgotten slash in path */
if (j && (name[j-1] != '/')) 
   strncat(name,"/",2);
strncat(name, "msextra.dat", 12);

if ((outfile[MAXF] = create_hydro_file(name,mode)) < 0) {
       switch (mode) {
          case APPEND:
                fprintf(stderr,"\nUnable to open %s in append mode.", name);
                exit(1);
          case OVERWRITE:
                fprintf(stderr,"\nUnable to open %s.", name);
                exit(1);
          default:
                fprintf(stderr,"\nUnable to open %s.", name);
                fprintf(stderr,"  It may already exist...\n");
                exit(1);
       } /* end switch */
}
      
/* loop for each input file ... */

do {
   if (( infile = open_hydro_file(dir, argv[curfile], extin, PRINT_MSG)) < 0) {
      goto NEXTFILE;
   }

   /* loop to read each station */

   while ((status = get_station(infile, &hdr, &data)) == 0) {

           /* is ms file already open? */
      j = 0;   
      while ((j < nextfile) && !((msq5[j] == msq1[hdr.ms1])&&(msq10[j] == hdr.ms10))) 
          ++j; 

      if ((j == nextfile) && (j < MAXF)) {     /* open  an output file */
          outfile[j] = openfile(hdr.ms10, hdr.ms1, outpath, extout, mode);
          msq5[j] = msq1[hdr.ms1];
          msq10[j] = hdr.ms10;
          ++nextfile;
      } 

        /* write station to appropriate outfile */

      write_hydro_station(outfile[j], hdr, data);
      ++count[j];

   }  /* end while */
   report_status(status, stderr);

   close(infile);
NEXTFILE:
    ;
}  while (curfile++ < nfiles);


/*  write summary of distribution to stdout and close output files */

fprintf(stdout, "\n\n accounting of stations written to each file ...\n");

for (j=0; j < nextfile; ++j) {
       fprintf(stdout, "   %4d%s%s     %d \n", msq10[j], fext[msq5[j]], extout, count[j]);
       close(outfile[j]);
}

fprintf(stdout,"  msextra.dat    %d \n", count[MAXF]);
close(outfile[MAXF]);                            /* close scratch file */

exit(0);
} /* end of main */


/****************************************************************************/

void print_usage(program)
char *program;
{
   fprintf(stderr,"\nUsage:  %s list_of_filename_root(s)", program);

   fprintf(stderr," [-Ddir] [-Eextent] -Nnew_extent [-Ooutpath][-A] [-T]");
   fprintf(stderr," > logfile\n");
   fprintf(stderr,"\n   [-D] : specifies directory of input files (default is ./) ");
   fprintf(stderr,"\n        ex: -D../data/ ");
   fprintf(stderr,"\n   [-E] : specifies input file extent (default is no extent)");  
   fprintf(stderr,"\n        ex: -E.dat ");
   fprintf(stderr,"\n   [-O] : specifies directory of output files (default is ./) ");
   fprintf(stderr,"\n        ex: -O../natl/ ");
   fprintf(stderr,"\n    -N  : specifies output file extent (default is no extent)");  
   fprintf(stderr,"\n        ex: -N.dat ");
   fprintf(stderr,"\n   [-A] : append to existing files (default is do not alter an existing file.)");
   fprintf(stderr,"\n   [-T] : truncate existing files (default is do not alter an existing file.)");
   fprintf(stderr,"\n\n");  
   return;
} /* end print_usage() */


/****************************************************************************
* Opens an existing file or creates a new file for output. The files are named
* <outpath>####.<extent>, where #### is a 4-digit integer.
****************************************************************************/
int openfile(n, m, outpath, extent, mode)
int n, m;         /* ten-degree and one-degree MSquare designations */
char *outpath;    /* a null terminated string */
char *extent;
int mode;         /* APPEND, OVERWRITE or NOCLOBBER for output file */
{
   char fname[80];
   int  i;    

/* move beyond any leading dot in file extent... */
   if (*extent == '.')
      ++extent;

   /* check pathname for easily forgotten slash */

   i = strlen(outpath);
   if (i  && (outpath[i-1] != '/')) 
     sprintf(fname, "%s/%4d%s.%s", outpath, n, fext[msq1[m]], extent);  
   else
     sprintf(fname, "%s%4d%s.%s", outpath, n, fext[msq1[m]], extent);  
      

   if ((i = create_hydro_file(fname,mode)) < 0)  {
       switch (mode) {
          case APPEND:
                fprintf(stderr,"\nUnable to open %s in append mode.", fname);
                exit(1);
          case OVERWRITE:
                fprintf(stderr,"\nUnable to open %s.", fname);
                exit(1);
          default:
                fprintf(stderr,"\nUnable to open %s.", fname);
                fprintf(stderr,"  It may already exist...\n");
                exit(1);
       }
   }
   switch (mode) {
          case APPEND:
                fprintf(stderr,"\nOpening or appending %s.", fname);
                return(i);
          case OVERWRITE:
                fprintf(stderr,"\nOpening or overwriting %s.", fname);
                return(i);
          default:
                fprintf(stderr,"\nOpening %s.", fname);
                return(i);
   }

}  /* end of openfile */

