/* prop_subs.c
.
.  Functions which facilitate use of properties listed in "hydrobase.h"
.
.  int get_prop_indx(char *s) :
.           translates character string into an integer
.           representing an index to the list of properties.
.  void print_prop_menu() : 
.           prints a list of available properties on the stderr device.
.
.  char *get_prop_mne(int i)  :
.           returns the mnemonic associated with property index i.
.
.  char *get_prop_descrip(int i) :
.           Returns a char description of property[i].
.
.  char *get_prop_units(int i) :
.           Returns the units associated with property[i].
.
.  int get_field_width(int i) :
.           Returns the field width associated with property[i].
.
.  int get_field_precis(int i) :
.           Returns the decimal precision associated with property[i].
.
.  void compute_sigma(double pref, int nobs, double *sigma, double *p, 
.                     double *t, double *s):
.           Computes an array of sigma values for each level in p,t,s arrays.
.
.  void compute_svan(double pref, int nobs, double *sva, double *p, 
.                      double *t, double *s).
.           Computes an array of sp vol anom values referenced to a pr surface
.           for each level in p,t,s arrays.
.
.  void compute_sp_vol( int nobs, double *sv, double *p,  double *t, double *s)
.           Computes an array of spec. volume values (in situ) for each level  
.            in p,t,s arrays.
.
.  void compute_height(int nobs, double *p, double *t, double *s, double *ht) :
.           Computes dynamic height rel to sea surface for each level in the
.           p, t, s arrays.
.
.  void compute_theta(int nobs, double *th, double *p, double *t, double *s);
.           Computes potential temp rel to the sea surface for each level 
.           in the p, t, s arrays.
.
. double buoyancy(double p0, double *p, double *t, double *s, 
.                 int  nobs,  float pr_int, char units);
.           Computes brunt-vaisala frequency in cycles/hour or N^2 ( stability 
.           parameter) in  radians/sec**2  at pressure, p0 given a set of p,t,s
.           observations.
.
. double potvort(double e, double lat)
.           Returns potential vorticity for the specified value of stability 
.           parameter (n^2) and latitude.
.
. double interp(xval, x, y, nypts)
.           Linear interpolation to find value in y array corresponding to 
            position of xval in x array.
*/
#include <stdio.h>
#include <malloc.h>

#include "hydrobase.h"




/* character string mnemonics which correspond to properties */

char *prop_mne[MAXPROP] =  { "pr",   
                             "de",  
                             "te",   
                             "th",   
                             "sa",   
                             "ox",   
                             "n2",
                             "n3",
                             "p4",
                             "si",  
                             "ht",  
                             "s0",  
                             "s1",  
                             "s2", 
                             "s3", 
                             "s4",
                             "s_",
                             "bf",  
                             "pv",
                             "sv",
                             "va",   
                             "sf"
};

char *prop_descrip[MAXPROP] = { "pressure",
                                "depth",
                                "in situ temperature",
                                "potential temperature: pref= 0.",
                                "salinity",
                                "oxygen",
                                "nitrite",
                                "nitrate",
                                "phosphate",
                                "silicate",
                                "dynamic height",
                                "potential density: pref = 0.",
                                "potential density: pref = 1000.",
                                "potential density: pref = 2000.",
                                "potential density: pref = 3000.",
                                "potential density: pref = 4000.",
                                "potential density: pref = ?.",
                                "buoyancy frequency",
                                "potential vorticity",
                                "specific volume",
                                "specific volume anomaly",
                                "stream function"
} ;

char *prop_units[MAXPROP] = { "dbars",
                              "meters",
                              "degrees C",
                              "degrees C",
                              "psu",
                              "ml/liter",
                              "micromole/kg",
                              "micromole/kg",
                              "micromole/kg",
                              "micromole/kg",
                              "dyn. meters (= 10^-1 * m**2/s**2)",
                              "kg/m**3",
                              "kg/m**3",
                              "kg/m**3",
                              "kg/m**3",
                              "kg/m**3",
                              "kg/m**3",
                              "1.e^5 * radians/sec",
                              "1.e^12 * m^-1 sec^-1",
                              "1.e^8 * m**3/kg",
                              "1.e^8 * m**3/kg",
                              "m**2/s**2"
};

int field_width[MAXPROP] = {     8, /* pr */
                                 8, /* de */
                                 8, /* te */
                                 8, /* th */
                                 8, /* sa */
                                 8, /* ox */
                                 8, /* n2 */
                                 8, /* n3 */
                                 8, /* p4 */
                                 8, /* si */
                                 8, /* ht */
                                 8, /* s0 */
                                 8, /* s1 */
                                 8, /* s2 */
                                 8, /* s3 */
                                 8, /* s4 */
                                 8, /* s_ */
                                 8, /* bf */
                                 8, /* pv */
                                10, /* sv */
                                 8, /* va */
                                 8  /* sf */
};

int field_precis[MAXPROP] =  {   1, /* pr */
                                 1, /* de */
                                 3, /* te */
                                 3, /* th */
                                 3, /* sa */
                                 3, /* ox */
                                 3, /* n2 */
                                 3, /* n3 */
                                 3, /* p4 */
                                 3, /* si */
                                 3, /* ht */
                                 4, /* s0 */
                                 4, /* s1 */
                                 4, /* s2 */
                                 4, /* s3 */
                                 4, /* s4 */
                                 4, /* s_ */
                                 4, /* bf */
                                 3, /* pv */
                                 3, /* sv */
                                 4, /* va */
                                 3  /* sf */
};

void print_prop_menu()
{
   int i;

   for (i=0; i < MAXPROP; ++i) {
     fprintf(stderr,"\n%s : %s [%s]", prop_mne[i], prop_descrip[i], prop_units[i]);
   }
   fprintf(stderr,"\n");
   return;
} /* end print_prop_menu() */


int get_prop_indx(str)
char *str;
{
   int error = -1;
   char *s;

   s = str;
   switch (*s) {
      case 'b':
      case 'B':
            switch (*(++s)) {
               case 'F':
               case 'f':
                   return ((int) BF);  
               default:      
                   return (error);
             } 
            break;
      case 'd':
      case 'D':
            switch (*(++s)) {
               case 'E':
               case 'e':
                   return ((int) DE);  
               default:      
                   return (error);
             } 
            break;
      case 'H':
      case 'h':
            switch (*(++s)) {
               case 'T':
               case 't':
                   return ((int) HT);  
               default:      
                   return (error);
             } 
            break;
      case 'N':
      case 'n':
            switch (*(++s)) {
              case '2':
                 return ((int) N2);
              case '3':
                 return ((int) N3);
              default:
                 return error;
            }
            break;
      case 'O':
      case 'o':
            switch (*(++s)) {
               case 'X':
               case 'x':
                   return ((int) OX);  
               default:      
                   return (error);
             } 
            break;
      case 'P':
      case 'p':
            switch (*(++s)) {
               case 'R':
               case 'r':
                   return ((int) PR);
               case 'V':
               case 'v':
                   return ((int) PV);  
               case '4':
                   return ((int) P4);  
               default:      
                   return (error);
             } 
            break;
      case 'S':
      case 's':
            switch (*(++s)) {
              case 'A':
              case 'a':
                 return ((int) SA);
              case 'F':
              case 'f':
                 return ((int) SF);
              case 'I':
              case 'i':
                 return ((int) SI);
              case 'V':
              case 'v':
                 return ((int) SV);
              case '_':
                 return ((int) S_);
              case '0':
                 return ((int) S0);
              case '1':
                 return ((int) S1);
              case '2':
                 return ((int) S2);
              case '3':
                 return ((int) S3);
              case '4':
                 return ((int) S4);
              default:
                 return error;
            } /* end switch */
            break;
      case 'T':
      case 't':
            switch (*(++s)) {
              case 'E':
              case 'e':
                 return ((int) TE);
              case 'H':
              case 'h':
                 return ((int) TH);
     
              default:
                 return error;
            } /* end switch */
            break;
      case 'V':
      case 'v':
            switch (*(++s)) {
              case 'A':
              case 'a':
                 return ((int) VA);
              default:
                 return error;
            } /* end switch */
            break;

      default:
            return error;
   } /* end switch */
} /* end get_prop_indx() */



char *get_prop_mne(i)
int i;
/*  returns the ith property mnemonic */
{
   return prop_mne[i];
}



char *get_prop_descrip(i)
int i;
/* returns the ith property description */
{
   return prop_descrip[i];
}

int get_field_width(i)
int i;
/* returns the ith property field width */
{
   return field_width[i];
}

int get_field_precis(i)
int i;
/* returns the decimal precision used to describe property i */
{
   return field_precis[i];
}

char *get_prop_units(i)
int i;
/* returns the units for property[i] */
{
   return prop_units[i];
}
/****************************************************************************/
/*  Computes sigma values from  arrays of p, t, s at each observation
    level.     */

void compute_sigma(pref, nobs, sigma, p, t, s)
double pref;
int    nobs;
double *sigma, *p, *t, *s;
{
   int    j;
   double tref;
   double svan_(), theta_();    /* fortran functions */

      for (j = 0; j < nobs; ++j) {
         tref = theta_(&s[j], &t[j], &p[j], &pref);
         svan_(&s[j], &tref, &pref, &sigma[j]);
      }

      return;
}  /* end compute_sigma() */
/****************************************************************************/
/*  Computes in situ Specific Volume from p, t, s at each observation
    level.     */

void compute_sp_vol(nobs, spvol, p, t, s)
int    nobs;
double *spvol, *p, *t, *s;
{
   int    j;
   double  sigma;
   double svan_();    /* fortran function */


   /* Convert the sigma value (kg/m**3) to a rho value by adding 1000.
      Specific volume is 1/rho, but is multiplied here by 10^8 to 
      convert to units where significant digits can be seen. */
 
      for (j = 0; j < nobs; ++j) {
         svan_(&s[j], &t[j], &p[j], &sigma);
         spvol[j] = 1.0e08 / (sigma + 1000.);
      }

      return;

}  /* end compute_sp_vol() */
/****************************************************************************/
/*  Computes specific volume anomaly ( = spvol(p,t,s) - spvol(p,0,35)
     from  arrays of p, t, s at each observation level.     */

void compute_svan(nobs, sva, p, t, s)
int    nobs;
double *sva, *p, *t, *s;
{
   int    j;
   double sigma;
   double svan_();    /* fortran functions */

      for (j = 0; j < nobs; ++j) {
         sva[j] = svan_(&s[j], &t[j], &p[j], &sigma);
      }

      return;
}  /* end compute_svan() */
/****************************************************************************/
void compute_height(nobs, p, t, s, h)
int nobs;
double *p, *t, *s, *h;

/* computes dynamic height relative to 0 db. at each level in pressure
   array.  The units are :  
             dynamic height  : dyn meters = 1/10 * m**2/s**2 .  
           spec vol anomaly  :  1e8 * m**3/kg
                   pressure  :  dbars = 1e4 N/m**2 = 1e4 kg/m s**2

   For stations missing a surface value, the first available t & s are
    used to approximate the sp.vol.anom at the surface if the first observation
    is within 100 meters of the surface.  Otherwise the value -999 is
    returned for every observation.  If a vertical datagap is encountered, 
    no height is computed beneath that level
 */
{
   int j, start, i, datagap;
   double sva1, sva0, sig, last_h, last_p;
   double svan_();  /* fortran functions */


/* first observation below 100 meters? */

   start = 0;
   while (s[start] < -8. || t[start] < -8. || p[start] < -8) {
       ++start;
       h[start] = -999.;
   }

   if (p[start] > 110.) {
        for (j = 0; j < nobs; ++j) {
          h[j] = -999.;
        }
        return;
   }

/*  set up an initial value at the surface in case there is no actual surface
    observation.  Use the first available t and s to compute svan to
    approximate a mixed layer. */

   sva0 = svan_(&s[start], &t[start], &p[start], &sig);
   h[0] = 0.;
   if (p[start] > 0.) {
      sva1 = svan_(&s[start], &t[start], &p[start], &sig);
      h[start] = ((sva0 + sva1) *.5e-5 * p[start] );
      sva0 = sva1;
   }
   last_h = h[start];
   last_p = p[start];

/* now do the rest of the station, check for missing values in p,t,s
   and for vertical datagaps ... */

   for (j = start+1; j < nobs; ++j) {
      if (s[j] < -8. || t[j] < -8. || p[j] < -8) {
         h[j] = -999.;
         continue;
      }

      if (last_p < 1001.)
        datagap = (p[j] - last_p) > 200;
      else
        datagap = (p[j] - last_p) > 600;

      if (datagap) {
        for (i= j; i < nobs; ++i) {
          h[i] = -999.;
        }
/*        fprintf(stderr," gap: %lf %lf\n", last_p, p[j]);  */
        return;
      }

      sva1 = svan_(&s[j], &t[j], &p[j], &sig);

/* the 1e-5 term corrects the units : 10^-8  * 10^4 * 10^-1 
                                      (svan)   Pa/db   dyn meters   */

         h[j] = last_h + ((sva0 + sva1) *.5e-5 * (p[j] - last_p));
         last_h = h[j];
         last_p = p[j];
         sva0 = sva1;
   }
   return;
      
} /* end compute_height() */

/***************************************************************************/
void compute_theta(n, th, p, t, s)
int n;   /* # of levels in arrays */
double *th;    /* array of theta values */
double *p;     /*  pressure */
double *t;     /* in situ temperature */
double *s;     /* salinity */
{
   int i;
   double theta_();   /* function in phyprops.f */
   double pref = 0.0;

   for (i = 0; i < n; ++i) {
       th[i] = theta_(&s[i], &t[i], &p[i], &pref);
   }
   return;
}

/***************************************************************************/
double buoyancy(p0, p, t, s, nobs, pr_int)
double p0;         /* pressure level at which buoyancy is computed */
double *p, *t, *s; /* observed pressure, temperature, salinity values */
int nobs;          /* number of observations in p,t,s arrays */
float pr_int;      /* size of pressure interval (db) on either side of p0  
                      to use in estimating t,s gradients*/

/* Computes buoyancy frequency in radians/sec using gradients of 
   t and s near the pressure designated by p0.  The t,s gradients are 
   estimated by linearly interpolating t and s as a function of p from 
   the nearest observations which bracket p0.  If the p0 specified does 
   not exist in the pressure series represented by p, the value -9999. is
   returned.
*/
{
   double *p_win, *t_win, *s_win;
   double bv, e, interp(), bvgrdts_(), bvfrq_();
   double cph2rps = 0.001745329;   /* (2*pi)/3600 */
   int npts_win=3;
   int i;

   p_win = (double *) malloc(npts_win * sizeof(double));
   t_win = (double *) malloc(npts_win * sizeof(double));
   s_win = (double *) malloc(npts_win * sizeof(double));

/* determine whether specified pressure level exists at this station */

   if (interp(p0, p, t, nobs) < -9990.) {
           return (-9999.);
   }

/* set up the window of pressure around the specifed pressure level... */

   p_win[0] = p0 - (double)pr_int;
   p_win[1] = p0;
   p_win[2] = p0 + (double)pr_int;

   for (i = 0; i < npts_win; ++i) {
      t_win[i] = interp(p_win[i], p, t, nobs);
      s_win[i] = interp(p_win[i], p, s, nobs);
   }

/* account for cases where window juts above top pressure or below bottom
   pressure ... i is the index of the first viable pressure in window*/

   i = 0;
   if (t_win[0] < -9990. || t_win[2] < -9990.) {
      npts_win = 2;
      if (t_win[0] < -9990.) {
         i = 1;
      }
   }

/*   bv = bvgrdts_(&p0, &s_win[i], &t_win[i], &p_win[i], &npts_win, &e);  */
   bv = bvfrq_(&s_win[i], &t_win[i], &p_win[i], &npts_win, &p0, &e);

   return (bv * cph2rps);

} /* end buoyancy() */

/****************************************************************************/
double potvort( e, lat)
double e;    /* n-squared (buoyancy freq) in rad/sec^2  */
double lat;  /* latitude in degrees */
/* Returns potential vorticity for the specified value of stability 
   parameter (n-squared) and latitude. */
{
   double coriol_();   /* Coriolis parameter (m/sec^-2) */
   double gravity_();  /* acceleration due to gravity (cm/sec^2) */

   if (e < -9990.)       /* check for missing value flag ... */
       return (-9999.);

    return (e * 1.0e14  * coriol_(&lat) / gravity_(&lat) );

} /* end potvort() */

/****************************************************************************/
double interp(xval, x, y, nypts)
double xval, *x, *y;
int    nypts;
/* Performs a linear interpolation to find the position of xval in array, x,
   and returns the corresponding value in the array, y.  If xval does not
   appear in array x, the value -9999 is returned.  This routine assumes that
   the x array is monotonic and continuous (no missing values); and it
   assumes the y array is continuous.    */

{
   int    k;
   double v1, v2;

   for (k = 0; k < nypts-1; ++k) {

      v1 = xval - x[k];
      v2 = xval - x[k+1];

      if (v1 == 0)             /* x[k] == xval */
          return (y[k]);
      if (v2 == 0)             /* x[k+1] == xval */
          return (y[k+1]);
      if (v1 < 0. && v2 < 0.)  /* xval not between x1 and x2 */  
          continue;
      if (v1 > 0. && v2 > 0.) 
          continue;

      return ( y[k] + (y[k+1] - y[k]) * v1 / (x[k+1] - x[k]) );
   }

   return (-9999.0);

}   /* end interp() */


/****************************************************************************/
