/*  propcalc.c

____________________________________________________________________________
  USAGE:  

 propcalc filename(_roots) -P<property_list> [-D<dirname>] [-E<file_extent>]

  list of filenames MUST be first argument!

 -P : list of properties to evaluate; ex: -Ppr/te/th/sa/ox
      -P (by itself) will print a list of the available properties;

 -D : specifies directory for input files (default is current dir)  
      ex: -D/home/pete/data

 -E : specifies input_file extent (default is no extent) ex: -E.sum

____________________________________________________________________________
propcalc computes properties at each standard level in a station and 
outputs a (hydrobase format) ascii station file containing all the properties
specified with the -P option.                                                    ____________________________________________________________________________
*/


#include <stdio.h>
#include <math.h>
#include <fcntl.h>
#include <string.h>
#include <malloc.h>
#include "hydrobase.h"

#define   NINT(x)	((int)((x) + 0.5))
#define    ABS(x)       (((x) < 0) ? -(x) : (x)) 

#define   STDOUT   1          

/* input file pathnames */

#define    EXTENT   ""
#define    DIR      ""

    /* flags to indicate which properties are to be output and which are
       needed for computation */
      
short prop_req[MAXPROP];         /* set of props requested for output */
short prop_needed[MAXPROP];      /* set of props requested || defining a
                                    surface || needed for computation */

   /* global variables to store station ... */

struct HYDRO_DATA station;
struct HYDRO_HDR hdr;
double s_pref;           /* ref lev for computing a non-standard sigma level */

     /* in prop_subs.c */

extern int   get_prop_indx();               
extern char *get_prop_descrip(), *get_prop_mne();
extern void  print_prop_menu();
extern void  compute_sigma(), compute_height(); 
extern void  compute_theta();
extern double buoyancy(), potvort();

     /* in hydro_utils.c */

extern int open_hydro_file();           
extern int get_station();
extern int write_hydro_station();
extern void report_status();

main (argc, argv)
int   argc;
char *argv[];
{
   short   popt;
   int     index, nprops = 0;
   int     i;
   int     curfile = 1, nfiles = 0; 
   int     print_msg = 1;       /* print message in file open routines*/
   int     error, prompt;
   char    *extent, *dir, *st;
   int     infile;
   void    print_usage();
   void    get_hydro_data();
   int     parse_prop_list();


/* are there command line arguments? */

   if (argc < 2) {
      print_usage(argv[0]);
      exit(1);
   }
 
/*  set these default values */

    dir = DIR;
    extent = EXTENT;
    popt = 0;
    error = 0;
    s_pref = -1;

/* initialize these ... */

   for (i = 0; i < MAXPROP; ++i) {
       prop_req[i] = 0;
       prop_needed[i] = 0;
   }



/*  parse the command line arguments */

   for (i = 1; i < argc; i++) {
      if (argv[i][0] == '-') {
            switch (argv[i][1]) {
               case 'D':                   /* get input dir */
                        dir = &argv[i][2];
                        break;
               case 'E':                    /* get file extent */
                        extent = &argv[i][2];
                        break;

               case 'P':
                        popt = 1;
                        if ( argv[i][2] == '\0') {
                               print_prop_menu();
                               exit(0);
                        }
                        nprops = parse_prop_list(&argv[i][2]);
                        if (nprops <= 0)
                             error = 1;
                        break;

               default:
                        error = 1;

          }    /* end switch */

          if (error ) {
             fprintf(stderr,"\nError parsing command line args.\n");
             fprintf(stderr,"     in particular: '%s'\n", argv[i]);
             exit(1);
          }

       }  /* end if */

       else  {
           ++nfiles;
       }
   }  /* end for */

   if ( !nfiles || !popt) {
       fprintf(stderr,"\nYou must specify input file(s) and properties.\n");
       exit(1);
   }



/* loop for each input file */

   do {

       infile = open_hydro_file(dir, argv[curfile], extent, print_msg);
       if (infile < 0)
          goto NEXTFILE;

     
            /* read each file completely */

 
         get_hydro_data(infile);

NEXTFILE:

         close(infile);

   } while (curfile++ < nfiles );

   fprintf(stderr,"\n\nEnd of propcalc.\n");
   exit(0);

}   /* end main */

/****************************************************************************/

void print_usage(program)
char *program;
{
   fprintf(stderr,"\nUsage:  %s filename_root(s)  -P<list_of_properties>  [-D<dirname>] [-E<file_extent>]", program);

   fprintf(stderr,"\n\n  List of filenames MUST be first argument!");
   fprintf(stderr,"\n    -P  : list of properties to project onto surface;");
   fprintf(stderr,"\n          ex:  -Ppr/th/sa/ox/ht");
   fprintf(stderr,"\n               -P (by itself) produces a list of available properties");
   fprintf(stderr,"\n    -D  : specifies dirname for input files (default is current directory) ");
   fprintf(stderr,"\n            ex: -D../data/ ");
   fprintf(stderr,"\n    -E  : specifies input_file extent (default is no extent)");  
   fprintf(stderr,"\n            ex: -E.dat ");

   fprintf(stderr,"\n\n");  
   return;
}

/*****************************************************************************/
int parse_prop_list(st)
char *st;   /* the list with -P stripped off */
    /*  Parses a list of property mnemonics and sets global flags accordingly.
        Returns the number of properties.  An error will cause an exit. */
{
   int index, nprops;
   char prop[4];
   double ref_val;

   nprops = 0;

   do {
      if (*st == '/')
         ++st;
      sscanf(st,"%2s", prop);
      index = get_prop_indx(prop);
      if (index < 0) {
         fprintf(stderr,"\n Unknown property requested: %s\n", prop);
         exit(1);
      }
      prop_req[index] = 1;
      prop_needed[index] = 1;
      ++nprops;
      ++st;
      ++st;

  /* some special cases ... */

   /* s_ */
      if ((enum property)index == S_ ) {
         if (sscanf(st, "%lf", &ref_val) != 1) {
             fprintf(stderr,"\n Specify a ref pressure when requesting the property 's_'");
             fprintf(stderr,"\n   ex: -Ps_1500/th/sa\n");
             exit(1);
         }
         while (!(*st=='/' || *st=='\0' || *st==' '))
            ++st;

         if (s_pref >= 0) {   /* has this already been requested? */
                     if ( NINT( s_pref) != NINT(ref_val)) {
                         fprintf(stderr,"Sorry. You have requested the property s_ with multiple reference levels.\n");
                         fprintf(stderr,"You can only use one of those prefs");
                         fprintf(stderr,"  You will probably have to do multiple runs for each different pref you want to associate with s_\n\n");
                        exit(1);
                     }
          }
          s_pref = ref_val;
      }

   /* sf... stream function gets computed in an external
      program:  just output pr, sv, and ht for now.
   */

      if ((enum property)index == SF ) {
                         
          prop_req[index] = 0;     /* reset these */
          prop_needed[index] = 0;
          fprintf(stderr,"\nSorry, stream function not supported.\n");
      }
   /* end of special cases */

  } while (*st == '/');

  return (nprops);
}  /* end parse_prop_list() */
 

/*****************************************************************************/
void get_hydro_data(file)
int file;
   /*  Reads each station in a HydroBase file and computes property values
       at each standard level.  This module requires that the HydroBase
       file contains a minimum of pr, de, te, sa observations.  */
{
   int error, i, j, nreq;
   double dlat, deltap;
   void  free_and_alloc();


/* read each station in file ... */

    while ((error = get_station(file, &hdr, &station)) == 0) {

  /* ensure that pr, de, te, and sa are available ... */

       if (!(available(PR, hdr) && available(DE, hdr) && available(TE, hdr) 
           && available(SA, hdr))) {
         fprintf(stderr,"\n>>>>> WARNING!!!  ");
         fprintf(stderr,"Station does not include pr, de, te, and sa.");
         break;
       }


 /* compute appropriate properties at each level in station ... */

/* !**! Special cases for individual properties... */

    for (i = 0; i < MAXPROP; ++i) {
       if (prop_needed[i]) {
          switch ((enum property) i) {
             case TH:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_theta(hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA] );
               break;
             case S0:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sigma(0., hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;
             case S1:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sigma(1000., hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;
             case S2:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sigma(2000., hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;
             case S3:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sigma(3000., hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;
             case S4:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sigma(4000., hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;
             case S_:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sigma(s_pref, hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;
             case HT:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_height(hdr.nobs, station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA], station.observ[i]);
               break;
             case SV:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_sp_vol( hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;

             case VA:
               free_and_alloc(&station.observ[i], hdr.nobs);
               compute_svan( hdr.nobs, station.observ[i], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA]);
               break;

             case PV: 
               free_and_alloc(&station.observ[i], hdr.nobs);
               deltap = 50;   /* pressure window in db.*/
               dlat = (double) hdr.lat;
               for (j = 0; j < hdr.nobs; ++j) {
                    station.observ[i][j] = buoyancy(station.observ[(int)PR][j], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA], hdr.nobs, deltap);
                    if (station.observ[i][j] < -9998.)  break;
                    station.observ[i][j] = potvort((station.observ[i][j] *station.observ[i][j]), dlat);
               }

               break;
             case BF:
               free_and_alloc(&station.observ[i], hdr.nobs);
               deltap = 50;  /* pressure window in db. */
               for (j = 0; j < hdr.nobs; ++j) {
                    station.observ[i][j] = buoyancy(station.observ[(int)PR][j], station.observ[(int)PR], station.observ[(int)TE],station.observ[(int)SA], hdr.nobs, deltap);
                    if (station.observ[i][j] > -9998.)   /* convert the units */
                        station.observ[i][j] *= 1.0e5;
               }
               break;

             default:
               break;
          } /* end switch */
       } /* end if */
     } /* end for */
     
 /* determine availability of requested properties... */
    nreq = 0;
    for (i = 0; i < MAXPROP; ++i) {
       if (prop_req[i] && station.observ[i] != NULL )
          ++nreq;
    }
    
 /* output the station */
    hdr.nprops = station.nprops = nreq;
    free(hdr.prop_id);
    hdr.prop_id = (int *) malloc(nreq * sizeof(int));
    j = 0;
    for (i = 0; i < MAXPROP; ++i) {
       if (! prop_req[i]) {
           free(station.observ[i]);
           station.observ[i] = NULL;
       }
       else if (station.observ[i] != NULL)  {
         hdr.prop_id[j++] = i;
       }
    }
    write_hydro_station(STDOUT, hdr, station);

    for (i = 0; i < MAXPROP; ++i) {
       if (station.observ[i] != NULL) {
          free(station.observ[i]);
          station.observ[i] = NULL;
       }
    }    
   }  /* end while */

   if (error > 0)
      report_status(error, stderr);

   return;

} /* end get_hydro_data() */

/****************************************************************************/

void free_and_alloc(dptr, n)
double **dptr;   
int n;           /* number of elements to allocate */

   /* Frees up the space pointed to by dptr.  It MUST have been allocated
      using malloc()!!  Then mallocs space for n double values and sets
      dptr pointing to it.  If insufficient memory, a message is written to
      stderr and an exit() is made. */
{

   if (*dptr != NULL) {
      free((char *)*dptr);
      *dptr = NULL;
   }

   *dptr = (double *) malloc(n * sizeof(double));

   if (*dptr == NULL) {
      fprintf(stderr, "\nInsufficient memory for call to malloc()\n");
      exit(1);
   }

   return;
}  /* end free_and_alloc() */
/****************************************************************************/
   
