/*  smoothsurf.c
_______________________________________________________________________________

   USAGE:  smoothsurf -I<input_filename> -O<output_file_root> -P<property_list>
            -M<minobs> -R<search_radius> [-N] [-S] [-B<blanking_info_file>]
_______________________________________________________________________________
.  Smooths a matrix of points containing mean and stddev values gridded on 
.  an m * n mesh .  If the total # of observations < minobs (a user specifed 
.  value) each point lying one square further out is incorporated, weighted 
.  not only by its # of obs, but also by a distance factor = 1/(d*d), where
.  d = # of gridpts away from the point being calculated.  The user specifies 
.  how far out (radius of influence) to search to find minobs # of observations.
.
.  The program outputs up to 4 files for each property specified: 
.     1) property info:   lon, lat, av_value [, radius, n]
.     2) blank gridpts:   lon, lat
.     3) stddev info  :   lon, lat, stddev [, radius, n]
.     4) stddev blanks:   lon, lat
.  The stddev information is only output if the -S option is specified.
.
.   Output files are named:     <root><property_id>.xyz
.                                  "       "       .blk
.                                  "       "       _var.xyz
.                                  "       "       _var.blk
.   where root is supplied by the -O option and property_id is the same as
.   the property_list_ids in the -P option.
.
.   The -B option permits you to supply a file containing the lon,lat of
.   gridpoints to be blanked.  Its purpose is to mask out bathymetry,
.   coastlines, etc...  This list is checked before computing each new
.   gridpoint.

*/


#include <stdio.h>
#include <math.h>
#include <string.h>
#include "hydrobase.h"


#define IMASK  0x01     /* masks for option flags */
#define OMASK  0x02
#define PMASK  0x04
#define RMASK  0x08
#define MMASK  0x10
#define ALL    0x1f     /* bitwise OR of all above masks */

#define  ABS(x)   (((x) < 0) ? -(x) : (x))      /* macro */

/* data structures to represent matrix */

struct old_grid_pt {
       float mean;
      double var;
         int n;
 };

struct new_grid_pt {
       float sum;
      double sumsq;
         int n;      
       float wghts;
};


void    add_wghted_layer();

extern int    get_prop_indx(); 
extern char  *get_prop_mne();
extern void   print_prop_menu();

void main(argc, argv)
int   argc;
char *argv[];
{
   int    row, col, n, i, j, sq;
   int    rad,  maxrad, minobs;
   int    ncols, nrows;
   int    gridsize;
   int    error, index, nprops;
   int    prop_index[MAXPROP];     
   char   prop_request[MAXPROP], prop_avail[MAXPROP];                      
   char   optflag = 0, blankopt = 0, nflag = 0, stddev_flag = 0;
   char   *id, field_descrip[12];
   double var, mean, stddev;
   float  lat, lon;
   float  xmin, ymin,xmax, ymax, xspacing, yspacing;
   FILE  *infile, *outfile[4], *blankfile;
   char  *fname, *root, *s;
   char  *blank;
   struct old_grid_pt *gridpt, *old_matrix[MAXPROP];
   struct new_grid_pt *newpt, *new_matrix[MAXPROP];
   void  print_usage();

/* check for command line arguments */

   if (argc < 2) {
     print_usage(argv[0]);
     exit(1);
   }

/* initialize flags */

    for (i = 0; i < MAXPROP; ++i) {
        prop_request[i] = prop_avail[i] = 0;
    }
    id = (char *) malloc(3);

/*  parse the command line arguments */

   for (i = 1; i < argc; i++) {
      error =  (argv[i][0] == '-') ? 0 : 1;
      if (!error) {
            switch (argv[i][1]) {
               case 'I':                   /* get input file */
                        optflag = optflag | IMASK;
                        fname = &argv[i][2];
                        break;
               case 'O':                    /* get output file root_name */
                        optflag = optflag | OMASK;
                        root = &argv[i][2];
                        break;

               case 'P':                    /* get list of properties */
                        optflag = optflag | PMASK;
                        s = &argv[i][2];
                        if (*s == '\0') {
                               print_prop_menu();
                               exit(0);
                        }
                        do {
                         if (*s == '/')
                               ++s;
                         sscanf(s,"%2s", id);
                         index = get_prop_indx(id);
                         if (error = (index < 0) ? 1 : 0)
                            break;
                         prop_request[index] = '1';
                         ++s;
                         ++s;
                        } while (*s == '/');
                        break;

               case 'M':
                       optflag = optflag | MMASK;
                       error = (sscanf(&argv[i][2],"%d", &minobs) == 1) ? 0 : 1;
                       break;

               case 'N':
                       nflag = 1;
                       break;

               case 'R':
                       optflag = optflag | RMASK;
                       error = (sscanf(&argv[i][2],"%d", &maxrad) == 1) ? 0 : 1;
                       break;

               case 'S':
                       stddev_flag = '1';
                       break;

               case 'B':
                       blankopt = '1';
                       if ((blankfile = fopen(&argv[i][2], "r")) == NULL) {
                           error = 1;
                           fprintf(stderr,"\nError opening %s\n", &argv[i][2]);
                       }
                       break;
               default:
                       error = 1;

          }    /* end switch */
       }  /* end if */


       if (error ) {
             print_usage(argv[0]);
             fprintf(stderr,"\nError parsing command line args.\n");
             fprintf(stderr,"     in particular: '%s'\n", argv[i]);
             exit(1);
       }


   }  /* end for */

   if (optflag != ALL) {
       print_usage(argv[0]);
       fprintf(stderr,"\nYou must specify -I-O-P-M-R options!\n");
       exit(1);
   }



/*  open input file and get row/column dimensions */
 
   if ((infile = fopen(fname, "r")) == NULL) {
        fprintf(stderr, "\n Unable to open %s for input\n\n", fname);
        exit(1);
   }
   fprintf(stderr, "\nOpened %s ... \n", fname);
   
   if (fscanf(infile,"%d%d%f%f%f%f%f%f%d", &nrows, &ncols, &xspacing, &yspacing, &xmin,
      &ymin, &xmax, &ymax, &nprops )  != 9) {
        fprintf(stderr,"\nError reading heading from infile\n\n");
        exit(1);
   }
   fprintf(stderr, "\nInput grid is %d rows by %d cols", nrows, ncols);

   gridsize = nrows * ncols;



   fprintf(stderr,"\n Properties available: ");
   for (i = 0; i < nprops; ++i) {
       fscanf(infile,"%2s", id);
       prop_index[i] = get_prop_indx(id);
       prop_avail[prop_index[i]] = '1';
       fprintf(stderr," %2s", id);
   }
   fprintf(stderr,"\n");

/* check that requested properties are available and allocate space for matrices */

   for (i = 0; i < MAXPROP; ++i ) {
       if (prop_request[i] && !prop_avail[i]) {
          fprintf(stderr,"\nRequested property: %2s not available.", get_prop_mne(i));
          prop_request[i] = 0;
       }

       new_matrix[i] = NULL;
       if (prop_request[i]) {
          new_matrix[i] = (struct new_grid_pt *) malloc(sizeof(struct new_grid_pt) * gridsize);
          if (new_matrix[i] == NULL) {
             fprintf(stderr,"\nUnable to allocate memory for new matrix.");
             exit(1);
          }

          for (sq= 0; sq < gridsize; ++sq) {
               newpt = &new_matrix[i][sq];
               newpt->sum = 0.;
               newpt->sumsq = 0.;
               newpt->n = 0;
               newpt->wghts = 0.;
          }
       }

       old_matrix[i] = NULL;
       if (prop_avail[i]) {
          old_matrix[i] = (struct old_grid_pt *) malloc(sizeof(struct old_grid_pt) * gridsize);
          if (old_matrix[i] == NULL) {
             fprintf(stderr,"\nUnable to allocate memory for old matrix.");
             exit(1);
          }
          for (sq= 0; sq < gridsize; ++sq) {
               gridpt = &old_matrix[i][sq];
               gridpt->mean = 0.;
               gridpt->var = 0.;
               gridpt->n = 0;
          }
       } 
   }

/* allocate and initialize blanking matrix */

   blank = (char *) malloc(gridsize);
   if (blank == NULL) {
       fprintf(stderr,"\nUnable to allocate memory for blanking matrix.");
       exit(1);
   }

   for (sq = 0; sq < gridsize; ++sq) {
         blank[sq] = 0;
   }
   if (blankopt) {
      while (fscanf(blankfile,"%f %f", &lon, &lat) != EOF) {
         row = (int) (.0001 + (lat - ymin) / yspacing);
         col = (int) (.0001 + (lon - xmin) / xspacing);
          /* check that it is within the bounds */
         if ((row >= 0) && (row < nrows) && (col >= 0) && (col < ncols)) {
            blank[row * ncols + col] = '1';
         }
      } 
     
   }



/*  read data from infile and store in matrices by row, col.  Multiply means
    by nobs to produce raw sums ; square the std dev before multiplying by
    (nobs - 1) and adding (sumx * sumx / n) to produce the raw sumofsquares */

   fprintf(stderr,"\nReading input file...\n");

   while ( fscanf(infile, "%f%f",  &lat, &lon) != EOF) {
     row = (int) (.0001 + (lat - ymin) / yspacing);
     col = (int) (.0001 + (lon - xmin) / xspacing);

     for (i = 0; i < nprops; ++i) {
        if (fscanf( infile,"%lf%lf%d", &mean, &var, &n) != 3) {
           fprintf(stderr,"\nError reading input file at ");
           fprintf(stderr,"values of (row, col) = ( %d , %d)\n", row, col);
        }

        if (n > 0) {                   /* compute raw sums */
         mean *= n;
         var *= var * (n - 1);
         var +=  mean * mean / n;
        }

        j = prop_index[i];
        sq = row * ncols + col;       /* find index to old_matrix */
        gridpt = &old_matrix[j][sq];  /* get addr of gridpoint */
        gridpt->mean = mean;          /* store info at that addr */
        gridpt->var = var;
        gridpt->n = n;
     }

   }  /* end while */

   

/*  for each pt in the new grid, sum the values from surrounding squares
    in old grid, weighted by distance. Move successively outward until 
    either minobs is achieved or max search radius is reached.  
    Compute means and standard deviations from weighted sums   */

   fprintf(stderr,"\nComputing new matrix ...\n");
   fname = (char *)malloc(50);
   free(id);

   for (i = 0; i < MAXPROP; ++i) {
     if (prop_request[i]) {

        id = get_prop_mne(i);    
        sprintf(field_descrip, " %c%d.%dlf", '%',get_field_width(i), get_field_precis(i)); 
        fprintf(stderr,"\n%s ", id);        /* print prop id on stderr */

        fname = strcpy(fname,root);
        fname = strncat(fname, id, 2);
        fname = strncat(fname, ".xyz", 5);
        outfile[0] = fopen(fname, "w");       /* open output file for prop */

        fname = strcpy(fname,root);
        fname = strncat(fname, id, 2);
        fname = strncat(fname, ".blk", 4);
        outfile[1] = fopen(fname, "w");       /* open blanking file for prop */

        if (stddev_flag) {
           fname = strcpy(fname,root);
           fname = strncat(fname, id, 2);
           fname = strncat(fname, "_var.xyz", 9);
           outfile[2] = fopen(fname, "w");    /* open stddev file for prop */

           fname = strcpy(fname,root);
           fname = strncat(fname, id, 2);
           fname = strncat(fname, "_var.blk", 8);
           outfile[3] = fopen(fname, "w");    /* open stddev blanking file  */
        }

        sq = -1;
        for (row=0; row < nrows; ++row) {
           fprintf(stderr,"*");
           for (col=0; col < ncols; ++col) {
              ++sq;
              lat = ymin + (row + .5) * yspacing;
              lon = xmin + (col + .5) * xspacing;

              if (blank[sq]) {       /* automatically blank this gridpt? */

                    fprintf(outfile[1],"%.3f %.3f\n", lon, lat);
                    if (stddev_flag) {
                       fprintf(outfile[3],"%.3f %.3f\n", lon, lat);
                    }                  
              }
              else {                 /* smooth this gridpt */
                 rad = 0;
                /* initialize with old matrix values */
                 newpt = &new_matrix[i][sq];
                 gridpt = &old_matrix[i][sq];
                 newpt->n = gridpt->n;
                 newpt->sum = gridpt->mean;
                 newpt->sumsq = gridpt->var;
                 newpt->wghts = gridpt->n;

                 while ((newpt->n < minobs) && (rad < maxrad)) {
                   ++rad;
                   add_wghted_layer(rad, newpt, row, col,
                                    old_matrix[i], nrows, ncols);
                 }
             
                 if ((n = newpt->n) == 0) {       /* write to blanking files */
                    fprintf(outfile[1],"%.3f %.3f\n", lon, lat);
                    if (stddev_flag) {
                       fprintf(outfile[3],"%.3f %.3f\n", lon, lat);
                    }
                 }
                 else {                     /* write to property file */
                    mean = newpt->sum / newpt->wghts;
                    fprintf(outfile[0],"%.3f %.3f ", lon, lat); 
                    fprintf(outfile[0], field_descrip, mean);
                    if (nflag) {
                         fprintf(outfile[0]," %2d %4d", rad, n);
                    }
                    fprintf(outfile[0],"\n");

                    if (stddev_flag) {         
                       stddev = 0.0;
                       if (n > 1) {         /* write to stddev file */
                          var = (newpt->sumsq - newpt->sum * newpt->sum /
                              (double) newpt->wghts ) / (double) newpt->wghts;
                          stddev = sqrt(ABS(var));
                          fprintf(outfile[2],"%.3f %.3f %.7lf", lon, lat, stddev);
                          if (nflag) {
                             fprintf(outfile[2]," %2d %4d", rad, n);
                           }
                          fprintf(outfile[2],"\n");
                       }
                       else {         /* write to stddev blanking file */
                          fprintf(outfile[3],"%.3f %.3f\n", lon, lat);
                       }
                    }

                 } /* end else */
              } /* end else */
           } /* end for */

        } /* end for */

       fclose(outfile[0]);
       fclose(outfile[1]);
       if (stddev_flag) {         
         fclose(outfile[2]);
         fclose(outfile[3]);
       }

     } /* end if */
   }  /* end for */

   fprintf(stderr,"\nEnd of smoothsurf.\n");
   exit(0);
} /* end main */




/****************************************************************************/

void print_usage(program)
char *program;
{
   fprintf(stderr,"\nUsage:  %s -I<input_filename> -O<output_file_root> -P<property_list> -M<minobs> -R<search_radius>  [-N] [-S] [-B<blanking_info_file>", program);
   fprintf(stderr,"\n\n    -I  : specifies input file_name");
   fprintf(stderr,"\n    -O  : specifies root name for output files");
   fprintf(stderr,"\n    -P  : specifies properties to project onto surface");
   fprintf(stderr,"\n          Enter a list of your choices (separated by slashes)...");
   fprintf(stderr,"\n     ex: -Ppr/th/sa/ox/de/ht");
   fprintf(stderr,"\n         -P (by itself) will print a menu of available properties");
   fprintf(stderr,"\n    -M  : minimum # of observations desired per gridpt ");
   fprintf(stderr,"\n            ex: -M10 ");
   fprintf(stderr,"\n    -R  : maximum search radius (# of squares away) to go");
   fprintf(stderr,"\n          to find minobs to compute each gridpt: ");
   fprintf(stderr,"\n            ex: -R3 ");
   fprintf(stderr,"\n    -N  : output number of obs and radius of smoothing info to each file.");
   fprintf(stderr,"\n    -S  : output std dev file for each property.");
   fprintf(stderr,"\n    -B  : specifies name of file with lon/lat of pts");
   fprintf(stderr,"\n          to be blanked.  This list is checked BEFORE");
   fprintf(stderr,"\n          computing each new gridpoint.");

   fprintf(stderr,"\n\n");  
   return;
}



/* 
_____________________________________________________________________________

 add_wghted_layer() identifies pts in an existing matrix (size: nrows * ncols)
    that are radius # of points away from a point centered at (i,j)  The values 
    of each property at each point thus identified,
    are weighted by the inverse of radius**2,
    and added to the appropriate gridpt in the new matrix.  The same
    is done for the squared sums and the number of pts. 
  
______________________________________________________________________________
*/

void add_wghted_layer(radius, gridpt, i, j, oldprop_ptr, nrows, ncols)
int radius;                    /* search radius  (in # of gridpts )*/
struct new_grid_pt *gridpt;    /*  address of gridpt  */
int i, j;                      /* row,col of gridpt in matrix  */
struct old_grid_pt *oldprop_ptr; /* starting addr of prop in old matrix */
int nrows, ncols;              /* dimensions of  matrix */
{
   int top, bot, left, right;
   int row, col, layer, side;
   float  weight;
   struct old_grid_pt *oldgridpt;
   
   if (radius <= 0) 
         return ;

/*.........................  WEIGHT FUNCTION  ............................. */

     /* define weight for values in this layer */

   weight = 1. / (radius * radius);
/*......................................................................... */

/* define row,col of lower left corner of first layer */

   row = i;   
   col = j;
   layer = 1;


/* move to lower left corner of layer corresponding to radius */

   while (layer++ < radius) {   
     --row;
     --col;
   }  /* end while */

/* side of square defined by locus of pts in layer  */

   side = 2 * radius + 1; 
 
/* define corners of square */

   bot = row;
   left = col;
   top = row + side - 1;
   right = col + side - 1;

/* move up left side of square */

   if ((col >= 0) && (col < ncols)) {     /* within column bounds of matrix? */
      while ( row < top) {
         if ((row >= 0) && (row < nrows)) {      /* within row bounds? */
           oldgridpt = &oldprop_ptr[row * ncols + col];
           if (oldgridpt->n > 0) {             /* any obs at this pt? */
             gridpt->sum += weight * oldgridpt->mean;
             gridpt->sumsq += weight * oldgridpt->var;
             gridpt->n += oldgridpt->n;
             gridpt->wghts += weight * oldgridpt->n;
           }
         }
         ++row;
      } /* end while */
   }  /* end if */

/* set row, col to point to top, left corner */

   row = top;
   col = left;

/*    move across top of square */

   if ((row >= 0) && (row < nrows)) {     /* within row bounds? */
      while ( col < right) {
         if ((col >= 0) && (col < ncols)) {      /* within column bounds? */
           oldgridpt = &oldprop_ptr[row * ncols + col];
           if (oldgridpt->n > 0) {             /* any obs at this pt? */
             gridpt->sum += weight * oldgridpt->mean;
             gridpt->sumsq += weight * oldgridpt->var;
             gridpt->n += oldgridpt->n;
             gridpt->wghts += weight * oldgridpt->n;
           }
         }
         ++col;
      } /* end while */
   }  /* end if */


/* set row, col to point to top, right corner */

   row = top;
   col = right;

/*    move down right side of square */

   if ((col >= 0) && (col < ncols)) {     /* within col bounds? */
      while ( row > bot) {
         if ((row >= 0) && (row < nrows)) {      /* within row bounds? */
           oldgridpt = &oldprop_ptr[row * ncols + col];
           if (oldgridpt->n > 0) {             /* any obs at this pt? */
             gridpt->sum += weight * oldgridpt->mean;
             gridpt->sumsq += weight * oldgridpt->var;
             gridpt->n += oldgridpt->n;
             gridpt->wghts += weight * oldgridpt->n;
           }
         }
         --row;
      } /* end while */
   }  /* end if */

/* set row, col to point to bottom, right corner */

   row = bot;
   col = right;

/*    move across bottom of square */

   if ((row >= 0) && (row < nrows)) {     /* within row bounds? */
      while ( col > left) {
         if ((col >= 0) && (col < ncols)) {      /* within col bounds? */
           oldgridpt = &oldprop_ptr[row * ncols + col];
           if (oldgridpt->n > 0) {             /* any obs at this pt? */
             gridpt->sum += weight * oldgridpt->mean;
             gridpt->sumsq += weight * oldgridpt->var;
             gridpt->n += oldgridpt->n;
             gridpt->wghts += weight * oldgridpt->n;
           }
         }
         --col;
      } /* end while */
   }  /* end if */

   return;
} /* end add_wghted_layer() */



