      subroutine get_data (iout,time,taux,tauy,u10m,cldc,airt,relh,sst,
     &                     pres,prcp)
c
c=======================================================================
c                                                                    ===
c  This routine reads one set of atmospheric data.                   ===
c                                                                    ===
c  ------                                                            ===
c  Input:                                                            ===
c  ------                                                            ===
c                                                                    ===
c     IOUT   Unit number for diagnostic output.    (integer)         ===
c                                                                    ===
c  -------                                                           ===
c  Output:                                                           ===
c  -------                                                           ===
c                                                                    ===
c     TIME   Time of data set.          (real; modified Julian day)  ===
c     TAUX   Zonal wind stress.         (real array; dynes/cm^2)     ===
c     TAUY   Meridional wind stress.    (real array; dynes/cm^2)     ===
c     U10M   10 meter wind speed.       (real array; m/s)            ===
c     CLDC   Fractional cloud cover.    (real array)                 ===
c     AIRT   Air temperature.           (real array; celsius)        ===
c     RELH   Relative humidity.         (real array)                 ===
c     SST    Sea surface temperature.   (real array; celsius)        ===
c     PRES   Surface pressure.          (real array; millibar)       ===
c     PRCP   Precipitation.             (real array; cm/day)         ===
c                                                                    ===
c  ------                                                            ===
c  Calls:                                                            ===
c  ------                                                            ===
c                                                                    ===
c     CLOUD_SCALE,  LENGTH,     MAXABS,      READ_FNMOC,  SCALE_FLD, ===
c     SETFMT,       SHIFT_FLD,  WINDSTRESS                           ===
#ifdef fixlims
c     IMPMXMN                                                        ===
#endif
c                                                                    ===
c=======================================================================
c
c-----------------------------------------------------------------------
c  Define global data.
c-----------------------------------------------------------------------
c
#include <param.h>
c
c-----------------------------------------------------------------------
c  Define local data.
c-----------------------------------------------------------------------
#ifdef fixlims
c
      logical no_use,use
      real    c0,c1
      parameter (c0=0.0, c1=1.0, use=.true., no_use=.false.)
#endif
c
      integer idum,iout,sbn1,sbn2,sln1,sln2,snd1,snd2
      logical first
      real    fac,mairt,mcldc,mprcp,mpres,mrelh,msst,mu10m,time
      real mmairt,mmcldc,mmprcp,mmpres,mmrelh,mmsst,mmu10m
      real airt(nx,ny),cldc(nx,ny),prcp(nx,ny),pres(nx,ny),relh(nx,ny),
     &     sst(nx,ny),taux(nx,ny),tauy(nx,ny),u10m(nx,ny),velx(nx,ny),
     &            vely(nx,ny)
      character*8  fmtbit
      character*80 airtfile,cldcfile,prcpfile,presfile,relhfile,
     &              sstefile,windfile_u,windfile_v
      character*256 fmt
c
      save first,mmairt,mmcldc,mmprcp,mmpres,mmrelh,mmsst,mmu10m
c
      data first /.true./
c
c=======================================================================
c  Begin executable code
c=======================================================================
c
c-----------------------------------------------------------------------
c  Read time of data set.
c-----------------------------------------------------------------------
c
      read (5,*) idum
      read (5,*) time
c
      call setfmt (9,time,fmtbit)
      call length (fmtbit,sln2,sbn2,snd2)
      fmt = '('//fmtbit(sbn2:snd2)
c
c-----------------------------------------------------------------------
c  Read names of data files.
c-----------------------------------------------------------------------
c
      read (5,*)     idum
      read (5,'(a)') presfile
c
      read (5,*)     idum
      read (5,'(a)') airtfile
c
      read (5,*)     idum
      read (5,'(a)') prcpfile
c
      read (5,*)     idum
      read (5,'(a)') windfile_u
      read (5,'(a)') windfile_v
c
      read (5,*)     idum
      read (5,'(a)') sstefile
c
      read (5,*)     idum
      read (5,'(a)') relhfile
c
      read (5,*)     idum
      read (5,'(a)') cldcfile
c
c-----------------------------------------------------------------------
c  Get wind stress.
c-----------------------------------------------------------------------
c
c  Read wind velocity components.
c
      call read_fnmoc (velx,windfile_u)
      call read_fnmoc (vely,windfile_v)
c
c  Compute wind stress.
c
      call windstress (velx,vely,taux,tauy,u10m)
c
c  Get diagnostic statistics.
c
      call maxabs (nx,nx,ny,u10m,mu10m)
c
      call setfmt (9,mu10m,fmtbit)
      call length (fmt,sln1,sbn1,snd1)
      call length (fmtbit,sln2,sbn2,snd2)
      fmt = fmt(sbn1:snd1)//',1x,'//fmtbit(sbn2:snd2)
c
c-----------------------------------------------------------------------
c  Get cloud cover.
c-----------------------------------------------------------------------
c
c  Read cloud cover.
c
      call read_fnmoc (cldc,cldcfile)
c
c  Convert cloud cover to a fractional value.
c
      call cloud_scale (cldc,fac)
      call scale_fld (1,nx,1,ny,fac,cldc)
c
#ifdef fixlims
c  Impose maximum/minimum values.
c
      call impmxmn (1,nx,1,ny,use,c0,use,c1,cldc)
#endif
c  Get diagnostic statistics.
c
      call maxabs (nx,nx,ny,cldc,mcldc)
c
      call setfmt (9,mcldc,fmtbit)
      call length (fmt,sln1,sbn1,snd1)
      call length (fmtbit,sln2,sbn2,snd2)
      fmt = fmt(sbn1:snd1)//',1x,'//fmtbit(sbn2:snd2)
c
c-----------------------------------------------------------------------
c  Get air temperature.
c-----------------------------------------------------------------------
c
c  Read air temperature.
c
      call read_fnmoc (airt,airtfile)
c
c  Convert to celsius.
c
      call shift_fld (1,nx,1,ny,-273.15,airt)
c
c  Get diagnostic statistics.
c
      call maxabs (nx,nx,ny,airt,mairt)
c
      call setfmt (9,mairt,fmtbit)
      call length (fmt,sln1,sbn1,snd1)
      call length (fmtbit,sln2,sbn2,snd2)
      fmt = fmt(sbn1:snd1)//',1x,'//fmtbit(sbn2:snd2)
c
c-----------------------------------------------------------------------
c  Read relative humidity.
c-----------------------------------------------------------------------
c
c  Read relative humidity.
c
      call read_fnmoc (relh,relhfile)
c
c  Convert relative humidity to a fractional value.
c
#ifdef normcr
      call scale_fld (1,nx,1,ny,1.0,relh)
#else
      call scale_fld (1,nx,1,ny,0.01,relh)
#endif
c
#ifdef fixlims
c  Impose maximum/minimum values.
c
      call impmxmn (1,nx,1,ny,use,c0,use,c1,relh)
#endif
c  Get diagnostic statistics.
c
      call maxabs (nx,nx,ny,relh,mrelh)
c
      call setfmt (9,mrelh,fmtbit)
      call length (fmt,sln1,sbn1,snd1)
      call length (fmtbit,sln2,sbn2,snd2)
      fmt = fmt(sbn1:snd1)//',1x,'//fmtbit(sbn2:snd2)
c
c-----------------------------------------------------------------------
c  Get sea surface temperature.
c-----------------------------------------------------------------------
c
c  Read sea surface temperature.
c
      call read_fnmoc (sst,sstefile)
c
c  Convert to celsius.
c
      call shift_fld (1,nx,1,ny,-273.15,sst)
c
c  Get diagnostic statistics.
c
      call maxabs (nx,nx,ny,sst,msst)
c
      call setfmt (9,msst,fmtbit)
      call length (fmt,sln1,sbn1,snd1)
      call length (fmtbit,sln2,sbn2,snd2)
      fmt = fmt(sbn1:snd1)//',1x,'//fmtbit(sbn2:snd2)
c
c-----------------------------------------------------------------------
c  Get surface pressure.
c-----------------------------------------------------------------------
c
c  Read surface pressure.
c
      call read_fnmoc (pres,presfile)
c
c  Convert from Pascal to millibar.
c
      call scale_fld (1,nx,1,ny,0.01,pres)
c
c  Get diagnostic statistics.
c
      call maxabs (nx,nx,ny,pres,mpres)
c
      call setfmt (9,mpres,fmtbit)
      call length (fmt,sln1,sbn1,snd1)
      call length (fmtbit,sln2,sbn2,snd2)
      fmt = fmt(sbn1:snd1)//',1x,'//fmtbit(sbn2:snd2)
c
c-----------------------------------------------------------------------
c  Get precipitation.
c-----------------------------------------------------------------------
c
c  Read precipitation.
c
      call read_fnmoc (prcp,prcpfile)
c
c  Convert from kg/m^2 in 12 hours to cm/day
c
      call scale_fld (1,nx,1,ny,0.2,prcp)
c
#ifdef fixlims
c  Impose minimum values.
c
      call impmxmn (1,nx,1,ny,use,c0,no_use,c1,prcp)
#endif
c  Get diagnostic statistics.
c
      call maxabs (nx,nx,ny,prcp,mprcp)
c
      call setfmt (9,mprcp,fmtbit)
      call length (fmt,sln1,sbn1,snd1)
      call length (fmtbit,sln2,sbn2,snd2)
      fmt = fmt(sbn1:snd1)//',1x,'//fmtbit(sbn2:snd2)//')'
c
c-----------------------------------------------------------------------
c  Write diagnostic statistics.
c-----------------------------------------------------------------------
c
c  On first call, write header & initialize global statistics.
c  On subsequent calls, update global statistics.
c
      if (.not.first) then
         mmairt = max (mairt, mmairt)
         mmcldc = max (mcldc, mmcldc)
         mmprcp = max (mprcp, mmprcp)
         mmpres = max (mpres, mmpres)
         mmrelh = max (mrelh, mmrelh)
         mmsst  = max (msst,  mmsst)
         mmu10m = max (mu10m, mmu10m)
       else
         first = .false.
         write (iout,10)
         mmairt = mairt
         mmcldc = mcldc
         mmprcp = mprcp
         mmpres = mpres
         mmrelh = mrelh
         mmsst  = msst
         mmu10m = mu10m
      endif
c
c  Write instantaneous statistics.
c
      call length (fmt,sln1,sbn1,snd1)
      write (iout,fmt(sbn1:snd1)) time,mu10m,mcldc,mairt,mrelh,msst,
     &                            mpres,mprcp
c
      return
c
  10  format (/'Maximal magnitudes'//3x,'time',6x,'wind',5x,'cloud',6x,
     &        'airT',6x,'relH',6x,'sst',6x,'press',5x,'precip'/
     &        'ModJulDay',4x,'m/s',15x,'celsius',13x,'celsius',3x,
     &        'millibar',3x,'cm/day')
c
c=======================================================================
c  This entry allows the output of the global statistics.
c=======================================================================
c
      entry glbstat (iout)
c
c  Create global formats.
c
      call setfmt (9,mmu10m,fmtbit)
      call length (fmtbit,sln2,sbn2,snd2)
      fmt = '(10x,'//fmtbit(sbn2:snd2)
c
      call setfmt (9,mmcldc,fmtbit)
      call length (fmt,sln1,sbn1,snd1)
      call length (fmtbit,sln2,sbn2,snd2)
      fmt = fmt(sbn1:snd1)//',1x,'//fmtbit(sbn2:snd2)
c
      call setfmt (9,mmairt,fmtbit)
      call length (fmt,sln1,sbn1,snd1)
      call length (fmtbit,sln2,sbn2,snd2)
      fmt = fmt(sbn1:snd1)//',1x,'//fmtbit(sbn2:snd2)
c
      call setfmt (9,mmrelh,fmtbit)
      call length (fmt,sln1,sbn1,snd1)
      call length (fmtbit,sln2,sbn2,snd2)
      fmt = fmt(sbn1:snd1)//',1x,'//fmtbit(sbn2:snd2)
c
      call setfmt (9,mmsst,fmtbit)
      call length (fmt,sln1,sbn1,snd1)
      call length (fmtbit,sln2,sbn2,snd2)
      fmt = fmt(sbn1:snd1)//',1x,'//fmtbit(sbn2:snd2)
c
      call setfmt (9,mmpres,fmtbit)
      call length (fmt,sln1,sbn1,snd1)
      call length (fmtbit,sln2,sbn2,snd2)
      fmt = fmt(sbn1:snd1)//',1x,'//fmtbit(sbn2:snd2)
c
      call setfmt (9,mmprcp,fmtbit)
      call length (fmt,sln1,sbn1,snd1)
      call length (fmtbit,sln2,sbn2,snd2)
      fmt = fmt(sbn1:snd1)//',1x,'//fmtbit(sbn2:snd2)//')'
c
c  Write global statistics.
c
      write (iout,20)
c
      call length (fmt,sln1,sbn1,snd1)
      write (iout,fmt(sbn1:snd1)) mmu10m,mmcldc,mmairt,mmrelh,mmsst,
     &                            mmpres,mmprcp
c
      return
c
  20  format (///'Global maximal magnitudes'//13x,'wind',5x,'cloud',6x,
     &        'airT',6x,'relH',6x,'sst',6x,'press',5x,'precip'/
     &        13x,'m/s',15x,'celsius',13x,'celsius',3x,
     &        'millibar',3x,'cm/day')
c
      end
