      subroutine get_data (iout,time,taux,tauy)
c
c=======================================================================
c                                                                    ===
c  This routine reads one set of atmospheric data.                   ===
c                                                                    ===
c  ------                                                            ===
c  Input:                                                            ===
c  ------                                                            ===
c                                                                    ===
c     IOUT   Unit number for diagnostic output.    (integer)         ===
c                                                                    ===
c  -------                                                           ===
c  Output:                                                           ===
c  -------                                                           ===
c                                                                    ===
c     TIME   Time of data set.          (real; modified Julian day)  ===
c     TAUX   Zonal wind stress.         (real array; dynes/cm^2)     ===
c     TAUY   Meridional wind stress.    (real array; dynes/cm^2)     ===
c                                                                    ===
c  ------                                                            ===
c  Calls:                                                            ===
c  ------                                                            ===
c                                                                    ===
c     CLOUD_SCALE,  LENGTH,     MAXABS,      READ_FNMOC,  SCALE_FLD, ===
c     SETFMT,       SHIFT_FLD,  WINDSTRESS                           ===
#ifdef fixlims
c     IMPMXMN                                                        ===
#endif
c                                                                    ===
c=======================================================================
c
c-----------------------------------------------------------------------
c  Define global data.
c-----------------------------------------------------------------------
c
#include <param.h>
c
c-----------------------------------------------------------------------
c  Define local data.
c-----------------------------------------------------------------------
#ifdef fixlims
c
      logical no_use,use
      real    c0,c1
      parameter (c0=0.0, c1=1.0, use=.true., no_use=.false.)
#endif
c
      integer idum,iout,sbn1,sbn2,sln1,sln2,snd1,snd2
      logical first
      real    fac,mairt,mcldc,mprcp,mpres,mrelh,msst,mu10m,time
      real mmairt,mmcldc,mmprcp,mmpres,mmrelh,mmsst,mmu10m
      real airt(nx,ny),cldc(nx,ny),prcp(nx,ny),pres(nx,ny),relh(nx,ny),
     &     sst(nx,ny),taux(nx,ny),tauy(nx,ny),u10m(nx,ny),velx(nx,ny),
     &            vely(nx,ny)
      character*8  fmtbit
      character*80 airtfile,cldcfile,prcpfile,presfile,relhfile,
     &              sstefile,windfile_u,windfile_v
      character*256 fmt
c
      save first,mmairt,mmcldc,mmprcp,mmpres,mmrelh,mmsst,mmu10m
c
      data first /.true./
c
c=======================================================================
c  Begin executable code
c=======================================================================
c
c-----------------------------------------------------------------------
c  Read time of data set.
c-----------------------------------------------------------------------
c
      read (5,*) idum
      read (5,*) time
c
      call setfmt (9,time,fmtbit)
      call length (fmtbit,sln2,sbn2,snd2)
      fmt = '('//fmtbit(sbn2:snd2)
c
c-----------------------------------------------------------------------
c  Read names of data files.
c-----------------------------------------------------------------------
c
      read (5,*)     idum
      read (5,'(a)') windfile_u
c
      read (5,*)     idum
      read (5,'(a)') windfile_v
c
c-----------------------------------------------------------------------
c  Get wind stress.
c-----------------------------------------------------------------------
c
c  Read wind velocity components.
c
      call read_fnmoc (velx,windfile_u)
      call read_fnmoc (vely,windfile_v)
c
c  Compute wind stress.
c
      call windstress (velx,vely,taux,tauy,u10m)
c
c  Get diagnostic statistics.
c
      call maxabs (nx,nx,ny,u10m,mu10m)
c
      call setfmt (9,mu10m,fmtbit)
      call length (fmt,sln1,sbn1,snd1)
      call length (fmtbit,sln2,sbn2,snd2)
      fmt = fmt(sbn1:snd1)//',1x,'//fmtbit(sbn2:snd2)//')'
c
c-----------------------------------------------------------------------
c  Write diagnostic statistics.
c-----------------------------------------------------------------------
c
c  On first call, write header & initialize global statistics.
c  On subsequent calls, update global statistics.
c
      if (.not.first) then
         mmu10m = max (mu10m, mmu10m)
       else
         first = .false.
         write (iout,10)
         mmu10m = mu10m
      endif
c
c  Write instantaneous statistics.
c
      call length (fmt,sln1,sbn1,snd1)
      write (iout,fmt(sbn1:snd1)) time,mu10m
c
      return
c
  10  format (/'Maximal magnitudes'//3x,'time',6x,'wind'/
     &        'ModJulDay',4x,'m/s')
c
c=======================================================================
c  This entry allows the output of the global statistics.
c=======================================================================
c
      entry glbstat (iout)
c
c  Create global formats.
c
      call setfmt (9,mmu10m,fmtbit)
      call length (fmtbit,sln2,sbn2,snd2)
      fmt = '(10x,'//fmtbit(sbn2:snd2)//')'
c
c  Write global statistics.
c
      write (iout,20)
c
      call length (fmt,sln1,sbn1,snd1)
      write (iout,fmt(sbn1:snd1)) mmu10m
c
      return
c
  20  format (///'Global maximal magnitudes'//13x,'wind'/13x,'m/s')
c
      end
