function [mcoeff,ncoeff,zfit] = fit_niilerkraus (fitflg,iguess,zmld,wstrss, ...
                                                 hflx,emp,rflx,lat,sst,sss);
%
% [mcoeff,ncoeff,zfit] = fit_niilerkraus (fitflg,iguess,zmld,wstrss, ...
%                                                hflx,emp,rflx,lat,sst,sss);
%
% This function "best fits" an the free coefficient in the Niiler
% Kraus formula to in situ mixed layer depths.
%
% ------
% Input:
% ------
%
%    FITFLG....Flag for best fitting.  [0] use IGUESS  [1] least squares
%    IGUESS....Initial Guess.
%    ZMLD......Mixed layer depths at test points.            (cm)
%    WSTRSS....Wind stress at test points.                   (cm/s)
%    HFLX......Heat flux at test points.                     (Celsius cm/s)
%    EMP.......Evaporation - Precipitation at test points.   (cm/s)
%    RFLX......Shortwave radiation flux at test points.      (Celsius cm/s)
%    LAT.......Latitudes of test points.
%
% -------
% Output:
% -------
%
%    MCOEFF....Wind stress cofficient.
%    NCOEFF....Buoyancy cofficient.
%    ZFIT......Constructed Mixed layer depths.   (cm)

%-------------------------------------------------------------------------------
% Set useful parameters.
%-------------------------------------------------------------------------------

tol   = sqrt(eps);
itmax = 100;

s_std = 35.0;
t_std = 20.0;
p_std = 0.0;

alpha = sw_alpha (sss,sst,p_std)';
beta = sw_beta (sss,sst,p_std)';

atth2o = 4.0e-4;  % Light extinction coefficient (1/cm)

m2cm = 100;

%-------------------------------------------------------------------------------
% Construct simpifying quanities.
%-------------------------------------------------------------------------------

%---------------------------------
%--- Compute buoyance factors. ---
%---------------------------------

grav = sw_g(lat,0) .* m2cm;

srf = -rflx.*grav.*alpha;

sbf = -hflx.*grav.*alpha - srf + emp.*grav.*beta;

windflux = wstrss.^3;

%-------------------------------------
%--- Construct improved groupings. ---
%-------------------------------------

x = srf./atth2o;
y = (sbf+abs(sbf))./2;
z = srf + (sbf-abs(sbf))./2;

%-------------------------------------------------------
%--- Sneak a missing factor of 2 into the numerator. ---
%-------------------------------------------------------

windflux = 2.*windflux;
x        = 2.*x;

%-------------------------------------------------------------------------------
% By-pass for examing test values only.
%-------------------------------------------------------------------------------

if ~fitflg
   mcoeff = iguess(1);
   ncoeff = iguess(2);
   numer = mcoeff.*windflux - srf./atth2o;
   denom = -srf - ((1+ncoeff).*sbf-(1-ncoeff).*abs(sbf))./2;
   zfit = numer./denom;
   return;
end;

%-------------------------------------------------------------------------------
% By-pass for case where surface bouyancy flux is always negative
% (ie. ncoeff is indeterminant).
%-------------------------------------------------------------------------------

if (max(y(:))==0)
   vldind = find (~isnan(windflux) & ~isnan(z) & ~isnan(zmld) & ~isnan(x) ...
                  & (sqrt(eps).*abs(windflux) < abs(z)) );
   A = windflux(vldind)./z(vldind);
   C = zmld(vldind) - x(vldind)./z(vldind);
   mcoeff = -sum(A.*C)/sum(A.*A);
   ncoeff = 0;

   disp (' ');
   disp (['   Wind stress factor:  ',num2str(mcoeff,9)]);
   disp (['   Buoyancy factor:  ',num2str(ncoeff,9)]);

   numer = mcoeff.*windflux(vldind) - x(vldind);
   denom = -z(vldind);
   zfit = mean(zmld(:)).*ones(size(zmld));
   zfit(vldind) = numer./denom;

   return;
end;

%-------------------------------------------------------------------------------
% Compute Coefficients.
%-------------------------------------------------------------------------------

%-------------------------------
%--- Create initial guesses. ---
%-------------------------------

if isnan(iguess(2))
   ncoeff0 = 0.5;
 else
   ncoeff0 = iguess(2);
end;

if isnan(iguess(1))
   denom   = ncoeff0.*y+z;
   valind  = find (denom~=0);
   frac    = windflux(valind)./denom(valind);
   thng    = (x(valind)./denom(valind)-zmld(valind));
   mcoeff0 = sum(frac.*thng)./sum(frac.*frac);
 else
   mcoeff0 = iguess(1);
end;

param0 = Ninv (ncoeff0);

%--------------------------------------------------
%--- Initialize quantites for Newton iteration. ---
%--------------------------------------------------

solvec = [mcoeff0; param0];
tstfn  = 1;
itcnt  = 0;

m = solvec(1);
p = solvec(2);
[n,dndp,d2ndp2] = Nfn (p);
numer = m.*windflux-x;
denom = n.*y+z;
valind  = find (denom~=0);
frac  = numer(valind)./denom(valind);
fitfn = zmld(valind) + frac;
dfdm = windflux(valind)./denom(valind);
thng = frac.*y(valind)./denom(valind);
dfdn = -thng.*dndp;

fn = [sum(fitfn.*dfdm); sum(fitfn.*dfdn)];
fitfn0 = fitfn;

%-------------------------------------------------
%--- Newton iteration for best fit parameters. ---
%-------------------------------------------------

while ( (max(tstfn)>0) & (max(abs(fn))>tol) & (itcnt<itmax) & ...
        (length(valind)>0) )

   n0 = n;

   d2fdmdn = -windflux(valind).*y(valind).*dndp./ ...
              (denom(valind).*denom(valind)).*(fitfn + frac);
   dndpsq  = dndp.*dndp;
   d2fdn2  = thng.*(thng.*dndpsq + ...
             fitfn.*(2.*y(valind).*dndpsq./denom(valind) - d2ndp2));

   jacfn = [sum(dfdm.*dfdm) sum(d2fdmdn); sum(d2fdmdn) sum(d2fdn2)];

   delta = jacfn\(-fn);
   solvec = solvec + delta;

   m = solvec(1);
   p = solvec(2);
   [n,dndp,d2ndp2] = Nfn (p);
   numer = m.*windflux-x;
   denom = n.*y+z;
   valind  = find (denom~=0);
   frac  = numer(valind)./denom(valind);
   fitfn = zmld(valind) + frac;
   dfdm = windflux(valind)./denom(valind);
   thng = frac.*y(valind)./denom(valind);
   dfdn = -thng.*dndp;

   fn = [sum(fitfn.*dfdm); sum(fitfn.*dfdn)];

   itcnt = itcnt + 1;
   tstfn = abs([delta(1) (n-n0)]) - tol.*abs([solvec(1) n]);

end

%------------------------
%--- Extract results. ---
%------------------------

mcoeff = solvec(1);
p      = solvec(2);
[ncoeff,dndp,d2ndp2] = Nfn (p);

disp (' ');
disp (['   Wind stress factor:  ',num2str(mcoeff,9)]);
disp (['   Buoyancy factor:  ',num2str(ncoeff,9)]);

%-------------------------------------------------------------------------------
% Compute "fitted" mixed layer depth.
%-------------------------------------------------------------------------------

numer = mcoeff.*windflux - srf./atth2o;
denom = -srf - ((1+ncoeff).*sbf-(1-ncoeff).*abs(sbf))./2;
valind  = find (denom~=0);
zfit(valind) = numer(valind)./denom(valind);
mnind = find (denom==0);
zfit(mnind) = mean(zmld);

%-------------------------------------------------------------------------------
% Check for errors.
%-------------------------------------------------------------------------------

errstr = '';
if (sqrt(mean(fitfn.*fitfn)) > sqrt(mean(fitfn0.*fitfn0)))
   errstr = strvcat(errstr,'             initial guess gave better fit');
end;
if any(isinf(solvec))
   errstr = strvcat(errstr,['             ', ...
                            'one or more infinite coefficients']);
end;
if any(isnan(solvec))
   errstr = strvcat(errstr,['             ', ...
                            'one or more NaN coefficients']);
end;
if any(isinf(zfit))
   errstr = strvcat(errstr,['             ', ...
                            'coefficients produced at least 1 infinite MLD']);
end;
if any(isnan(zfit))
   errstr = strvcat(errstr,['             ', ...
                            'coefficients produced at least 1 NaN MLD']);
end;

%-----------------------------------------
%--- If any errors, use initial guess. ---
%-----------------------------------------

if ~isempty(errstr)
   mcoeff = mcoeff0;
   ncoeff = ncoeff0;
   numer = mcoeff.*windflux - srf./atth2o;
   denom = -srf - ((1+ncoeff).*sbf-(1-ncoeff).*abs(sbf))./2;
   zfit = numer./denom;

   disp (' ');
   disp ('+++Warning:  FIT_NIILERKRAUS - Newton iteration failure.');
   disp (errstr);
   disp (' ');
   disp ('Using initial guess values:');
   disp (['   Wind stress factor:  ',num2str(mcoeff,9)]);
   disp (['   Buoyancy factor:  ',num2str(ncoeff,9)]);
end;
