function [fitflg,iguess,mixmld,zcut,mldscl,which,supttl,hydname, ...
                     gridname,fluxname,outname] = fitml_inparm (infile);
%
% [fitflg,iguess,mixmld,zcut,mldscl,which,supttl,hydname,gridname, ...
%                        fluxname,outname] = fitml_inparm (infile);
%
% This function reads the parameters needed to drive the first guess for
% PE mixed-layer model parameters.
%
% ------
% Input:
% ------
%
%    INFILE     Parameter input file.
%
% -------
% Output:
% -------
%
%    FITFLG.....Flag for best fitting.  [0] use IGUESS  [1] least squares
%    IGUESS.....Initial Guess.
%    MIXMLD.....Mixing value applied to mixed layer.  (cm^2/s)
%    ZCUT.......Initial guess for how shallow cast must be for the
%               mixed layer depth calculation to be meaningful.
%    MLDSCL.....Factor by which MLD is amplified.
%    WHICH......Which mixed layer model to fit
%                 'Eckman'
%                 'NiilerKraus'
%                 'NiilerKraus0'  (Niiler-Kraus with NCOEFF==0)
%               This is also used to create an output diary file
%               named "fit_ml_WHICH.log".
%    SUPTTL.....Super title for plots
%    HYDNAME....Name of hydrographic data file in MODS format.
%    GRIDNAME...Name of GRIDS netCDF file.
%    FLUXNAME...Name of forcing netCDF file.
%    OUTNAME....Name for output log file.

%-------------------------------------------------------------------------------
% Initialize output.
%-------------------------------------------------------------------------------

fitflg   = [];
iguess   = [];
mixmld   = [];
zcut     = [];
mldscl   = [];
which    = '';
supttl   = '';
hydname  = '';
gridname = '';
fluxname = '';
outname  = '';

%-------------------------------------------------------------------------------
%  Open input file.
%-------------------------------------------------------------------------------

[fid,mess] = fopen (infile,'r');

if (fid<0)
   disp (' ');
   disp ('***Error:  FITML_INPARM - unable to open input file:');
   disp (['           ',setstr(34),infile,setstr(34)]);
   disp (['           ',setstr(34),mess,setstr(34)]);
   disp (' ');
   status = fid;
   return;
end;

clear mess;

%-------------------------------------------------------------------------------
% Read input ASCII file.
%-------------------------------------------------------------------------------

fitflg   = num_read (fid);
iguess   = num_read (fid);
mixmld   = num_read (fid);
zcut     = num_read (fid);
mldscl   = num_read (fid);
which    = fliplr(deblank(fliplr(deblank(fgetl(fid)))));
supttl   = fliplr(deblank(fliplr(deblank(fgetl(fid)))));
hydname  = fliplr(deblank(fliplr(deblank(fgetl(fid)))));
gridname = fliplr(deblank(fliplr(deblank(fgetl(fid)))));
fluxname = fliplr(deblank(fliplr(deblank(fgetl(fid)))));

%-------------------------------------------------------------------------------
% Close input ASCII file.
%-------------------------------------------------------------------------------

status = fclose (fid);

if (status<0)
   disp (' ');
   disp ('***Error:  FITML_INPARM - unable to close input file:');
   disp (['           ',setstr(34),infile,setstr(34)]);
   disp (' ');
   return;
end;

clear fid status;

%-------------------------------------------------------------------------------
% Set output log file name.
%-------------------------------------------------------------------------------

if fitflg
   fitstr = 'fitml';
 else
   fitstr = 'testml';
end;

mixstr = num2str(mixmld);
ind = findstr ('.',mixstr);
if ~isempty(ind)
   mixstr(ind) = 'p';
end

sclstr = num2str(mldscl);
ind = findstr ('.',sclstr);
if ~isempty(ind)
   sclstr(ind) = 'p';
end

outname = [fitstr,sclstr,'_',which,'_',mixstr,'.log'];

clear fitstr ind mixstr sclstr
