function flux = intsclflux (ncid,flxnam,flxtim,gnx,gny,x,y,time,tdk,tback);
%
% flux = intsclflux (ncid,flxnam,flxtim,gnx,gny,x,y,time,tdk,tback);
%
% This function interpolates the requested scalar flux to the mixed layer
% positions and integrates it in time.
%
% ------
% Input:
% ------
%
%    NCID.....NetCDF identifier for opened flux file.
%    FLXNAM...Name of requested flux.
%    FLXTIM...Available times of requested flux.
%    GNX......Number of grid points in X-direction.
%    GNY......Number of grid points in Y-direction.
%    X........X-coordinate of mixed layer depth data.
%    Y........Y-coordinate of mixed layer depth data.
%    TIME.....Times of mixed layer depth data.
%    TDK......Temporal decay scale for mixed layer depths.
%    TBACK....Length of time to integrate in past for mixed layer depths.
%
% -------
% Output:
% -------
%
%    FLUX.....Integrated flux at mixed layer positions.

%-------------------------------------------------------------------------------
% Loop over all mixed layer depths.
%-------------------------------------------------------------------------------

nmld = length(x);
ndsp = unique(round((0.1:0.1:1.1).*nmld));

gnxm1 = gnx - 1;
gnym1 = gny - 1;

for n = 1:nmld

   %----------------------------------------
   %--- Find temporal extraction limits. ---
   %----------------------------------------

   stind = max(find(flxtim<(time(n)-tback(n))));
   if isempty(stind)
      stind = 1;
   end;

   ndind = min([(max(find(flxtim<time(n)))+1) length(flxtim)]);

   %------------------------------------
   %--- Extract working time vector. ---
   %------------------------------------

   timind = stind:ndind;
   twk    = flxtim(timind);
   ntim   = length(timind);

   %----------------------------------------------------------
   %--- Extract time series of fluxes at current location. ---
   %----------------------------------------------------------

   ii = min([fix(x(n)) gnxm1]);
   ip = ii + 1;
   dx = x(n) - ii;

   jj = min([fix(y(n)) gnym1]);
   jp = jj + 1;
   dy = y(n) - jj;

   flxiijj = ncid{flxnam}(timind,jj,ii);
   flxipjj = ncid{flxnam}(timind,jj,ip);
   flxiijp = ncid{flxnam}(timind,jp,ii);
   flxipjp = ncid{flxnam}(timind,jp,ip);

   flx = (1-dy).*(flxiijj.*(1-dx) + flxipjj.*dx) + ...
             dy.*(flxiijp.*(1-dx) + flxipjp.*dx);

   clear flxiijj flxipjj flxiijp flxipjp

   %-----------------------------------------------
   %--- Make sure last flux is at current time. ---
   %-----------------------------------------------

   if (flxtim(ndind)>time(n))
      flx(ntim) = interp1(twk,flx,time(n));
      twk(ntim) = time(n);
   end;

   %-------------------------------
   %--- Integrate flux in time. ---
   %-------------------------------

   wt  = exp((twk-time(n))./tdk(n));
   flx = wt.*flx;

   flux(n) = trapz(twk,flx)/trapz(twk,wt);

   %-------------------------
   %--- Report on status. ---
   %-------------------------

   if (n==ndsp(1))
      ndsp(1) = [];
      disp (['         completed ',num2str(n),' of ',num2str(nmld),' points.']);
   end;

end;
