function [nx,ny,dx,dy,lonc,latc,delx,dely,theta,coord,status] = read_grd(gname);

%
% function [nx,ny,dx,dy,lonc,latc,delx,dely,theta,coord,status] = ...
%                                                         read_grd (gname);
%
% This routine reads in domain parameters, bathymetry, and Land/Sea mask
% data (if any) from GRIDS NetCDF file.
%
% ------
% Input:
% ------
%
%    gname     GRIDS NetCDF file name (character string).
%
% -------
% Output:
% -------
%
%    NX        number grid points in x-direction.
%    NY        number grid points in y-direction.
%    DX        mean x-grid spacing (real; meters).
%    DY        mean y-grid spacing (real; meters).
%    LONC      transformation centroid longitude (real; degrees).
%    LATC      transformation centroid latitude (real; degrees).
%    DELX      X-offset trans-domain centers.    (m | deg).
%    DELY      Y-offset trans-domain centers.    (m | deg).
%    THETA     domain rotation angle (real; degrees).
%    COORD     grid type (integer).
%              [0] Cartesian, [1] spherical, [2] rotated spherical.
%    STATUS    Exit status.  [<0] error [>=0] no error

%----------------------------------------------------------------------------
% Open GRIDS NetCDF file.  Supress all error messages from NetCDF.
%----------------------------------------------------------------------------

ncid = mexcdf ('ncopen',gname,'nc_nowrite');
if (ncid < 0),
  disp (' ');
  disp ('***Error:  READ_GRD - unable to open file:');
  disp (['           ',setstr(34),gname,setstr(34)]);
  disp (' ');
  status = ncid;
  return
end

status = mexcdf('setopts',0);

%----------------------------------------------------------------------------
% Read in domain parameters.
%----------------------------------------------------------------------------

%--------------------------
%--- Read in grid type. ---
%--------------------------

name = 'coord';
varid = mexcdf ('ncvarid',ncid,name);
if (varid > 0),
   [coord,status]=mexcdf('ncvarget1',ncid,varid,[0]);
   if (status == -1),
      disp (' ');
      disp (['***Error:  READ_GRD - unable to read variable ', ...
                      setstr(34),name,setstr(34),' in file:']);
      disp (['           ',setstr(34),gname,setstr(34)]);
      disp (' ');
      return
   end
 else
   coord=0;
end

%-----------------------------
%--- Read in grid spacing. ---
%-----------------------------

name = 'meandx';
[dx,status] = get_var1 (ncid,name,0);
if (status<0),
   disp (' ');
   disp (['***Error:  READ_GRD - unable to read variable ', ...
                   setstr(34),name,setstr(34),' in file:']);
   disp (['           ',setstr(34),gname,setstr(34)]);
   disp (' ');
   return
end;

name = 'meandy';
[dy,status] = get_var1 (ncid,name,0);
if (status<0),
   disp (' ');
   disp (['***Error:  READ_GRD - unable to read variable ', ...
                   setstr(34),name,setstr(34),' in file:']);
   disp (['           ',setstr(34),gname,setstr(34)]);
   disp (' ');
   return
end;

%-----------------------------------------------------------
%--- Read transformation centroid (lon,lat) coordinates. ---
%-----------------------------------------------------------

name = 'rlngd';
[lonc,status] = get_var1 (ncid,name,0);
if (status<0),
   disp (' ');
   disp (['***Error:  READ_GRD - unable to read variable ', ...
                   setstr(34),name,setstr(34),' in file:']);
   disp (['           ',setstr(34),gname,setstr(34)]);
   disp (' ');
   return
end;

name = 'rlatd';
[latc,status] = get_var1 (ncid,name,0);
if (status<0),
   disp (' ');
   disp (['***Error:  READ_GRD - unable to read variable ', ...
                   setstr(34),name,setstr(34),' in file:']);
   disp (['           ',setstr(34),gname,setstr(34)]);
   disp (' ');
   return
end;

%--------------------------------------------------
%--- Read transformation-domain center offsets. ---
%--------------------------------------------------

old = 0;

name = 'delx';
varid = mexcdf ('ncvarid',ncid,name);
if (varid > 0),
   [delx,status] = mexcdf( 'ncvarget1',ncid,varid,[0]);
   if (status <0),
      disp (' ');
      disp (['***Error:  READ_GRD - unable to read variable ', ...
                      setstr(34),name,setstr(34),' in file:']);
      disp (['           ',setstr(34),gname,setstr(34)]);
      disp (' ');
      return
   end
  else
   old = 1;
end

name = 'dely';
varid = mexcdf ('ncvarid',ncid,name);
if (varid > 0),
   [dely,status] = mexcdf( 'ncvarget1',ncid,varid,[0]);
   if (status <0),
      disp (' ');
      disp (['***Error:  READ_GRD - unable to read variable ', ...
                      setstr(34),name,setstr(34),' in file:']);
      disp (['           ',setstr(34),gname,setstr(34)]);
      disp (' ');
      return
   end
  else
   old = 1;
end

if old,
   if (coord~=1),
      delx = 0;
      dely = 0;
     else
      delx = lonc;
      dely = latc;
      lonc = 0;
      latc = 0;
   end;
end;

%--------------------------------------
%--- Read in domain rotation angle. ---
%--------------------------------------

name = 'thetad';
[theta,status] = get_var1 (ncid,name,0);
if (status<0),
   disp (' ');
   disp (['***Error:  READ_GRD - unable to read variable ', ...
                   setstr(34),name,setstr(34),' in file:']);
   disp (['           ',setstr(34),gname,setstr(34)]);
   disp (' ');
   return
end;

%--------------------------------------
%--- Read in number of grid points. ---
%--------------------------------------

name = 'imt';
[nx,status] = get_var1 (ncid,name,0);
if (status<0),
   disp (' ');
   disp (['***Error:  READ_GRD - unable to read variable ', ...
                   setstr(34),name,setstr(34),' in file:']);
   disp (['           ',setstr(34),gname,setstr(34)]);
   disp (' ');
   return
end;

name = 'jmt';
[ny,status] = get_var1 (ncid,name,0);
if (status<0),
   disp (' ');
   disp (['***Error:  READ_GRD - unable to read variable ', ...
                   setstr(34),name,setstr(34),' in file:']);
   disp (['           ',setstr(34),gname,setstr(34)]);
   disp (' ');
   return
end;

%----------------------------------------------------------------------------
% Close Grids NetCDF file.
%----------------------------------------------------------------------------

status = mexcdf('ncclose',ncid);
if (status<0),
  disp (' ');
  disp ('***Error:  READ_GRD - unable to close file:');
  disp (['           ',setstr(34),gname,setstr(34)]);
  disp (' ');
  return
end
