function [mcoeff,ncoeff,zfit] = fit_NK_n0 (fitflg,iguess,zmld,wstrss,hflx, ...
                                           emp,rflx,lat,sst,sss);
%
% [mcoeff,ncoeff,zfit] = fit_NK_n0 (fitflg,iguess,zmld,wstrss,hflx, ...
%                                   emp,rflx,lat,sst,sss);
%
% This function "best fits" an the free coefficient in the Niiler
% Kraus formula to in situ mixed layer depths under the constraint
% that the bouyancy coefficient is zero.
%
% ------
% Input:
% ------
%
%    FITFLG....Flag for best fitting.  [0] use IGUESS  [1] least squares
%    IGUESS....Initial Guess.
%    ZMLD......Mixed layer depths at test points.            (cm)
%    WSTRSS....Wind stress at test points.                   (cm/s)
%    HFLX......Heat flux at test points.                     (Celsius cm/s)
%    EMP.......Evaporation - Precipitation at test points.   (cm/s)
%    RFLX......Shortwave radiation flux at test points.      (Celsius cm/s)
%    LAT.......Latitudes of test points.
%
% -------
% Output:
% -------
%
%    MCOEFF....Wind stress cofficient.
%    NCOEFF....Buoyancy cofficient.
%    ZFIT......Constructed Mixed layer depths.   (cm)

%-------------------------------------------------------------------------------
% Set useful parameters.
%-------------------------------------------------------------------------------

tol   = sqrt(eps);
itmax = 100;

s_std = 35.0;
t_std = 20.0;
p_std = 0.0;

alpha = sw_alpha (sss,sst,p_std)';
beta = sw_beta (sss,sst,p_std)';

atth2o = 4.0e-4;  % Light extinction coefficient (1/cm)

m2cm = 100;

%-------------------------------------------------------------------------------
% Construct simpifying quanities.
%-------------------------------------------------------------------------------

%---------------------------------
%--- Compute buoyancy factors. ---
%---------------------------------

grav = sw_g(lat,0) .* m2cm;

srf = -rflx.*grav.*alpha;

sbf = -hflx.*grav.*alpha - srf + emp.*grav.*beta;

windflux = wstrss.^3;

%-------------------------------------
%--- Construct improved groupings. ---
%-------------------------------------

x = srf./atth2o;
y = (sbf+abs(sbf))./2;
z = srf + (sbf-abs(sbf))./2;

%-------------------------------------------------------------------------------
% Simple case for when NCOEFF is defined to be zero.
%-------------------------------------------------------------------------------

ncoeff = 0.0;

vldind = find (~isnan(windflux) & ~isnan(z) & ~isnan(zmld) & ~isnan(x) ...
               & (sqrt(eps).*abs(windflux) < abs(z)) );

if (fitflg | any(isnan(iguess)))
   z(vldind) = z(vldind) + ncoeff.*y(vldind);
   A = windflux(vldind)./z(vldind);
   C = zmld(vldind) - x(vldind)./z(vldind);
   mcoeff = -sum(A.*C)/sum(A.*A);
 else
   mcoeff = iguess(1);
end;

disp (' ');
disp (['   Wind stress factor:  ',num2str(mcoeff,9)]);
disp (['   Buoyancy factor:  ',num2str(ncoeff,9)]);

numer = mcoeff.*windflux(vldind) - x(vldind);
denom = -z(vldind);
zfit = mean(zmld(:)).*ones(size(zmld));
zfit(vldind) = numer./denom;
