% fit_ml
%
% This script "best fits" the parameters for computing mixed layer depth.

%-------------------------------------------------------------------------------
% Get run parameters:
%-------------------------------------------------------------------------------

%-----------------------------------------
%--- Get name of parameter input file. ---
%-----------------------------------------

[fname,pname] = uigetfile ('*.in','Select parameter input file');

if (isstr(fname) & isstr(pname))
   infile = fullfile (pname,fname);
elseif isstr(fname)
   infile = fname;
else
   disp (' ');
   disp ('***Error:  FIT_ML - unspecified parameter input file');
   disp (' ');
   return;
end;

clear fname pname

%----------------------------------
%--- Read parameter input file. ---
%----------------------------------

[fitflg,iguess,mixmld,zcut,mldscl,which,supttl,hydname, ...
                         gridname,fluxname,outname] = fitml_inparm (infile);

if isempty(outname)
   disp (' ');
   disp ('***Error:  FIT_ML - unable to read parameter input file:');
   disp (['           ',setstr(34),infile,setstr(34)]);
   disp (' ');
   return;
end;

clear infile

%-------------------------------------------------------------------------------
% Start log file.
%-------------------------------------------------------------------------------

eval(['diary ',outname]);

clear outname

disp (['Wind Mixing coefficient:  ',num2str(mixmld),' (cm^2/s)']);

%-------------------------------------------------------------------------------
% Set useful constants.
%-------------------------------------------------------------------------------

cm2m = 0.01;

%-------------------------------------------------------------------------------
% Get data.
%-------------------------------------------------------------------------------

ann_tic('Getting data:');

[time,lat,zmld,wstrss,hflx,emp,rflx,sst,sss,stat] = ...
                       get_data (zcut,mixmld,mldscl,hydname,gridname,fluxname);

hms_toc;

if (stat<0)
   disp (' ');
   disp ('***Error:  FIT_ML - unable to read data for fitting.');
   disp (' ');
   return;
end;

clear hydname gridname fluxname stat;

%-------------------------------------------------------------------------------
% Fit parameters.
%-------------------------------------------------------------------------------

if strcmp(upper(which),'ECKMAN')
   [eckfac,zfit] = fit_eckman (fitflg,iguess,zmld,wstrss,lat);

  elseif strcmp(upper(which),'NIILERKRAUS')
   if ~isnan(iguess(2))
      [mcoeff,ncoeff,zfit] = fit_niilerkraus (fitflg,iguess,zmld,wstrss, ...
                                              hflx,emp,rflx,lat,sst,sss);
    else
      soln  = [];
      zsave = [];
      errmn = Inf;
      for n0 = 0:0.01:1
         iguess(2) = n0;
         [mcoeff,ncoeff,zfit] = fit_niilerkraus (fitflg,iguess,zmld,wstrss, ...
                                                 hflx,emp,rflx,lat,sst,sss);
         newerr = checkmld (zmld,zfit);
         if (newerr<errmn)
            soln  = [mcoeff ncoeff];
            zsave = zfit;
            errmn = newerr;
         end;
      end;
      mcoeff = soln(1);
      ncoeff = soln(2);
      zfit   = zsave;
      clear soln errmn n0 newerr zsave;
      disp (' ');
      disp ('Best NK coefficients:');
      disp (['   Wind stress factor:  ',num2str(mcoeff,9)]);
      disp (['   Buoyancy factor:  ',num2str(ncoeff,9)]);
   end;

  else
   [mcoeff,ncoeff,zfit] = fit_NK_n0 (fitflg,iguess,zmld,wstrss, ...
                                     hflx,emp,rflx,lat,sst,sss);
end;

if fitflg
   fitstr = 'Fitted';
 else
   fitstr = 'Test';
end;

clear lat wstrss hflx emp rflx sst sss fitflg iguess;

%-------------------------------------------------------------------------------
% Toss outliers from fit (PE model does protect against this).
%-------------------------------------------------------------------------------

olflag = 0;

cutval = min(zmld(:));
ind    = find (zfit<cutval);
if ~isempty(ind)
   zfit(ind) = cutval;
   olflag = 1;
end;

cutval = max(zmld(:));
ind    = find (zfit>cutval);
if ~isempty(ind)
   zfit(ind) = cutval;
   olflag = 1;
end;

%-------------------------------------------------------------------------------
% Determine whether to bother mentioning MLD scaling.
%-------------------------------------------------------------------------------

if (abs(mldscl-1)>sqrt(eps))
   sclstr = ['(x',num2str(mldscl),')'];
 else
   sclstr = '';
end;

%-------------------------------------------------------------------------------
% Plot results.
%-------------------------------------------------------------------------------

h = suptitle(supttl);
set (h,'FontName','Times','FontSize',14,'FontWeight','Bold');

figure;subplot(2,1,1);
plot(time,zmld.*cm2m,'kx',time,zfit.*cm2m,'ko');
set (gca,'FontName','Times','FontSize',14);
tlbl = get(gca,'XTick');
gtime = gregorian(tlbl+2440000);
str = [num2str(gtime(:,2),'%2.2d'), ...
       setstr(abs('/').*ones(length(tlbl),1)), ...
       num2str(gtime(:,3),'%2.2d')];
set(gca,'XTickLabel',str);
xlabel('Date');
if isempty(sclstr)
   ylabel('Mixed layer depth  (m)');
 else
   ylabel(['Mixed layer depth  (m)  ',sclstr]);
end;
if strcmp(upper(which),'ECKMAN')
   title ([fitstr,' Eckman Factor:  ',num2str(eckfac,9)]);
  else
   title ([fitstr,' Wind Stress Factor:  ',num2str(mcoeff,9)]);
end;
legend('Data','Fitted',0);

if olflag
   supttl = strvcat(supttl,'(outliers limited)');
end;

h = suptitle(supttl);
set (h,'FontName','Times','FontSize',14,'FontWeight','Bold');

clear gtime h str supttl tlbl

%-----------------------
%--- Compute errors. ---
%-----------------------

zerr = (zmld - zfit).*cm2m;

subplot(2,1,2);
x = nice([min(zerr) max(zerr)],4);
hist_pjh(zerr,length(x));
colormap([0.5 0.5 0.5; 0.9 0.9 0.9]);
set (gca,'FontName','Times','FontSize',14);
xlabel('Z_{MLD} - Z_{fit}   (m)');
if strcmp(upper(which),'ECKMAN')
   title ([fitstr,' Eckman Factor:  ',num2str(eckfac,9)]);
  else
   title ([fitstr,' Buoyancy Factor:  ',num2str(ncoeff,9)]);
end;

%--------------------------
%--- Report statistics. ---
%--------------------------

disp (' ');
disp ('Fit:');
disp (['   Mean error:  ',num2str(mean(zerr))]);
disp (['   RMS error:  ',num2str(sqrt(mean(zerr.*zerr)))]);

zmld = zmld.*cm2m;
zfit = zfit.*cm2m;
disp (' ');
if isempty(sclstr)
   disp ('Mixed-Layer Depth Statistics:');
 else
   disp (['Mixed-Layer Depth ',sclstr,' Statistics:']);
end;
disp ('   Data:');
disp (['      Minimum: ',num2str(min(zmld)),'  (m)']);
disp (['      Maximum: ',num2str(max(zmld)),'  (m)']);
disp (['      Mean:    ',num2str(mean(zmld)),'  (m)']);
disp (['      StdDev:  ',num2str(std(zmld)),'  (m)']);
if ~olflag
   disp (['   ',fitstr,':']);
 else
   disp (['   ',fitstr,' (outliers limited):']);
end;
disp (['      Minimum: ',num2str(min(zfit)),'  (m)']);
disp (['      Maximum: ',num2str(max(zfit)),'  (m)']);
disp (['      Mean:    ',num2str(mean(zfit)),'  (m)']);
disp (['      StdDev:  ',num2str(std(zfit)),'  (m)']);
disp (' ');

clear cm2m eckfac mcoeff ncoeff time which x zerr zfit zmld olflag ...
      sclstr fitstr;

%-------------------------------------------------------------------------------
% Stop log file.
%-------------------------------------------------------------------------------

diary off;
