function [time,lat,zmld,wstrss,hflx,emp,rflx,sst,sss,stat] = ...
                           get_data (zcut,mixmld,mldscl,hydnam,gridnam,flxnam);
%
% [time,lat,zmld,wstrss,hflx,emp,rflx,stat] = ...
%                          get_data (zcut,mixmld,mldscl,hydnam,gridnam,flxnam);
%
% This function reads in the necessary data for fitting
% Mixed-layer formulae.
%
% ------
% Input:
% ------
%
%    MIXMLD....Mixing value applied to mixed layer.  (cm^2/s)
%    ZCUT......Initial guess for how shallow cast must be for the
%              mixed layer depth calculation to be meaningful.
%    MLDSCL....Factor by which MLD is amplified.
%    HYDNAM....Name of hydrography data file.
%    GRIDNAM...Name of GRIDS netCDF file.
%    FLXNAM....Name of forcing netCDF file.
%
% -------
% Output:
% -------
%
%    TIME......Times of the test data.                       (Mod Julian day)
%    LAT.......Latitudes of test points.                     (degrees north)
%    ZMLD......Mixed layer depths at test points.            (cm)
%    WSTRSS....Wind stress at test points.                   (cm/s)
%    HFLX......Heat flux at test points.                     (Celsius cm/s)
%    EMP.......Evaporation - Precipitation at test points.   (cm/s)
%    RFLX......Shortwave radiation flux at test points.      (Celsius cm/s)
%    STAT......Exit status.

%-------------------------------------------------------------------------------
% Initialize output.
%-------------------------------------------------------------------------------

lat    = [];
zmld   = [];
wstrss = [];
bflx   = [];
rflx   = [];
emp    = [];
sst    = [];
sss    = [];

%-------------------------------------------------------------------------------
% Set Constant Conversion factors.  Also set cut-off fraction for integration.
%-------------------------------------------------------------------------------

m2cm = 100.0;
sec2day = 1/86400;

cutfrac = 0.1;

%-------------------------------------------------------------------------------
% Convert wind mixing coefficient from (cm^2/s) to (cm^2/day).
%-------------------------------------------------------------------------------

mixmld = mixmld/sec2day;

%-------------------------------------------------------------------------------
% Get grid parameters.
%-------------------------------------------------------------------------------

disp ('      reading grid parameters');

[nx,ny,dx,dy,lonc,latc,delx,dely,theta,coord,stat] = read_grd (gridnam);

if (stat<0)
   disp (' ');
   disp ('***Error:  GET_DATA - unable to read GRIDS netCDF file:');
   disp (['           ',setstr(34),gridnam,setstr(34)]);
   return;
end;

%-------------------------------------------------------------------------------
% Read hydrography data.
%-------------------------------------------------------------------------------

disp ('      reading hydrography data');

%------------------------------
%--- Read hydrography file. ---
%------------------------------

[header,hinfo,htype,z,t,s] = rhydro (hydnam,1);

%---------------------
%--- Extract data. ---
%---------------------

lon  = hinfo(:,4);
lat  = hinfo(:,5);
time = hinfo(:,7);

clear header hinfo htype;

%----------------------------------
%--- Remove out-of-domain data. ---
%----------------------------------

[x,y] = ll2xy (lon,lat,coord,nx,ny,dx,dy,lonc,latc,delx,dely,theta);

clear lon;

badind = find( (x<1.5) | (x>nx) | (y<1.5) | (y>ny) );
if ~isempty(badind)
   x(badind)    = [];
   y(badind)    = [];
   lat(badind)  = [];
   time(badind) = [];
   z(badind,:)  = [];
   t(badind,:)  = [];
   s(badind,:)  = [];
end;

%-------------------------------------------------------------------------------
% Open forcing file.
%-------------------------------------------------------------------------------

disp ('      opening forcing file, reading times');

nc_forc = netcdf(flxnam);

%-------------------------------------------------------------------------------
% Extract forcing times.
%-------------------------------------------------------------------------------

smf_time = nc_forc{'smf_time'}(:);
shf_time = nc_forc{'shf_time'}(:);
swf_time = nc_forc{'swf_time'}(:);
swr_time = nc_forc{'swr_time'}(:);

tmin = max(min([smf_time; shf_time; swf_time; swr_time]));
tmax = min(max([smf_time; shf_time; swf_time; swr_time]));

%---------------------------------------------
%--- Remove data not supported by forcing. ---
%---------------------------------------------

badind = find( (time<=tmin) | (time>tmax) );
if ~isempty(badind)
   x(badind)    = [];
   y(badind)    = [];
   lat(badind)  = [];
   time(badind) = [];
   z(badind,:)  = [];
   t(badind,:)  = [];
   s(badind,:)  = [];
end;

%-------------------------------------------------------------------------------
% Compute mixed layer depths.
%-------------------------------------------------------------------------------

disp ('      computing mixed layer depth');

%------------------------------------------
%--- First compute "MLD" for all casts. ---
%------------------------------------------

zmld = mixed_layer (z,t,s,lat);
sst = t(:,1)';
sss = s(:,1)';

lat = lat';
timwk = time';

%-------------------------------------------------------------------
%--- Next, only keep casts that start "close enough" to surface. ---
%-------------------------------------------------------------------

valind0 = [];
valind  = find (z(:,1)<=zcut);
disp (['            zcut = ',num2str(zcut),' (m)']);

while ( (length(valind)>length(valind0)) & (length(valind)<length(zmld)) )
   valind0 = valind;
   disp (['            zcut = ',num2str(mean(zmld(valind0))),' (m)']);
   valind  = find (z(:,1)<=mean(zmld(valind0)));
end;
disp (['         retaining ',num2str(length(valind)),' of ', ...
       num2str(length(zmld)),' stations']);

zcut  = mean(zmld(valind0));

zmld  = zmld(valind);
sst   = sst(valind);
sss   = sss(valind);
lat   = lat(valind);
timwk = timwk(valind);
x     = x(valind);
y     = y(valind);
z     = z(valind,:);
t     = t(valind,:);
s     = s(valind,:);

plot_castMLD (z,t,s,lat,zmld,zcut);

clear s t z valind0 valind zcut

%-------------------------------------
%--- Rescale MLD and sort by time. ---
%-------------------------------------

zmld = zmld .* (m2cm*mldscl);

[time,ind] = sort(timwk);
lat = lat(ind);
zmld = zmld(ind);
sst  = sst(ind);
sss  = sss(ind);
x    = x(ind);
y    = y(ind);

tdk   = (zmld.*zmld)./mixmld;
tback = tdk.*log(1/cutfrac);

%-------------------------------------------------------------------------------
% Set Variable Conversion factors.
%-------------------------------------------------------------------------------

disp ('      computing conversion factors');

%s_std = 35.0;
%t_std = 20.0;
%p_std = 0.0;

rho0 = sw_dens0 (sss,sst);      % Density sea-water:       kg/(m^3)
Cp   = sw_cp (sss,sst,0.0);     % Heat capcity sea-water:  J/(kg C)
crho0 = rho0./1000;             % Density sea-water  gm/(cm^3)

heatfac = m2cm./(rho0.*Cp);

%-------------------------------------------------------------------------------
% Extract Wind Stress.
%-------------------------------------------------------------------------------

disp ('      reading wind stress');

if ~isempty(smf_time)
   wstrss = intvmgflux (nc_forc,'smflux',smf_time,nx,ny,x,y,time,tdk,tback);
   wstrss = sqrt(wstrss./crho0);
 else
   wstrss = zeros(size(time));
end;

%-------------------------------------------------------------------------------
% Extract Net Heat Flux.
%-------------------------------------------------------------------------------

disp ('      reading net heat flux');

if ~isempty(shf_time)
   hflx = intsclflux (nc_forc,'shflux',shf_time,nx,ny,x,y,time,tdk,tback);
   hflx = hflx .* heatfac;
 else
   hflx = zeros(size(time));
end;

%-------------------------------------------------------------------------------
% Extract Evaporation - Precipitation.
%-------------------------------------------------------------------------------

disp ('      reading evaporation - precipitation');

if ~isempty(swf_time)
   emp = intsclflux (nc_forc,'swflux',swf_time,nx,ny,x,y,time,tdk,tback);
   emp = emp .* sec2day;
 else
   emp = zeros(size(time));
end;

%-------------------------------------------------------------------------------
% Extract Shortwave Radiation.
%-------------------------------------------------------------------------------

disp ('      reading shortwave radiation');

if ~isempty(swr_time)
   rflx = intsclflux (nc_forc,'swrad',swr_time,nx,ny,x,y,time,tdk,tback);
   rflx = rflx .* heatfac;
 else
   rflx = zeros(size(time));
end;
