      subroutine read_grids
c
c=======================================================================
c                                                                    ===
c  This routine reads in the output of GRIDS.                        ===
c                                                                    ===
c  On Input:                                                         ===
c                                                                    ===
c     Common Blocks:                                                 ===
c                                                                    ===
c     /IOUNITS/                                                      ===
c                                                                    ===
c        FNAME     input/output filenames (character array).         ===
c                  FNAME(5)  input; GRIDS file.                      ===
c        STDOUT    standard output unit (integer).                   ===
c                                                                    ===
c     /SWITCHES/                                                     ===
c                                                                    ===
c        IFLAG     Initialization flags (integer array):             ===
c                  IFLAG(6)   ageostophy switch:                     ===
c                             [1] none, [2] beta plane,              ===
c                             [3] full Coriolis.                     ===
c                  IFLAG(7)   bottom topography switch:              ===
c                             [0] flat, [1] provided file.           ===
c                                                                    ===
c  On Output:                                                        ===
c                                                                    ===
c     Common Blocks:                                                 ===
c                                                                    ===
c     /CSTSEG/                                                       ===
c                                                                    ===
c        CSTSEGX    x-index coordinate for each coastal segment.     ===
c        CSTSEGX    y-index coordinate for each coastal segment.     ===
c        IEIS       ending I-index for each island box.              ===
c        ISIS       starting I-index for each island box.            ===
c        JEIS       ending J-index for each island box.              ===
c        JSIS       starting J-index for each island box.            ===
c        LANDT      mask for tracer points: [0] land, [1] sea.       ===
c        LENSEG     number of points in each coastal segment.        ===
c        LSEGMX     current maximum number of points per coastal     ===
c                   segment.                                         ===
c        NISLE      number of islands in the model basin.            ===
c        NSEG       number of coastal segments.                      ===
c                                                                    ===
c     /GRDDAT/                                                       ===
c                                                                    ===
c        GEOLAT    geographic latitudes at every grid point          ===
c                  (degrees; real array).                            ===
c        TMETX     x-coordinate metric coefficients at tracer grid   ===
c                  (cm; real array).                                 ===
c        TMETY     y-coordinate metric coefficients at tracer grid   ===
c                  (cm; real array).                                 ===
c        VMETX     x-coordinate metric coefficients at velocity grid ===
c                  (cm; real array).                                 ===
c        VMETY     y-coordinate metric coefficients at velocity grid ===
c                  (cm; real array).                                 ===
c                                                                    ===
c     /HYBRID/                                                       ===
c                                                                    ===
c        DBLSIGMA  hybrid coordinates transformation (integer):      ===
c                  [0] single, [1] double                            ===
c        HZ        nominal vertical thicknesses (cm; real).          ===
c        KC        hybrid coordinates interface level (integer).     ===
c        ZC1       shallowest coord. interface depth (cm; real).     ===
c        ZC2       deepest coord. interface depth (cm; real).        ===
c        ZREF      reference depth for coord. interface (cm; real).  ===
c        ZSLOPE    slope parameter of the coord. interface (real).   ===
c                                                                    ===
c     /MODDAT/                                                       ===
c                                                                    ===
c        COORD     flag for horizontal coordinates:   (integer)      ===
c                  [0] cartesian.                                    ===
c                  [1] unrotated spherical.                          ===
c                  [2] rotated spherical.                            ===
c        DELX      offset domain-trans. centers.      (cm|deg; real) ===
c        DELY      offset domain-trans. centers.      (cm|deg; real) ===
c        DX        mean grid spacing in x-direction.  (cm|deg; real) ===
c        DY        mean grid spacing in y-direction.  (cm|deg; real) ===
c        RLAT0     transformation centroid latitude.  (deg; real)    ===
c        RLNG0     transformation centroid longitude. (deg; real)    ===
c        THETAD    domain central rotation.           (deg; real)    ===
c                                                                    ===
c     /NDIMEN/                                                       ===
c                                                                    ===
c        IM        number of vertical levels (integer).              ===
c        JM        number of vertical levels (integer).              ===
c        KM        number of vertical levels (integer).              ===
c                                                                    ===
c     /PI_NETCFD/                                                    ===
c                                                                    ===
c        NCGRDID   NetCDF ID for GRIDS file (integer).               ===
c        NCGRDFL   Flag for state of NetCDF GRIDS file (integer).    ===
c                                                                    ===
c     /SWITCHES/                                                     ===
c                                                                    ===
c        ICOAST    Coastal setup switch, see IFLAG(10), (integer).   ===
c        IFLAG     Initialization flags (integer array):             ===
c                  IFLAG(8)   hybrid coordinates interface level.    ===
c                  IFLAG(9)   hybrid coordinates transformation:     ===
c                             [0] single, [1] double                 ===
c                  IFLAG(10)  coastal setup switch:                  ===
c                             [0] none, [1] coast, no islands        ===
c                             [2] coasts & islands.                  ===
c                                                                    ===
c     /ZDAT/                                                         ===
c                                                                    ===
c        H         bottom topography at tracer points (cm; real).    ===
c                                                                    ===
c         calls:  EXITUS,  LENGTH,  SET_PMASK                        ===
c  NetCDF calls:  NCAGTC,  NCAINQ,  NCDID,     NCDINQ,  NCOPN,       ===
c                 NCVG1C,  NCVGT,   NCVGT1,    NCVID                 ===
c                                                                    ===
c=======================================================================
c
c-----------------------------------------------------------------------
c     Define global data
c-----------------------------------------------------------------------
c
#include <param.h>
#include <grddat.h>
#include <cstseg.h>
#include <hybrid.h>
#include <iounits.h>
#include <moddat.h>
#include <ndimen.h>
#include <netcdf.inc>
#include <pi_netcdf.h>
#include <switches.h>
#include <zdat.h>
c
c-----------------------------------------------------------------------
c     Define local data
c-----------------------------------------------------------------------
c
      logical flat
      integer atype,dimid,gbgn,gend,glen,i,ios,ip,j,k
      integer count(3),start(3)
      logical old
      real    c0,c1,cm1,deg2cm,fx,m2cm,p5,pi,rearth
      real cs(mx),cst(mx)
      character*1  dum
c
      parameter (c0=0.0, c1=1.0, cm1=-1.0, m2cm=100.0, p5=0.5,
     *           rearth=6371.315e5, pi=3.14159 26535 89793 23846,
     *           deg2cm=pi*rearth/180.0)
c
      data old /.false./
c
c=======================================================================
c  Begin executable code.
c=======================================================================
c
c-----------------------------------------------------------------------
c  Open GRIDS netCDF file.  Make sure its a GRIDS netCDF file.
c-----------------------------------------------------------------------
c
      call length (fname(5),glen,gbgn,gend)
      if (glen.lt.1) then
        write (stdout,900)
        call exitus ('READ_GRIDS')
      endif
c
      call ncpopt(ncverbos)
      ncgrdid = ncopn (fname(5)(gbgn:gend),ncnowrit,rcode)
      if (rcode.eq.0) then
        ncgrdfl=1
      else
        write (stdout,910) fname(5)(gbgn:gend)
        call exitus ('READ_GRIDS')
      endif
c
      call ncainq (ncgrdid,ncglobal,'type',atype,ltype,rcode)
      if (rcode.eq.0) then
        call ncagtc (ncgrdid,ncglobal,'type',type,ltype,rcode)
        if (rcode.ne.0) then
          write(stdout,920) 'attribute','(global):type',
     &                      fname(5)(gbgn:gend)
          call exitus('READ_GRIDS')
        elseif(type(1:ltype).ne.'GRIDS') then
          write(stdout,930) fname(5)(gbgn:gend)
          call exitus('READ_GRIDS')
        endif
      else
        write(stdout,940) 'attribute','(global):type',
     &                    fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
c
c-----------------------------------------------------------------------
c  Get domain dimensions.
c-----------------------------------------------------------------------
c
c  Longitudinal dimension
c
      dimid = ncdid (ncgrdid,'tlon',rcode)
      if (rcode.ne.0) then
        write(stdout,940) 'dimension','tlon',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      call ncdinq (ncgrdid,dimid,dimnam,im,rcode)
      if (rcode.ne.0) then
        write(stdout,920) 'dimension','tlon',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
c
c  Latitudinal dimension
c
      dimid = ncdid (ncgrdid,'tlat',rcode)
      if (rcode.ne.0) then
        write(stdout,940) 'dimension','tlat',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      call ncdinq (ncgrdid,dimid,dimnam,jm,rcode)
      if (rcode.ne.0) then
        write(stdout,920) 'dimension','tlat',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
c
c  Vertical dimension
c
      dimid = ncdid (ncgrdid,'level',rcode)
      if (rcode.ne.0) then
        write(stdout,940) 'dimension','level',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      call ncdinq (ncgrdid,dimid,dimnam,km,rcode)
      if (rcode.ne.0) then
        write(stdout,920) 'dimension','level',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
c
c  Check if allocated space is sufficient.
c
      if((im*jm).gt.np) then
        write (stdout,950) 'im*jm',im*jm,'np',np
        call exitus('READ_GRIDS')
      endif
      if(max(im,jm).gt.mx) then
        write (stdout,950) 'max(im,jm)',max(im,jm),'mx',mx
        call exitus('READ_GRIDS')
      endif
      if(km.gt.nz) then
        write (stdout,950) 'km',km,'nz',nz
        call exitus('READ_GRIDS')
      endif
c
c-----------------------------------------------------------------------
c  Get horizontal domain definition parameters.
c-----------------------------------------------------------------------
c
c  Coordinate type
c
      varid = ncvid (ncgrdid,'coord',rcode)
      if (rcode.ne.0) then
        write(stdout,940) 'variable','coord',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      call ncvgt1 (ncgrdid,varid,1,coord,rcode)
      if (rcode.ne.0) then
        write(stdout,920) 'variable','coord',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
       elseif ((coord.lt.0).or.(coord.gt.2)) then
        write(stdout,960) 'coord',coord
        call exitus('READ_GRIDS')
      endif
c
c  Center of transformation
c
      varid = ncvid (ncgrdid,'rlngd',rcode)
      if (rcode.ne.0) then
        write(stdout,940) 'variable','rlngd',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      call ncvgt1 (ncgrdid,varid,1,rlng0,rcode)
      if (rcode.ne.0) then
        write(stdout,920) 'variable','rlngd',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
c
      varid = ncvid (ncgrdid,'rlatd',rcode)
      if (rcode.ne.0) then
        write(stdout,940) 'variable','rlatd',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      call ncvgt1 (ncgrdid,varid,1,rlat0,rcode)
      if (rcode.ne.0) then
        write(stdout,920) 'variable','rlatd',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
c
c  Shift between centers of grid & transformation
c
      varid = ncvid (ncgrdid,'delx',rcode)
      if (rcode.eq.0) then
         call ncvgt1 (ncgrdid,varid,1,delx,rcode)
         if (rcode.ne.0) then
           write(stdout,920) 'variable','delx',fname(5)(gbgn:gend)
           call exitus('READ_GRIDS')
         endif
        else
         old = .true.
      endif
c
      varid = ncvid (ncgrdid,'dely',rcode)
      if (rcode.eq.0) then
         call ncvgt1 (ncgrdid,varid,1,dely,rcode)
         if (rcode.ne.0) then
           write(stdout,920) 'variable','dely',fname(5)(gbgn:gend)
           call exitus('READ_GRIDS')
         endif
        else
         old = .true.
      endif
c
      if (old) then
         if (coord.ne.1) then
            delx = c0
            dely = c0
            write (stdout,980)
           else
            delx  = rlng0
            dely  = rlat0
            rlng0 = c0
            rlat0 = c0
            write (stdout,990) rlng0,rlat0
         end if
      end if
c
c  Grid size.
c
      varid = ncvid (ncgrdid,'meandx',rcode)
      if (rcode.ne.0) then
        write(stdout,940) 'variable','meandx',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      call ncvgt1 (ncgrdid,varid,1,dx,rcode)
      if (rcode.ne.0) then
        write(stdout,920) 'variable','meandx',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
c
      varid = ncvid (ncgrdid,'meandy',rcode)
      if (rcode.ne.0) then
        write(stdout,940) 'variable','meandy',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      call ncvgt1 (ncgrdid,varid,1,dy,rcode)
      if (rcode.ne.0) then
        write(stdout,920) 'variable','meandy',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
c
c  Domain rotation.
c
      varid = ncvid (ncgrdid,'thetad',rcode)
      if (rcode.ne.0) then
        write(stdout,940) 'variable','thetad',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      call ncvgt1 (ncgrdid,varid,1,thetad,rcode)
      if (rcode.ne.0) then
        write(stdout,920) 'variable','thetad',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
c
c-----------------------------------------------------------------------
c  Get vertical domain definition parameters.
c-----------------------------------------------------------------------
c
c  Hybrid system parameters.
c
      varid = ncvid (ncgrdid,'kc',rcode)
      if (rcode.ne.0) then
        write(stdout,940) 'variable','kc',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      call ncvgt1 (ncgrdid,varid,1,kc,rcode)
      if (rcode.ne.0) then
        write(stdout,920) 'variable','kc',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      iflag(8)=kc
c
      varid = ncvid (ncgrdid,'flat',rcode)
      if (rcode.ne.0) then
        write(stdout,940) 'variable','flat',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      call ncvg1c (ncgrdid,varid,1,dum,rcode)
      if (rcode.ne.0) then
        write(stdout,920) 'variable','flat',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      read (dum,'(l1)',iostat=ios) flat
      if (ios.eq.0) then
        if (flat) then
          dblsigma=0
         else
          dblsigma=1
        endif
        iflag(9)=dblsigma
       else
        write (stdout,970) 'flat',dum
        call exitus('READ_GRIDS')
      endif
c
c  Coordinate interface parameters.
c
      varid = ncvid (ncgrdid,'zc1',rcode)
      if (rcode.ne.0) then
        write(stdout,940) 'variable','zc1',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      call ncvgt1 (ncgrdid,varid,1,zc1,rcode)
      if (rcode.ne.0) then
        write(stdout,920) 'variable','zc1',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
c
      varid = ncvid (ncgrdid,'zc2',rcode)
      if (rcode.ne.0) then
        write(stdout,940) 'variable','zc2',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      call ncvgt1 (ncgrdid,varid,1,zc2,rcode)
      if (rcode.ne.0) then
        write(stdout,920) 'variable','zc2',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
c
      varid = ncvid (ncgrdid,'zref',rcode)
      if (rcode.ne.0) then
        write(stdout,940) 'variable','zref',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      call ncvgt1 (ncgrdid,varid,1,zref,rcode)
      if (rcode.ne.0) then
        write(stdout,920) 'variable','zref',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
c
      varid = ncvid (ncgrdid,'zslope',rcode)
      if (rcode.ne.0) then
        write(stdout,940) 'variable','zslope',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      call ncvgt1 (ncgrdid,varid,1,zslope,rcode)
      if (rcode.ne.0) then
        write(stdout,920) 'variable','zslope',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
c
c  Vertical discretization.
c
      varid = ncvid (ncgrdid,'hz',rcode)
      if (rcode.ne.0) then
        write(stdout,940) 'variable','hz',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      call ncvgt (ncgrdid,varid,1,km,hz,rcode)
      if (rcode.ne.0) then
        write(stdout,920) 'variable','hz',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
c
c-----------------------------------------------------------------------
c  Read in metric factors.
c-----------------------------------------------------------------------
c
c  At velocity points.
c
      varid = ncvid (ncgrdid,'cs',rcode)
      if (rcode.ne.0) then
        write(stdout,940) 'variable','cs',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      call ncvgt (ncgrdid,varid,1,jm,cs,rcode)
      if (rcode.ne.0) then
        write(stdout,920) 'variable','cs',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
c
c  At tracer points.
c
      varid = ncvid (ncgrdid,'cst',rcode)
      if (rcode.ne.0) then
        write(stdout,940) 'variable','cst',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
      call ncvgt (ncgrdid,varid,1,jm,cst,rcode)
      if (rcode.ne.0) then
        write(stdout,920) 'variable','cst',fname(5)(gbgn:gend)
        call exitus('READ_GRIDS')
      endif
c
c-----------------------------------------------------------------------
c  Read in geographic latitudes, if appropriate.
c-----------------------------------------------------------------------
c
      if ((iflag(6).eq.2).or.(iflag(6).eq.3)) then
        start(1)=2
        count(1)=1
        start(2)=1
        count(2)=im
        start(3)=1
        count(3)=jm
        varid = ncvid (ncgrdid,'tgrid2',rcode)
        if (rcode.ne.0) then
          write(stdout,940) 'variable','tgrid2',fname(5)(gbgn:gend)
          call exitus('READ_GRIDS')
        endif
        call ncvgt (ncgrdid,varid,start,count,geolat,rcode)
        if (rcode.ne.0) then
          write(stdout,920) 'variable','tgrid2',fname(5)(gbgn:gend)
          call exitus('READ_GRIDS')
        endif
      endif
c
c-----------------------------------------------------------------------
c  Read in bottom topography, if appropriate.
c-----------------------------------------------------------------------
c
      if (iflag(7).eq.1) then
        start(1)=1
        count(1)=im
        start(2)=1
        count(2)=jm
        varid = ncvid (ncgrdid,'tbath',rcode)
        if (rcode.ne.0) then
          write(stdout,940) 'variable','tbath',fname(5)(gbgn:gend)
          call exitus('READ_GRIDS')
        endif
        call ncvgt (ncgrdid,varid,start,count,h,rcode)
        if (rcode.ne.0) then
          write(stdout,920) 'variable','tbath',fname(5)(gbgn:gend)
          call exitus('READ_GRIDS')
        endif
#ifndef gridold
c
        start(1)=1
        count(1)=im
        start(2)=1
        count(2)=jm
        call ncpopt (0)
        varid = ncvid (ncgrdid,'vbath',rcode)
        call ncpopt (ncverbos)
        if (rcode.eq.0) then
           call ncvgt (ncgrdid,varid,start,count,hv,rcode)
           if (rcode.ne.0) then
             write(stdout,920) 'variable','vbath',fname(5)(gbgn:gend)
             call exitus('READ_GRIDS')
           endif
         else
           call setvgd (im,jm,h,hv)
        endif
c
        start(1)=1
        count(1)=im
        start(2)=1
        count(2)=jm
        varid = ncvid (ncgrdid,'interface',rcode)
        if (rcode.ne.0) then
          write(stdout,940) 'variable','interface',fname(5)(gbgn:gend)
          call exitus('READ_GRIDS')
        endif
        call ncvgt (ncgrdid,varid,start,count,tinter,rcode)
        if (rcode.ne.0) then
          write(stdout,920) 'variable','interface',fname(5)(gbgn:gend)
          call exitus('READ_GRIDS')
        endif
c
        start(1)=1
        count(1)=im
        start(2)=1
        count(2)=jm
        call ncpopt (0)
        varid = ncvid (ncgrdid,'viface',rcode)
        call ncpopt (ncverbos)
        if (rcode.eq.0) then
           call ncvgt (ncgrdid,varid,start,count,vinter,rcode)
           if (rcode.ne.0) then
             write(stdout,920) 'variable','viface',fname(5)(gbgn:gend)
             call exitus('READ_GRIDS')
           endif
         else
           call setvgd (im,jm,tinter,vinter)
        endif
#endif
      endif
c
c-----------------------------------------------------------------------
c  Read in coastal information.
c-----------------------------------------------------------------------
c
c  Look for number of coastal segments.  Use to determine if
c  any land masking is present.
c
      call ncpopt(0)
      varid = ncvid (ncgrdid,'ncseg',rcode)
      call ncpopt(ncverbos)
      if (rcode.eq.0) icoast=1
      iflag(10)=icoast
c
c  If land masking present, read necessary data.
c
      if (icoast.eq.1) then
c
c    Coastal segments.
c
        call ncvgt1 (ncgrdid,varid,1,nseg,rcode)
        if (rcode.ne.0) then
          write(stdout,920) 'variable','ncseg',fname(5)(gbgn:gend)
          call exitus('READ_GRIDS')
        endif
        if (nseg.gt.mcseg) then
          write (stdout,950) 'ncseg',nseg,'mcseg',mcseg
          call exitus('READ_GRIDS')
        endif
c
        varid = ncvid (ncgrdid,'lcseg',rcode)
        if (rcode.ne.0) then
          write(stdout,940) 'variable','lcseg',fname(5)(gbgn:gend)
          call exitus('READ_GRIDS')
        endif
        call ncvgt1 (ncgrdid,varid,1,lsegmx,rcode)
        if (rcode.ne.0) then
          write(stdout,920) 'variable','lcseg',fname(5)(gbgn:gend)
          call exitus('READ_GRIDS')
        endif
        if (lsegmx.gt.mpseg) then
          write (stdout,950) 'lcseg',lsegmx,'mpseg',mpseg
          call exitus('READ_GRIDS')
        endif
c
        varid = ncvid (ncgrdid,'lencoast',rcode)
        if (rcode.ne.0) then
          write(stdout,940) 'variable','lencoast',fname(5)(gbgn:gend)
          call exitus('READ_GRIDS')
        endif
        call ncvgt (ncgrdid,varid,1,nseg,lenseg,rcode)
        if (rcode.ne.0) then
          write(stdout,920) 'variable','lencoast',fname(5)(gbgn:gend)
          call exitus('READ_GRIDS')
        endif
c
        start(1)=1
        count(2)=1
        do 10 k = 1,nseg
          count(1)=lenseg(k)
          start(2)=k
c
          varid = ncvid (ncgrdid,'icoast',rcode)
          if (rcode.ne.0) then
            write(stdout,940) 'variable','icoast',fname(5)(gbgn:gend)
            call exitus('READ_GRIDS')
          endif
          call ncvgt (ncgrdid,varid,start,count,cstsegx(1,k),rcode)
          if (rcode.ne.0) then
            write(stdout,920) 'variable','icoast',fname(5)(gbgn:gend)
            call exitus('READ_GRIDS')
          endif
c
          varid = ncvid (ncgrdid,'jcoast',rcode)
          if (rcode.ne.0) then
            write(stdout,940) 'variable','jcoast',fname(5)(gbgn:gend)
            call exitus('READ_GRIDS')
          endif
          call ncvgt (ncgrdid,varid,start,count,cstsegy(1,k),rcode)
          if (rcode.ne.0) then
            write(stdout,920) 'variable','jcoast',fname(5)(gbgn:gend)
            call exitus('READ_GRIDS')
          endif
  10    continue
c
c    Land mask.
c
        start(1)=1
        count(1)=im
        start(2)=1
        count(2)=jm
        varid = ncvid (ncgrdid,'landt',rcode)
        if (rcode.ne.0) then
          write(stdout,940) 'variable','landt',fname(5)(gbgn:gend)
          call exitus('READ_GRIDS')
        endif
        call ncvgt (ncgrdid,varid,start,count,landt,rcode)
        if (rcode.ne.0) then
          write(stdout,920) 'variable','landt',fname(5)(gbgn:gend)
          call exitus('READ_GRIDS')
        endif
c
        call set_pmask (im,jm,nseg,lenseg,cstsegx,cstsegy,landt,
     &                  landp)
c
c    Determine if any islands are present.
c
        call ncpopt(0)
        varid = ncvid (ncgrdid,'nisle',rcode)
        call ncpopt(ncverbos)
c
c    Read island data, if any.
c
        if (rcode.eq.0) then
c
          call ncvgt1 (ncgrdid,varid,1,nisle,rcode)
          if (rcode.ne.0) then
            write(stdout,920) 'variable','nisle',fname(5)(gbgn:gend)
            call exitus('READ_GRIDS')
          endif
          if (nisle.gt.misle) then
            write (stdout,950) 'nisle',nisle,'misle',misle
            call exitus('READ_GRIDS')
          endif
c
          varid = ncvid (ncgrdid,'isis',rcode)
          if (rcode.ne.0) then
            write(stdout,940) 'variable','isis',fname(5)(gbgn:gend)
            call exitus('READ_GRIDS')
          endif
          call ncvgt (ncgrdid,varid,1,nisle,isis,rcode)
          if (rcode.ne.0) then
            write(stdout,920) 'variable','isis',fname(5)(gbgn:gend)
            call exitus('READ_GRIDS')
          endif
c
          varid = ncvid (ncgrdid,'ieis',rcode)
          if (rcode.ne.0) then
            write(stdout,940) 'variable','ieis',fname(5)(gbgn:gend)
            call exitus('READ_GRIDS')
          endif
          call ncvgt (ncgrdid,varid,1,nisle,ieis,rcode)
          if (rcode.ne.0) then
            write(stdout,920) 'variable','ieis',fname(5)(gbgn:gend)
            call exitus('READ_GRIDS')
          endif
c
          varid = ncvid (ncgrdid,'jsis',rcode)
          if (rcode.ne.0) then
            write(stdout,940) 'variable','jsis',fname(5)(gbgn:gend)
            call exitus('READ_GRIDS')
          endif
          call ncvgt (ncgrdid,varid,1,nisle,jsis,rcode)
          if (rcode.ne.0) then
            write(stdout,920) 'variable','jsis',fname(5)(gbgn:gend)
            call exitus('READ_GRIDS')
          endif
c
          varid = ncvid (ncgrdid,'jeis',rcode)
          if (rcode.ne.0) then
            write(stdout,940) 'variable','jeis',fname(5)(gbgn:gend)
            call exitus('READ_GRIDS')
          endif
          call ncvgt (ncgrdid,varid,1,nisle,jeis,rcode)
          if (rcode.ne.0) then
            write(stdout,920) 'variable','jeis',fname(5)(gbgn:gend)
            call exitus('READ_GRIDS')
          endif
c
        endif
      endif
c
c-----------------------------------------------------------------------
c  Convert from MKS to CGS.
c-----------------------------------------------------------------------
c
      if (coord.eq.0) then
        dx   = dx*m2cm
        dy   = dy*m2cm
        delx = delx*m2cm
        dely = dely*m2cm
      endif
      fx = m2cm*cm1
      zc1=zc1*fx
      zc2=zc2*fx
      zref=zref*fx
      do 20 k=1,km
        hz(k)=hz(k)*m2cm
  20  continue
      do 30 ip=1,im*jm
        h(ip)=h(ip)*fx
#ifndef gridold
        hv(ip)=hv(ip)*fx
#endif
  30  continue
#ifndef gridold
      do 3000 ip=1,im*jm
        tinter(ip)=tinter(ip)*fx
        vinter(ip)=vinter(ip)*fx
 3000 continue
#endif
c
c-----------------------------------------------------------------------
c  Set flat level depths (cm).
c-----------------------------------------------------------------------
c
      zpe(1)=p5*hz(1)
      do 40 k=2,km
        zpe(k)=zpe(k-1)+p5*(hz(k-1)+hz(k))
  40  continue
c
c-----------------------------------------------------------------------
c  Set-up metric coefficients.
c-----------------------------------------------------------------------
c
      if(coord.eq.0) then
        do 50 j=1,jm
        do 50 i=1,im
          ip=i+(j-1)*im
          tmetx(ip)=c1
          tmety(ip)=c1
          vmetx(ip)=c1
          vmety(ip)=c1
  50    continue
      else
        do 60 j=1,jm
        do 60 i=1,im
          ip=i+(j-1)*im
          tmetx(ip)=deg2cm*cst(j)
          tmety(ip)=deg2cm
          vmetx(ip)=deg2cm*cs(j)
          vmety(ip)=deg2cm
  60    continue
      endif
c
 900  format (/'***Error:  READ_GRIDS - illegal file name (blank).')
 910  format (/'***Error:  READ_GRIDS - unable to open file ',1h",a,1h")
 920  format (/'***Error:  READ_GRIDS - unable to read ',a,1x,1h",a,1h",
     &         ' in file ',1h",a,1h")
 930  format (/'***Error:  READ_GRIDS -  File ',1h",a,1h",' was not ',
     &         'created by GRIDS.')
 940  format (/'***Error:  READ_GRIDS - unable to find ',a,1x,1h",a,1h",
     &         ' in file ',1h",a,1h")
 950  format (/'***Error:  READ_GRIDS - Underdimensioned arrays:',
     &        2(/14x,a,' = ',i10)/14x,'Change ',1h",'param.h',1h",
     &        ' and recompile.')
 960  format (/'***Error:  READ_GRIDS - Illegal value: ',1h",a,1h",' = '
     &        ,i10)
 970  format (/'***Error:  READ_GRIDS - Illegal value: ',1h",a,1h",' = '
     &        ,a)
 980  format (/'+++Warning:  READ_GRIDS - old GRIDS file.'/13x,
     &        'Replacing GRIDS values (DELX,DELY) by (0,0)')
 990  format (/'+++Warning:  READ_GRIDS - old GRIDS file.'/13x,
     &        'Replacing GRIDS values (DELX,DELY) by (',f8.3,', ',f8.3,
     &        ')'/13x,'and GRIDS values (RLNG0,RLAT0) by (0,0)')
c
      return
      end
