function [ncid,tlon,tlat,vlon,vlat,zout] = defcdf (run_file,run_parm,did,sid);
%
% function [ncid,tlon,tlat,vlon,vlat,zout] = defcdf (run_file,run_parm,did,sid);
%
% This function defines a reference surface netCDF output file.
%
% ------
% Input:
% ------
%
%    RUN_FILE...Datafiles governing run.
%                  RUN_FILE(1,:) - DEEP_FILE      Deep ICON netCDF file.
%                  RUN_FILE(2,:) - SHALLOW_FILE   Shallow ICON netCDF file.
%                  RUN_FILE(3,:) - GRID_FILE      GRIDS netCDF file.
%                  RUN_FILE(4,:) - OUT_FILE       Output netCDF file, (melded).
%    RUN_PARM...Parameters governing run.
%                  RUN_PARM( 1) - EMIN     Minimum relative error.
%                  RUN_PARM( 2) - ETRANS   Error transition width.
%    DID........NetCDF file identifers for DEEP_FILE.
%    SID........NetCDF file identifers for SHALLOW_FILE.
%
% -------
% Output:
% -------
%
%    NCID.....NetCDF file identifier.  If NCID<0,
%             an error has occurred and the file may
%             not have been created and will certainly
%             be closed.
%    TLON.....Domain tracer grid longitudes.
%    TLAT.....Domain tracer grid latitudes.
%    VLON.....Domain velocity grid longitudes.
%    VLAT.....Domain velocity grid latitudes.
%    ZOUT.....OA/FM levels.

%-------------------------------------------------------------------------------
% Initialize output.
%-------------------------------------------------------------------------------

lon  = [];
lat  = [];
zout = [];

%-------------------------------------------------------------------------------
% Define special values.
%-------------------------------------------------------------------------------

spval = 1.0e35;

%-------------------------------------------------------------------------------
% Create netCDF file.
%-------------------------------------------------------------------------------

ind    = find(abs(run_file(4,:))~=0);
ncname = run_file(4,ind);

ncid = netcdf (ncname,'clobber');

if isempty(ncid),
   disp (' ');
   disp ('***Error:  DEFCDF - unable to create file:');
   disp (['           ',setstr(34),ncname,setstr(34)]);
   disp (' ');
   return
end;

%-------------------------------------------------------------------------------
% Get domain characteristics.
%-------------------------------------------------------------------------------

%-----------------------------------
%--- Horizontal Grid Parameters. ---
%-----------------------------------

ind   = find(abs(run_file(3,:))~=0);
gname = run_file(3,ind);

[domdat,bath,mask,mcseg,mpseg,misle] = read_mask (gname);

coord = domdat(1);
lonc  = domdat(2);
latc  = domdat(3);
theta = domdat(4);
nx    = domdat(5);
ny    = domdat(6);
dx    = domdat(7);
dy    = domdat(8);
delx  = domdat(9);
dely  = domdat(10);
ismask = max(mask(:)) > 0;

clear domdat bath mcseg mpseg misle

[tlon,tlat,topo,vlon,vlat,dxt,dyt,metxt,metyt, ...
          dxu,dyu,metxu,metyu,vertparm,status] = gridsread (gname);

if (status<0),
   disp (' ');
   disp ('***Error:  DEFCDF - unable to read file:');
   disp (['           ',setstr(34),gname,setstr(34)]);
   disp (' ');
   dum = mexcdf ('ncabort',ncid);
   ncid = status;
   return
end;

clear topo dxt dyt metxt metyt dxu dyu metxu metyu vertparm status

%---------------------------------
%--- Vertical Grid Parameters. ---
%---------------------------------

name = 'zout';

ind   = find(abs(run_file(1,:))~=0);
oname = run_file(1,ind);

z1 = did{name}(:);
if isempty(z1),
   disp (' ');
   disp ('***Error:  DEFCDF - unable to read file:');
   disp (['           ',setstr(34),oname,setstr(34)]);
   disp (' ');
   dum = mexcdf ('ncabort',ncid);
   ncid = [];
   return
end;

ind   = find(abs(run_file(2,:))~=0);
oname = run_file(2,ind);

z2 = sid{name}(:);
if isempty(z2),
   disp (' ');
   disp ('***Error:  DEFCDF - unable to read file:');
   disp (['           ',setstr(34),oname,setstr(34)]);
   disp (' ');
   dum = mexcdf ('ncabort',ncid);
   ncid = [];
   return
end;

zout = unique([z1 z2]);
nz   = length(zout);

clear z1 z2;

%-------------------------------------------------------------------------------
% Define dimensions.
%-------------------------------------------------------------------------------

%----------------------------------
%--- Define spatial dimensions. ---
%----------------------------------

name = 'lon';
ncid(name) = nx;
if isempty(ncid(name)),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define dimension ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

name = 'lat';
ncid(name) = ny;
if isempty(ncid(name)),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define dimension ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

name = 'level';
ncid(name) = nz;
if isempty(ncid(name)),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define dimension ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

%----------------------------------
%--- Define temporal dimension. ---
%----------------------------------

name = 'time';
ncid(name) = 0;
if isempty(ncid(name)),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define dimension ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

%------------------------------------
%--- Define auxillary dimensions. ---
%------------------------------------

if (ismask)
   name = 'axis2';
   ncid(name) = 2;
   if isempty(ncid(name)),
      disp (' ');
      disp (['***Error:  DEFCDF - unable to define dimension ',setstr(34), ...
                                             name,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncname,setstr(34),'.']);
      disp (' ');
      dum = abort (ncid);
      ncid = [];
      return;
   end;
end;

name = 'axis3';
ncid(name) = 3;
if isempty(ncid(name)),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define dimension ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

name = 'vector';
ncid(name) = 2;
if isempty(ncid(name)),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define dimension ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

%===============================================================================
% Define variables.
%===============================================================================

%-------------------------------------------------------------------------------
% Define horizontal domain definition.
%-------------------------------------------------------------------------------

%------------------------
%--- Coordinate type. ---
%------------------------

name  = 'coord';
ncid{name} = nclong;
if isempty(ncid{name}),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'long_name';
attval  = 'Coordinate type flag';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'option_0';
attval  = 'Cartesian';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'option_1';
attval  = 'unrotated spherical';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'option_2';
attval  = 'rotated spherical';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

%---------------------
%--- Grid Spacing. ---
%---------------------

name  = 'dx';
ncid{name} = ncfloat;
if isempty(ncid{name}),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'long_name';
attval  = 'zonal grid spacing';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'units';
if (coord==0)
   attval  = 'meter';
 else
   attval  = 'degrees';
end;
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

name  = 'dy';
ncid{name} = ncfloat;
if isempty(ncid{name}),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'long_name';
attval  = 'meridional grid spacing';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'units';
if (coord==0)
   attval  = 'meter';
 else
   attval  = 'degrees';
end;
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

%--------------------------------
%--- Transformation centroid. ---
%--------------------------------

name  = 'rlngd';
ncid{name} = ncfloat;
if isempty(ncid{name}),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'long_name';
attval  = 'Transformation centroid longitude';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'units';
attval  = 'degrees_east';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

name  = 'rlatd';
ncid{name} = ncfloat;
if isempty(ncid{name}),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'long_name';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'units';
attval  = 'degrees_north';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

%--------------------
%--- Grid Offset. ---
%--------------------

name  = 'delx';
ncid{name} = ncfloat;
if isempty(ncid{name}),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'long_name';
attval  = 'Longitudinal offset from transformation point to grid center';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'units';
if (coord==0)
   attval  = 'meter';
 else
   attval  = 'degrees';
end;
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

name  = 'dely';
ncid{name} = ncfloat;
if isempty(ncid{name}),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'long_name';
attval  = 'Latitudinal offset from transformation point to grid center';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'units';
if (coord==0)
   attval  = 'meter';
 else
   attval  = 'degrees';
end;
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

%----------------------
%--- Grid Rotation. ---
%----------------------

name  = 'thetad';
ncid{name} = ncfloat;
if isempty(ncid{name}),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'long_name';
attval  = 'domain rotation angle';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'units';
attval  = 'degrees';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

%-------------------------------------------------------------------------------
% Define run parameters.
%-------------------------------------------------------------------------------

%----------------------
%--- Minimum error. ---
%----------------------

name  = 'emin';
ncid{name} = ncfloat;
if isempty(ncid{name}),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'long_name';
attval  = 'Minimum relative error';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

%-------------------------------
%--- Error transition width. ---
%-------------------------------

name  = 'etrans';
ncid{name} = nclong;
if isempty(ncid{name}),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'long_name';
attval  = 'Error transition width around missing values.';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'units';
attval  = 'grid points';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

%-------------------------------------------------------------------------------
% Define spatial positions.
%-------------------------------------------------------------------------------

%---------------------------
%--- Vertical Positions. ---
%---------------------------

name   = 'zout';
ncid{name} = ncfloat('level','axis3');
if isempty(ncid{name}),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'long_name';
attval  = 'depths at center of the grid vertical boxes';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'axis3';
attval  = '1: longitude, 2: latitude, 3: depth';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'units';
attval  = 'degrees_east, degrees_north, meter';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'FillValue_';
attval  = spval;
eval(['ncid{name}.',attname,' = ncfloat(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

%---------------------
%--- 2D Positions. ---
%---------------------

if (ismask)

   name   = 'grid2';
   ncid{name} = ncfloat('lat','lon','axis2');
   if isempty(ncid{name}),
      disp (' ');
      disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncname,setstr(34),'.']);
      disp (' ');
      dum = abort (ncid);
      ncid = [];
      return;
   end;

   attname = 'long_name';
   attval  = '2D grid positions';
   eval(['ncid{name}.',attname,' = ncchar(attval);']);
   eval(['status = isempty(ncid{name}.',attname,');']);
   if (status),
      disp (' ');
      disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncname,setstr(34),'.']);
      disp (' ');
      dum = abort (ncid);
      ncid = [];
      return;
   end;

   attname = 'axis';
   attval  = '1: longitude, 2: latitude';
   eval(['ncid{name}.',attname,' = ncchar(attval);']);
   eval(['status = isempty(ncid{name}.',attname,');']);
   if (status),
      disp (' ');
      disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncname,setstr(34),'.']);
      disp (' ');
      dum = abort (ncid);
      ncid = [];
      return;
   end;

   attname = 'units';
   attval  = 'degrees_east, degrees_north';
   eval(['ncid{name}.',attname,' = ncchar(attval);']);
   eval(['status = isempty(ncid{name}.',attname,');']);
   if (status),
      disp (' ');
      disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncname,setstr(34),'.']);
      disp (' ');
      dum = abort (ncid);
      ncid = [];
      return;
   end;

   attname = 'FillValue_';
   attval  = spval;
   eval(['ncid{name}.',attname,' = ncfloat(attval);']);
   eval(['status = isempty(ncid{name}.',attname,');']);
   if (status),
      disp (' ');
      disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncname,setstr(34),'.']);
      disp (' ');
      dum = abort (ncid);
      ncid = [];
      return;
   end;

end;

%---------------------
%--- 3D Positions. ---
%---------------------

name   = 'grid3';
ncid{name} = ncfloat('lat','lon','axis3');
if isempty(ncid{name}),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'long_name';
attval  = '3D grid positions';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'axis3';
attval  = '1: longitude, 2: latitude, 3: depth';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'units';
attval  = 'degrees_east, degrees_north, meter';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'FillValue_';
attval  = spval;
eval(['ncid{name}.',attname,' = ncfloat(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

%------------------------------
%--- Velocity 3D Positions. ---
%------------------------------

name   = 'vgrid3';
ncid{name} = ncfloat('lat','lon','axis3');
if isempty(ncid{name}),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'long_name';
attval  = '3D grid positions at velocity points';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'axis3';
attval  = '1: longitude, 2: latitude, 3: depth';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'units';
attval  = 'degrees_east, degrees_north, meter';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'FillValue_';
attval  = spval;
eval(['ncid{name}.',attname,' = ncfloat(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

%-------------
%--- Mask. ---
%-------------

if (ismask)

   name   = 'mask';
   ncid{name} = nclong('lat','lon');
   if isempty(ncid{name}),
      disp (' ');
      disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncname,setstr(34),'.']);
      disp (' ');
      dum = abort (ncid);
      ncid = [];
      return;
   end;

   attname = 'long_name';
   attval  = 'land/sea mask';
   eval(['ncid{name}.',attname,' = ncchar(attval);']);
   eval(['status = isempty(ncid{name}.',attname,');']);
   if (status),
      disp (' ');
      disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncname,setstr(34),'.']);
      disp (' ');
      dum = abort (ncid);
      ncid = [];
      return;
   end;

   attname = 'option_0';
   attval  = 'land';
   eval(['ncid{name}.',attname,' = ncchar(attval);']);
   eval(['status = isempty(ncid{name}.',attname,');']);
   if (status),
      disp (' ');
      disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncname,setstr(34),'.']);
      disp (' ');
      dum = abort (ncid);
      ncid = [];
      return;
   end;

   attname = 'option_1';
   attval  = 'sea';
   eval(['ncid{name}.',attname,' = ncchar(attval);']);
   eval(['status = isempty(ncid{name}.',attname,');']);
   if (status),
      disp (' ');
      disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncname,setstr(34),'.']);
      disp (' ');
      dum = abort (ncid);
      ncid = [];
      return;
   end;

   attname = 'field';
   attval  = 'mask, scalar';
   eval(['ncid{name}.',attname,' = ncchar(attval);']);
   eval(['status = isempty(ncid{name}.',attname,');']);
   if (status),
      disp (' ');
      disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncname,setstr(34),'.']);
      disp (' ');
      dum = abort (ncid);
      ncid = [];
      return;
   end;

   attname = 'positions';
   attval  = 'grid2';
   eval(['ncid{name}.',attname,' = ncchar(attval);']);
   eval(['status = isempty(ncid{name}.',attname,');']);
   if (status),
      disp (' ');
      disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncname,setstr(34),'.']);
      disp (' ');
      dum = abort (ncid);
      ncid = [];
      return;
   end;

   attname = 'FillValue_';
   attval  = -1;
   eval(['ncid{name}.',attname,' = nclong(attval);']);
   eval(['status = isempty(ncid{name}.',attname,');']);
   if (status),
      disp (' ');
      disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
      disp (['           ',setstr(34),ncname,setstr(34),'.']);
      disp (' ');
      dum = abort (ncid);
      ncid = [];
      return;
   end;

end;

%-------------------------------------------------------------------------------
% Define time.
%-------------------------------------------------------------------------------

name  = 'time';
ncid{name} = ncfloat('time');
if isempty(ncid{name}),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'long_name';
attval  = 'estimate time';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'units';
attval  = 'modified Julian day';
eval(['ncid{name}.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'add_offset';
attval  = 2440000;
eval(['ncid{name}.',attname,' = ncfloat(attval);']);
eval(['status = isempty(ncid{name}.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                             name,':',attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

%-------------------------------------------------------------------------------
% Define fields.
%-------------------------------------------------------------------------------

name = 'temp';
dims = {'time' 'lat' 'lon' 'level'};
dims2 = {'time' 'level'};
status = def_flds (ncid,name,ncname,dims,dims2, ...
                   'temperature','Celsius','temperature','grid3',ismask,spval);
if isempty(status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

name = 'salt';
status = def_flds (ncid,name,ncname,dims,dims2, ...
                   'salinity','PSU','salinity','grid3',ismask,spval);
if isempty(status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

name = 'vtot';
dims = {'time' 'lat' 'lon' 'level' 'vector'};
dims2 = {'time' 'level' 'vector'};
status = def_flds (ncid,name,ncname,dims,dims2, ...
                   'total velocity','centimeter second-1', ...
                   'velocity','vgrid3',ismask,spval);
if isempty(status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define variable ',setstr(34), ...
                                          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

%-------------------------------------------------------------------------------
% Define global attributes.
%-------------------------------------------------------------------------------

attname = 'deep_file';
ind     = find (abs(run_file(1,:))~=0);
attval  = run_file(1,ind);
eval(['ncid.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define global attribute ', ...
                             setstr(34),attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'shallow_file';
ind     = find (abs(run_file(2,:))~=0);
attval  = run_file(2,ind);
eval(['ncid.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define global attribute ', ...
                             setstr(34),attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'grids_file';
ind     = find (abs(run_file(3,:))~=0);
attval  = run_file(3,ind);
eval(['ncid.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define global attribute ', ...
                             setstr(34),attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'type';
attval  = 'FM MODEL';
eval(['ncid.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define global attribute ', ...
                             setstr(34),attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'version';
attval  = '1.1 (Matlab)';
eval(['ncid.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define global attribute ', ...
                             setstr(34),attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

attname = 'history';
attval  = ['icon2hops, version 1.1, ',date_stamp];
eval(['ncid.',attname,' = ncchar(attval);']);
eval(['status = isempty(ncid.',attname,');']);
if (status),
   disp (' ');
   disp (['***Error:  DEFCDF - unable to define global attribute ', ...
                             setstr(34),attname,setstr(34),' in file:']);
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

%===============================================================================
% Leave definition mode.
%===============================================================================

status = endef (ncid);

if isempty(status),
   disp (' ');
   disp ('***Error:  DEFCDF - unable to end definition of file:');
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = abort (ncid);
   ncid = [];
   return;
end;

%===============================================================================
% Write data.
%===============================================================================

%-------------------------------------------------------------------------------
% Write domain definition variables.
%-------------------------------------------------------------------------------

%------------------------
%--- Coordinate type. ---
%------------------------

name  = 'coord';
ncid{name}(:) = round(coord);

%---------------------
%--- Grid Spacing. ---
%---------------------

name  = 'dx';
ncid{name}(:) = dx;

name  = 'dy';
ncid{name}(:) = dy;

%--------------------------------
%--- Transformation centroid. ---
%--------------------------------

name  = 'rlngd';
ncid{name}(:) = lonc;

name  = 'rlatd';
ncid{name}(:) = latc;

%--------------------
%--- Grid Offset. ---
%--------------------

name  = 'delx';
ncid{name}(:) = delx;

name  = 'dely';
ncid{name}(:) = dely;

%----------------------
%--- Grid Rotation. ---
%----------------------

name  = 'thetad';
ncid{name}(:) = theta;

%-------------------------------------------------------------------------------
% Write run parameters.
%-------------------------------------------------------------------------------

%--------------------
%--- Front Width. ---
%--------------------

name  = 'emin';
ncid{name}(:) = run_parm(1);

%----------------------------
%--- Minimum Front Slope. ---
%----------------------------

name  = 'etrans';
ncid{name}(:) = round(run_parm(2));

%-------------------------------------------------------------------------------
% Write positions.
%-------------------------------------------------------------------------------

%---------------------------
%--- Vertical Positions. ---
%---------------------------

name   = 'zout';
ncid{name}(:,3) = -zout;

%-----------------------------
%--- Horizontal Positions. ---
%-----------------------------

name   = 'grid3';
ncid{name}(:,:,1) = tlon;
ncid{name}(:,:,2) = tlat;

name   = 'vgrid3';
ncid{name}(:,:,1) = vlon;
ncid{name}(:,:,2) = vlat;

if (ismask)
   name   = 'grid2';
   ncid{name}(:,:,1) = tlon;
   ncid{name}(:,:,2) = tlat;
end;

%-------------------------------------------------------------------------------
% Land mask.
%-------------------------------------------------------------------------------

if (ismask)
   name   = 'mask';
   ncid{name}(:) = mask;
end;

%tlon = tlon';
%tlat = tlat';
%vlon = vlon';
%vlat = vlat';

status = sync (ncid);

if isempty(status)
   disp (' ');
   disp ('***Error:  DEFCDF - unable to synchronize file:');
   disp (['           ',setstr(34),ncname,setstr(34),'.']);
   disp (' ');
   dum = close (ncid);
   ncid = status;
   return;
end;
