function [tlon,tlat,topo,vlon,vlat,dxt,dyt,metxt,metyt, ...
          dxu,dyu,metxu,metyu,vertparm,status] = gridsread (gridsfile);
%
% function [tlon,tlat,topo,vlon,vlat,dxt,dyt,metxt,metyt, ...
%           dxu,dyu,metxu,metyu,vertparm,status] = gridsread (gridsfile);
%
% This function sets up the geometric parameters needed for the topographic
% conditioning.
%
% ------
% Input:
% ------
%
%    GRIDSFILE   Name of GRIDS netCDF file.
%
% -------
% Output:
% -------
%
%    TLON        Longitudes of tracer grid (topography).
%    TLAT        Latitudes of tracer grid (topography).
%    TOPO        Initial topography.
%    VLON        Longitudes of velocity grid (slope factor).
%    VLAT        Latitudes of velocity grid (slope factor).
%    DXT         Zonal grid spacing at tracer points.
%    DYT         Meridional grid spacing at tracer points.
%    METXT       Zonal metric factors at tracer points.
%    METYT       Meridional metric factors at tracer points.
%    DXU         Zonal grid spacing at velocity points.
%    DYU         Meridional grid spacing at velocity points.
%    METXU       Zonal metric factors at velocity points.
%    METYU       Meridional metric factors at velocity points.
%    VERTPARM    Parameters to describe double sigma system.
%                   vertparm(1):  ZC1     shallow bound
%                   vertparm(2):  ZC2     Deep bound
%                   vertparm(3):  ZREF    Reference depth.
%                   vertparm(4):  ZSLOPE  Slope bounding factor
%                   vertparm(5):  SIGRES  A measure of reduced sigma thickness
%    STATUS      Exit status.  [0] no error  [<0] error

%-------------------------------------------------------------------------------
% Set useful parameters.
%-------------------------------------------------------------------------------

status    = mexcdf ('setopts',0);
nc_nowrite = mexcdf ('parameter','nc_nowrite');

%-------------------------------------------------------------------------------
% Open GRIDS netCDF file.
%-------------------------------------------------------------------------------

ncid = mexcdf ('ncopen',gridsfile,nc_nowrite);

if (ncid<0),
   disp (' ');
   disp ('***Error:  GRIDSREAD - unable to open GRIDS netCDF file:');
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   status = ncid;
   return
end;

%-------------------------------------------------------------------------------
% Read dimensions.
%-------------------------------------------------------------------------------

%-------------------------------
%--- Number of zonal points. ---
%-------------------------------

name        = 'tlon';
[nx,status] = get_dim (ncid,name);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read dimension ',setstr(34), ...
                              name,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

%------------------------------------
%--- Number of meridional points. ---
%------------------------------------

name        = 'tlat';
[ny,status] = get_dim (ncid,name);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read dimension ',setstr(34), ...
                              name,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

%-------------------------------
%--- Number of model levels. ---
%-------------------------------

name        = 'level';
[nz,status] = get_dim (ncid,name);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read dimension ',setstr(34), ...
                              name,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

%-------------------------------------------------------------------------------
% Read topography and positions.
%-------------------------------------------------------------------------------

%---------------------------
%--- Current topography. ---
%---------------------------

name          = 'tbath';
[topo,status] = get_var (ncid,name,[0 0],[ny nx]);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read variable ',setstr(34), ...
                              name,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

topo = topo';

%-------------------
%--- Longitudes. ---
%-------------------

name         = 'tgrid2';
comp         = 'lon';
[tlon,status] = get_var (ncid,name,[0 0 0],[ny nx 1]);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read variable ',setstr(34), ...
                    name,':',comp,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

tlon = tlon';

name         = 'vgrid2';
comp         = 'lon';
[vlon,status] = get_var (ncid,name,[0 0 0],[ny nx 1]);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read variable ',setstr(34), ...
                    name,':',comp,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

vlon = vlon';

%------------------
%--- Latitudes. ---
%------------------

name         = 'tgrid2';
comp         = 'lat';
[tlat,status] = get_var (ncid,name,[0 0 1],[ny nx 1]);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read variable ',setstr(34), ...
                    name,':',comp,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

tlat = tlat';

name         = 'vgrid2';
comp         = 'lat';
[vlat,status] = get_var (ncid,name,[0 0 1],[ny nx 1]);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read variable ',setstr(34), ...
                    name,':',comp,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

vlat = vlat';

%-------------------------------------------------------------------------------
% Read discretization.
%-------------------------------------------------------------------------------

%-------------------------------------
%--- Zonal spacing: tracer points. ---
%-------------------------------------

name          = 'dxt';
[dxt1D,status] = get_var (ncid,name,0,nx);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read variable ',setstr(34), ...
                              name,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

dxt1D = dxt1D';

%---------------------------------------
%--- Zonal spacing: velocity points. ---
%---------------------------------------

name          = 'dxu';
[dxu1D,status] = get_var (ncid,name,0,nx);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read variable ',setstr(34), ...
                              name,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

dxu1D = dxu1D';

%------------------------------------------
%--- Meridional spacing: tracer points. ---
%------------------------------------------

name          = 'dyt';
[dyt1D,status] = get_var (ncid,name,0,ny);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read variable ',setstr(34), ...
                              name,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

%--------------------------------------------
%--- Meridional spacing: velocity points. ---
%--------------------------------------------

name          = 'dyu';
[dyu1D,status] = get_var (ncid,name,0,ny);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read variable ',setstr(34), ...
                              name,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

%--------------------------------------
%--- Cosine metrics: tracer points. ---
%--------------------------------------

name         = 'cst';
[cst,status] = get_var (ncid,name,0,ny);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read variable ',setstr(34), ...
                              name,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

%----------------------------------------
%--- Cosine metrics: velocity points. ---
%----------------------------------------

name        = 'cs';
[cs,status] = get_var (ncid,name,0,ny);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read variable ',setstr(34), ...
                              name,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

%-------------------------------------------------------------------------------
% Read vertical parameterization.
%-------------------------------------------------------------------------------

%----------------------------
%--- Number upper levels. ---
%----------------------------

name        = 'kc';
[kc,status] = get_var1 (ncid,name,0);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read variable ',setstr(34), ...
                              name,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

%-------------------------
%--- Vertical spacing. ---
%-------------------------

name          = 'hz';
[dsig,status] = get_var (ncid,name,kc,(nz-kc));

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read variable ',setstr(34), ...
                              name,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

dsig = dsig';

%----------------------------------
%--- Upper bound for interface. ---
%----------------------------------

name        = 'zc1';
[zc1,status] = get_var1 (ncid,name,0);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read variable ',setstr(34), ...
                              name,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

%----------------------------------
%--- Lower bound for interface. ---
%----------------------------------

name        = 'zc2';
[zc2,status] = get_var1 (ncid,name,0);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read variable ',setstr(34), ...
                              name,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

%--------------------------------------
%--- Reference depth for interface. ---
%--------------------------------------

name          = 'zref';
[zref,status] = get_var1 (ncid,name,0);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read variable ',setstr(34), ...
                              name,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

%--------------------------------------
%--- Slope parameter for interface. ---
%--------------------------------------

name            = 'zslope';
[zslope,status] = get_var1 (ncid,name,0);

if (status<0),
   disp (' ');
   disp (['***Error:  GRIDSREAD - unable to read variable ',setstr(34), ...
                              name,setstr(34),' in GRIDS netCDF file:']);
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

%-------------------------------------------------------------------------------
% Close GRIDS netCDF file.
%-------------------------------------------------------------------------------

status = mexcdf ('ncclose',ncid);

if (status<0),
   disp (' ');
   disp ('***Error:  GRIDSREAD - unable to close GRIDS netCDF file:');
   disp (['           ',setstr(34),gridsfile,setstr(34)]);
   disp (' ');
   return
end;

%-------------------------------------------------------------------------------
% Compute 2D metrics.
%-------------------------------------------------------------------------------

dxt = ones(ny,1)*dxt1D;
dxu = ones(ny,1)*dxu1D;

dyt = dyt1D*ones(1,nx);
dyu = dyu1D*ones(1,nx);

metxt = cst*ones(1,nx);
metxu = cs*ones(1,nx);

metyt = ones(ny,nx);
metyu = ones(ny,nx);

clear cs cst dxt1D dxu1D dyt1D dyu1D;

%-------------------------------------------------------------------------------
% Set-up vertical parameters.
%-------------------------------------------------------------------------------

%-----------------------------
%--- Compute sigma factor. ---
%-----------------------------

dsig = dsig./sum(dsig);

nsig = nz - kc;
sig  = dsig*(0.5.*diag(ones(nsig,1)) + triu(ones(nsig,nsig),1));

if ( (zc1==zc2) | (zslope==0) ),
   sigres = max(sig./dsig);
  else
   sigres = max( ((1-sig).*zslope + sig)./dsig );
end;

%------------------------------------------
%--- Assemble vertical parameter block. ---
%------------------------------------------

vertparm = [zc1 zc2 zref zslope sigres];
