function [domdat,bath,landt,mcseg,mpseg,misle] = read_mask (gname);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                           %
% function [domdat,bath,landt,mcseg,mpseg,misle] = read_mask (gname);       %
%                                                                           %
% This routine reads in domain parameters, bathymetry, and Land/Sea mask    %
% data (if any) from GRIDS NetCDF file.                                     %
%                                                                           %
% On Input:                                                                 %
%                                                                           %
%    gname     GRIDS NetCDF file name (character string).                   %
%                                                                           %
% On Output:                                                                %
%                                                                           %
%    DOMDAT    Vector containing domain definition parameters.              %
%                 DOMDAT( 1) = COORD - grid type                            %
%                 DOMDAT( 2) = LONC  - longitude trans. center              %
%                 DOMDAT( 3) = LATC  - latitude trans. center               %
%                 DOMDAT( 4) = THETA - grid rotation angle                  %
%                 DOMDAT( 5) = IM    - number of x-grid points              %
%                 DOMDAT( 6) = JM    - number of y-grid points              %
%                 DOMDAT( 7) = DX    - x-grid spacing                       %
%                 DOMDAT( 8) = DY    - y-grid spacing                       %
%                 DOMDAT( 9) = DELX  - x-offset grid-trans. centers         %
%                 DOMDAT(10) = DELY  - y-offset grid-trans. centers         %
%    BATH      raw bathymetry (real matrix; meters, negative below surface).%
%    LANDT     Land/Sea mask at tracer points (integer matrix):             %
%              LANDT=0 land, LANDT=1 Sea.                                   %
%    MCSEG     maximun number of coastal segments (integer).                %
%    MPSEG     maximun number of points per coastal segments (integer).     %
%    MISLE     maximun number of islands (integer).                         %
%                                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%-------------------------------------------------------------------------------
% Open GRIDS NetCDF file.
%-------------------------------------------------------------------------------

nc_nowrite = mexcdf ('parameter','nc_nowrite');
ncfid = mexcdf ('open',gname,nc_nowrite);
if (ncfid == -1),
  error(['READ_MASK: ncopen - unable to open file: ' gname])
  return
end

%-------------------------------------------------------------------------------
% Supress all error messages from NetCDF.
%-------------------------------------------------------------------------------

ncopts = mexcdf ('setopts',0);

%----------------------------------------------------------------------------
% Inquire values about dimensions.
%----------------------------------------------------------------------------

dimname = 'ncseg';
did = mexcdf ('dimid',ncfid,dimname);
if (did >= 0),
  [dimnam,dimen,status] = mexcdf ('diminq',ncfid,did);
  if (status == -1),
    error(['READ_MASK: ncdiminq - error while reading dimension: ',dimname])
  end
  mcseg=dimen;
else
  mcseg=0;
end

dimname = 'lcseg';
did = mexcdf ('dimid',ncfid,dimname);
if (did >= 0),
  [dimnam,dimen,status] = mexcdf ('diminq',ncfid,did);
  if (status == -1),
    error(['READ_MASK: ncdiminq - error while reading dimension: ',dimname])
  end
  mpseg=dimen;
else
  mpseg=0;
end

dimname = 'island';
did = mexcdf ('dimid',ncfid,dimname);
if (did >= 0),
  [dimnam,dimen,status] = mexcdf ('diminq',ncfid,did);
  if (status == -1),
    error(['READ_MASK: ncdiminq - error while reading dimension: ',dimname])
  end
  misle=dimen;
else
  misle=0;
end

%----------------------------------------------------------------------------
% Read in domain parameters.
%----------------------------------------------------------------------------

%------------------------------------------------------------
%--- Read in grid type. If not present, assume Cartesian. ---
%------------------------------------------------------------

varname = 'coord';
vid = mexcdf ('varid',ncfid,varname);
if (vid >= 0),
  indx = 0;
  [coord,status] = mexcdf ('varget1',ncfid,vid,indx);
  if (status == -1),
    error(['READ_MASK: ncvarget1 - error while reading: ',varname])
  end
else
  coord=0;
end

%-----------------------------
%--- Read in grid spacing. ---
%-----------------------------

varname = 'meandx';
vid = mexcdf ('varid',ncfid,varname);
if (vid >= 0),
  indx = 0;
  [dx,status] = mexcdf ('varget1',ncfid,vid,indx);
  if (status == -1),
    error(['READ_MASK: ncvarget1 - error while reading: ',varname])
  end
else
  error(['READ_MASK: ncvarid - cannot find variable: ',varname])
end

varname = 'meandy';
vid = mexcdf ('varid',ncfid,varname);
if (vid >= 0),
  indx = 0;
  [dy,status] = mexcdf ('varget1',ncfid,vid,indx);
  if (status == -1),
    error(['READ_MASK: ncvarget1 - error while reading: ',varname])
  end
else
  error(['READ_MASK: ncvarid - cannot find variable: ',varname])
end

%--------------------------------------------
%--- Read centroid (lon,lat) coordinates. ---
%--------------------------------------------

varname = 'rlngd';
vid = mexcdf ('varid',ncfid,varname);
if (vid >= 0),
  indx = 0;
  [lonc,status] = mexcdf ('varget1',ncfid,vid,indx);
  if (status == -1),
    error(['READ_MASK: ncvarget1 - error while reading: ',varname])
  end
else
  error(['READ_MASK: ncvarid - cannot find variable: ',varname])
end


varname = 'rlatd';
vid = mexcdf ('varid',ncfid,varname);
if (vid >= 0),
  indx = 0;
  [latc,status] = mexcdf ('varget1',ncfid,vid,indx);
  if (status == -1),
    error(['READ_MASK: ncvarget1 - error while reading: ',varname])
  end
else
  error(['READ_MASK: ncvarid - cannot find variable: ',varname])
end

%----------------------------
%--- Read in grid offset. ---
%----------------------------

oldgrid = 0;

varname = 'delx';
vid = mexcdf ('varid',ncfid,varname);
if (vid >= 0),
  indx = 0;
  [delx,status] = mexcdf ('varget1',ncfid,vid,indx);
  if (status == -1),
    error(['READ_MASK: ncvarget1 - error while reading: ',varname])
  end
else
  oldgrid = 1;
end

varname = 'dely';
vid = mexcdf ('varid',ncfid,varname);
if (vid >= 0),
  indx = 0;
  [dely,status] = mexcdf ('varget1',ncfid,vid,indx);
  if (status == -1),
    error(['READ_MASK: ncvarget1 - error while reading: ',varname])
  end
else
  oldgrid = 1;
end

if oldgrid,
   if (coord~=1),
      delx = 0;
      dely = 0;
     else
      delx = lonc;
      dely = latc;
      lonc = 0;
      latc = 0;
   end;
end;

clear oldgrid;

%--------------------------------------
%--- Read in domain rotation angle. ---
%--------------------------------------

varname = 'thetad';
vid = mexcdf ('varid',ncfid,varname);
if (vid >= 0),
  indx = 0;
  [theta,status] = mexcdf ('varget1',ncfid,vid,indx);
  if (status == -1),
    error(['READ_MASK: ncvarget1 - error while reading: ',varname])
  end
else
  error(['READ_MASK: ncvarid - cannot find variable: ',varname])
end

%--------------------------------------
%--- Read in number of grid points. ---
%--------------------------------------

varname = 'imt';
vid = mexcdf ('varid',ncfid,varname);
if (vid >= 0),
  indx = 0;
  [im,status] = mexcdf ('varget1',ncfid,vid,indx);
  if (status == -1),
    error(['READ_MASK: ncvarget1 - error while reading: ',varname])
  end
else
  error(['READ_MASK: ncvarid - cannot find variable: ',varname])
end

varname = 'jmt';
vid = mexcdf ('varid',ncfid,varname);
if (vid >= 0),
  indx = 0;
  [jm,status] = mexcdf ('varget1',ncfid,vid,indx);
  if (status == -1),
    error(['READ_MASK: ncvarget1 - error while reading: ',varname])
  end
else
  error(['READ_MASK: ncvarid - cannot find variable: ',varname])
end

%-------------------------
%--- Save domain data. ---
%-------------------------

domdat = [coord lonc latc theta im jm dx dy delx dely];

%----------------------------------------------------------------------------
% Read in raw bathymetry (meters).
%----------------------------------------------------------------------------

varname = 'raw_bath';
vid = mexcdf ('varid',ncfid,varname);
if (vid >= 0),
  indx = [0 0];
  cnt  = [jm im];
  [bath,status] = mexcdf ('varget',ncfid,vid,indx,cnt);
  if (status == -1),
    error(['READ_MASK: ncvarget - error while reading: ',varname])
  end
  bath=bath';
else
  error(['READ_MASK: ncvarid - cannot find variable: ',varname])
end

%----------------------------------------------------------------------------
% Read in Land/Sea mask, if any.
%----------------------------------------------------------------------------

varname = 'landt';
vid = mexcdf ('varid',ncfid,varname);
if (vid >= 0),
  indx = [0 0];
  cnt  = [jm im];
  [landt,status] = mexcdf ('varget',ncfid,vid,indx,cnt);
  if (status == -1),
    error(['READ_MASK: ncvarget - error while reading: ',varname])
  end
  landt=landt';
else
  landt=zeros(jm,im);
end

%----------------------------------------------------------------------------
% Close Grids NetCDF file.
%----------------------------------------------------------------------------

status = mexcdf ('close',ncfid);
if (status == -1),
  error(['READ_MASK: ncclose - unable to close NetCDF file.'])
end

return
