function [bpdist,btdist,bvdist,bzdist] = ...
                                    bndy_dist (cstmask,lencst,icst,jcst,method);
%
% function [bpdist,btdist,bvdist,bzdist] = ...
%                                   bndy_dist (cstmask,lencst,icst,jcst,method);
%
% This function find the distance of every grid point to the
% nearest boundary.  In this context, boundary refers to either
% the domain outer boundary or a coastal boundary.
%
% METHOD  Flag indicating which boundaries are used.
%            'Exterior'  -  Only exterior domain boundaries.
%            'Coastal'   -  Only coastal domain boundaries.
%            'All'       -  All domain boundaries.

%-------------------------------------------------------------------------------
% Create indices matrix.
%-------------------------------------------------------------------------------

[ny nx] = size(cstmask);

ivec = 1:nx;
jvec = 1:ny;

[Imat,Jmat] = meshgrid (ivec,jvec);

clear ivec jvec;

%-------------------------------------------------------------------------------
% Initialize boundary matrix.
%-------------------------------------------------------------------------------

bpdist = sqrt(nx.*nx + ny.*ny + 10) .* ones(ny,nx);
btdist = bpdist;
bvdist = bpdist;
bzdist = bpdist;

%-------------------------------------------------------------------------------
% Find distances to nearest open boundary.
%-------------------------------------------------------------------------------

if ~strcmp (upper(method),'COASTAL');

   %--------------------------------------
   %--- Construct inverted LandT mask. ---
   %--------------------------------------

   landti = ones(ny,nx);
   ind = find(cstmask==0);
   if ~isempty(ind)
      landti(ind) = 0;
   end;

   n = [2:ny ny];
   s = [1 1:(ny-1)];
   e = [2:nx nx];
   w = [1 1:(nx-1)];

   landti = landti(n,w).*landti(n,:).*landti(n,e).* ...
            landti(:,w).*   landti  .*landti(:,e).* ...
            landti(s,w).*landti(s,:).*landti(s,e);

   %--------------------------------------
   %--- Construct inverted LandV mask. ---
   %--------------------------------------

   landt = 1 - landti;

   s = 1:ny;
   w = 1:nx;

   landvi = 1 - landt(n,w).*landt(n,e).*landt(s,w).*landt(s,e);

   clear e landt n s w

   %----------------------------------------
   %--- Construct open boundary indices. ---
   %----------------------------------------

   ipbnd = [1:nx       nx.*ones(1,ny-1)   (nx-1):-1:1    ones(1,ny-2)];
   jpbnd = [ones(1,nx)   2:ny           ny.*ones(1,nx-1)  (ny-1):-1:2];

   itbnd = [ipbnd      (nx-1):-1:2     ];
   jtbnd = [jpbnd  (ny-1).*ones(1,nx-2)];

   ivbnd = [itbnd  (nx-1).*ones(1,ny-2)];
   jvbnd = [jtbnd        2:(ny-1)      ];

   izbnd = [ivbnd 2.*ones(1,ny-2)    2:(nx-1)    ];
   jzbnd = [jvbnd    2:(ny-1)     2.*ones(1,nx-2)];

   bbnd = jpbnd + (ipbnd-1).*ny;
   ind = find (landti(bbnd)==1);
   if ~isempty(ind)
      ipbnd(ind) = [];
      jpbnd(ind) = [];
   end;

   bbnd = jtbnd + (itbnd-1).*ny;
   ind = find (landti(bbnd)==1);
   if ~isempty(ind)
      itbnd(ind) = [];
      jtbnd(ind) = [];
   end;

   bbnd = jzbnd + (izbnd-1).*ny;
   ind = find (landti(bbnd)==1);
   if ~isempty(ind)
      izbnd(ind) = [];
      jzbnd(ind) = [];
   end;

   bbnd = jvbnd + (ivbnd-1).*ny;
   ind = find (landvi(bbnd)==1);
   if ~isempty(ind)
      ivbnd(ind) = [];
      jvbnd(ind) = [];
   end;

   clear bbnd ind landti landvi;

   %--------------------------------------
   %--- Distance from open boundaries. ---
   %--------------------------------------

   for j = 1:ny
   for i = 1:nx
      bpdist(j,i) = sqrt(min( (ipbnd-Imat(j,i)).^2 + (jpbnd-Jmat(j,i)).^2) );
      btdist(j,i) = sqrt(min( (itbnd-Imat(j,i)).^2 + (jtbnd-Jmat(j,i)).^2) );
      bvdist(j,i) = sqrt(min( (ivbnd-Imat(j,i)).^2 + (jvbnd-Jmat(j,i)).^2) );
      bzdist(j,i) = sqrt(min( (izbnd-Imat(j,i)).^2 + (jzbnd-Jmat(j,i)).^2) );
   end;
   end;

end;

%-------------------------------------------------------------------------------
% Find distances to nearest coastline.
%-------------------------------------------------------------------------------

if ~strcmp (upper(method),'EXTERIOR')

   %-------------------------
   %--- Clear under land. ---
   %-------------------------

   ind = find ( (cstmask>=1) & (cstmask<=length(lencst)) );

   if ~isempty(ind)
      bpdist(ind) = 0;
      btdist(ind) = 0;
      bvdist(ind) = 0;
      bzdist(ind) = 0;
   end;

   %----------------------------------------------
   %--- Find distance from wet points to land. ---
   %----------------------------------------------

   repind = find ((bpdist>0) | (btdist>0) | (bvdist>0) | (bzdist>0));

   for k = 1:length(repind)

      csind  = 1:lencst(1);
      mndist = min( (icst(csind,1)-Imat(repind(k))).^2 + ...
                     (jcst(csind,1)-Jmat(repind(k))).^2);

      for n = 2:length(lencst)
         csind = 1:lencst(n);
         dist  = min( (icst(csind,n)-Imat(repind(k))).^2 + ...
                      (jcst(csind,n)-Jmat(repind(k))).^2 );
         if (dist<mndist),
            mndist = dist;
         end;
      end;

      mndist = sqrt(mndist);

      if (mndist<bpdist(repind(k)))
         bpdist(repind(k)) = mndist;
      end;
      if (mndist<btdist(repind(k)))
         btdist(repind(k)) = mndist;
      end;
      if (mndist<bvdist(repind(k)))
         bvdist(repind(k)) = mndist;
      end;
      if (mndist<bzdist(repind(k)))
         bzdist(repind(k)) = mndist;
      end;

   end;

end;
