function [tbath,vbath,dxt,dyt,metxt,metyt,dxu,dyu,metxu,metyu, ...
          cenlat,cstmask,landv,lncst,icst,jcst,nisle,status] = ...
                                                   get_grids (gfile,resettopo);
%
% function [tbath,vbath,dxt,dyt,metxt,metyt,dxu,dyu,metxu,metyu, ...
%           cenlat,cstmask,landv,lncst,icst,jcst,nisle,status] = ...
%                                                  get_grids (gfile,resettopo);
%
% This function reads in the necessary topography and coastline data
% for resetting the OA/FM streamfunction variable.
%
% ------
% Input:
% ------
%
%    GFILE.......Name of GRIDS netCDF file.
%    RESETTOPO...T-grid topo re-averaging flag.
%                   [0] no re-averaging
%                   [1] Reset T-grid topo as average of V-grid topo.
%
% -------
% Output:
% -------
%
%    TBATH.....PE topography at tracer points.
%    VBATH.....PE topography at velocity points.
%    DXT.......Zonal grid spacing at tracer points.
%    DYT.......Meridional grid spacing at tracer points.
%    METXT.....Zonal metric factors at tracer points.
%    METYT.....Meridional metric factors at tracer points.
%    DXU.......Zonal grid spacing at velocity points.
%    DYU.......Meridional grid spacing at velocity points.
%    METXU.....Zonal metric factors at velocity points.
%    METYU.....Meridional metric factors at velocity points.
%    CENLAT....Central latitude.
%    CSTMASK...Land + coastline mask at tracer points.
%    LANDV.....Land mask at velocity points.
%    LNCST.....Length of coastal segments.
%    ICST......Coastal segment i-indices.
%    JCST......Coastal segment j-indices.
%    NISLE.....Number of coastal segments which are islands.
%    STATUS....Exit status.  [0] no error  [<0] error

%-------------------------------------------------------------------------------
%  Initialize output and some working storage.
%-------------------------------------------------------------------------------

tbath   = [];
vbath   = [];
dxt     = [];
dyt     = [];
metxt   = [];
metyt   = [];
dxu     = [];
dyu     = [];
metxu   = [];
metyu   = [];
cenlat  = [];
cstmask = [];
landv   = [];
lncst   = [];
icst    = [];
jcst    = [];
nisle   = [];

landt   = [];

m2cm = 100.0;

%-------------------------------------------------------------------------------
%  Set netCDF parameters.
%-------------------------------------------------------------------------------

status = mexcdf ('setopts',0);

%-------------------------------------------------------------------------------
%  Get topography variables.
%-------------------------------------------------------------------------------

[tlon,tlat,tbath,vlon,vlat,dxt,dyt,metxt,metyt, ...
          dxu,dyu,metxu,metyu,vertparm,status] = gridsread (gfile);

if (status<0),
   disp (' ');
   disp (['***Error:  GET_GRIDS - unable to get basic grid variables in ', ...
          'file:']);
   disp (['           ',setstr(34),gfile,setstr(34)]);
   disp (' ');
   return;
end;

cenlat = mean (tlat(:));

metxt = metxt.*m2cm;
metyt = metyt.*m2cm;
metxu = metxu.*m2cm;
metyu = metyu.*m2cm;

clear tlon tlat vlon vlat vertparm;

%-------------------------------------------------------------------------------
%  Re-open netCDF file.
%-------------------------------------------------------------------------------

ncid = mexcdf ('ncopen',gfile,mexcdf('parameter','nc_nowrite'));

if (ncid<0),
   disp (' ');
   disp ('***Error:  GET_GRIDS - unable to open input file:');
   disp (['           ',setstr(34),gfile,setstr(34)]);
   disp (' ');
   status = ncid;
   return;
end;

%-------------------------------------------------------------------------------
%  Get dimensions.
%-------------------------------------------------------------------------------

name = 'tlon';

[nx, status] = get_dim (ncid,name);

if (status<0),
   disp (' ');
   disp (['***Error:  GET_GRIDS - unable to get dimension ',setstr(34), ...
          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),gfile,setstr(34)]);
   disp (' ');
   return;
end;

name = 'tlat';

[ny, status] = get_dim (ncid,name);

if (status<0),
   disp (' ');
   disp (['***Error:  GET_GRIDS - unable to get dimension ',setstr(34), ...
          name,setstr(34),' in file:']);
   disp (['           ',setstr(34),gfile,setstr(34)]);
   disp (' ');
   return;
end;

%-------------------------------------------------------------------------------
%  Get mask/coastline variables, if present.
%-------------------------------------------------------------------------------

%--------------------------------------------
%--- Test to see if coastal mask present. ---
%--------------------------------------------

name = 'ncseg';

varid = mexcdf ('ncvarid',ncid,name);

if (varid>=0),

   %---------------------------------
   %--- Get number of coastlines. ---
   %---------------------------------

   [ncseg,status] = get_var1 (ncid,name,0);

   if (status<0),
      disp (' ');
      disp (['***Error:  GET_GRIDS - unable to get variable ',setstr(34), ...
             name,setstr(34),' in file:']);
      disp (['           ',setstr(34),gfile,setstr(34)]);
      disp (' ');
      return;
   end;

   %---------------------------
   %--- Get coastline data. ---
   %---------------------------

   name = 'lencoast';

   [lncst,status] = get_var (ncid,name,0,ncseg);

   if (status<0),
      disp (' ');
      disp (['***Error:  GET_GRIDS - unable to get variable ',setstr(34), ...
             name,setstr(34),' in file:']);
      disp (['           ',setstr(34),gfile,setstr(34)]);
      disp (' ');
      return;
   end;

   mxcst = max(lncst);

   name = 'icoast';

   [icst,status] = get_var (ncid,name,[0 0],[ncseg mxcst]);

   if (status<0),
      disp (' ');
      disp (['***Error:  GET_GRIDS - unable to get variable ',setstr(34), ...
             name,setstr(34),' in file:']);
      disp (['           ',setstr(34),gfile,setstr(34)]);
      disp (' ');
      return;
   end;

   name = 'jcoast';

   [jcst,status] = get_var (ncid,name,[0 0],[ncseg mxcst]);

   if (status<0),
      disp (' ');
      disp (['***Error:  GET_GRIDS - unable to get variable ',setstr(34), ...
             name,setstr(34),' in file:']);
      disp (['           ',setstr(34),gfile,setstr(34)]);
      disp (' ');
      return;
   end;

   %---------------------------
   %--- Get land mask data. ---
   %---------------------------

   name = 'landt';

   [landt,status] = get_var (ncid,name,[0 0],[ny nx]);

   if (status<0),
      disp (' ');
      disp (['***Error:  GET_GRIDS - unable to get variable ',setstr(34), ...
             name,setstr(34),' in file:']);
      disp (['           ',setstr(34),gfile,setstr(34)]);
      disp (' ');
      return;
   end;

   landt = landt';

   name = 'landv';

   [landv,status] = get_var (ncid,name,[0 0],[ny nx]);

   if (status<0),
      disp (' ');
      disp (['***Error:  GET_GRIDS - unable to get variable ',setstr(34), ...
             name,setstr(34),' in file:']);
      disp (['           ',setstr(34),gfile,setstr(34)]);
      disp (' ');
      return;
   end;

   landv = landv';

   %---------------------------------
   %--- Get number of coastlines. ---
   %---------------------------------

   name = 'nisle';

   varid = mexcdf ('ncvarid',ncid,name);

   if (varid>=0),

      [nisle,status] = get_var1 (ncid,name,0);

      if (status<0),
         disp (' ');
         disp (['***Error:  GET_GRIDS - unable to get variable ',setstr(34), ...
                name,setstr(34),' in file:']);
         disp (['           ',setstr(34),gfile,setstr(34)]);
         disp (' ');
         return;
      end;

    else

      nisle = 0;

   end;

end;

%-------------------------------------------------------------------------------
%  Close netCDF file.
%-------------------------------------------------------------------------------

status = mexcdf ('ncclose',ncid);

if (status<0),
   disp (' ');
   disp ('***Error:  GET_GRIDS - unable to close input file:');
   disp (['           ',setstr(34),gfile,setstr(34)]);
   disp (' ');
   return;
end;

%-------------------------------------------------------------------------------
%  Compute V-grid topography.  Reset T-grid topography.
%-------------------------------------------------------------------------------

e = [2:nx nx];
w = 1:nx;

n = [2:ny ny];
s = 1:ny;

vbath = 0.25 .* (tbath(n,e) + tbath(n,w) + tbath(s,e) + tbath(s,w));

if (resettopo)
   e = 2:nx;
   w = e-1;

   n = 2:ny;
   s = n-1;

   tbath(n,e) = 0.25 .* (vbath(n,e) + vbath(n,w) + vbath(s,e) + vbath(s,w));
   tbath(n,1) = tbath(n,2);
   tbath(1,e) = tbath(2,e);
   tbath(1,1) = 0.5 .* (tbath(2,1)+tbath(1,2));
end;

%-------------------------------------------------------------------------------
%  Compute coastal mask.
%-------------------------------------------------------------------------------

if ~isempty(landt),

   %----------------------------------------------------------
   %--- Compute mask with 1s under land/coast, 0s for sea. ---
   %----------------------------------------------------------

   ip1 = [2:nx nx];
   i   = 1:nx;
   im1 = [1 1:(nx-1)];

   jp1 = [2:ny ny];
   j   = 1:ny;
   jm1 = [1 1:(ny-1)];

   cstmask = 1 - landt(jp1,im1).*landt(jp1,i).*landt(jp1,ip1).* ...
                 landt(j  ,im1).*landt(j  ,i).*landt(j  ,ip1).* ...
                 landt(jm1,im1).*landt(jm1,i).*landt(jm1,ip1);

   clear i im1 ip1 j jm1 jp1;

   %----------------------------------------------------------------
   %--- Re-arrange islands coasts in descending order of length. ---
   %----------------------------------------------------------------

   if (nisle>0)

      ist = ncseg - nisle + 1;
      [dum I] = sort(lncst(ist:ncseg));
      if (nisle<ncseg)
         xI = 1:(ist-1);
        else
         xI = [];
      end;
      I2 = [xI (flipud(I)+(ncseg-nisle))'];
      icst  = icst(:,I2);
      jcst  = jcst(:,I2);
      lncst = lncst(I2);

   end;

   %----------------------------------------------------------------
   %--- Determine coastal segment associated w/ each land point. ---
   %----------------------------------------------------------------

   for j = 1:ny;
   for i = 1:nx;

      if (cstmask(j,i)>0),

         csptr  = 1;
         csind  = 1:lncst(1);
         mndist = min((icst(csind,1)-i).^2 + (jcst(csind,1)-j).^2);

         for n = 2:ncseg
            csind = 1:lncst(n);
            dist  = (icst(csind,n)-i).^2 + (jcst(csind,n)-j).^2;
            if (min(dist)<mndist),
               csptr  = n;
               mndist = min(dist);
            end;
         end;

         cstmask(j,i) = csptr;

      end;
   end;
   end;

end;
