function [urms,vrms] = vel_rms (errtol,petind,pitind,lonlim,latlim,zlev, ...
                                fldnam,ncpe,ncpi,outid,lon,lat);
%
% function [urms,vrms] = vel_rms (errtol,petind,pitind,lonlim,latlim,zlev, ...
%                                 fldnam,ncpe,ncpi,outid,lon,lat);
%
% This function constructs the RMS error fields for a velocity variable.
%
% ------
% Input:
% ------
%
%    ERRTOL   Maximum "OA" error for comparison.
%    PETIND   PE output time level to test.
%    PITIND   Correspondint PE_initial output time level for comparison.
%    LONLIM   Longitude limits for comparison.
%    LATLIM   Latitude limits for comparison.
%    ZLEV     Depth levels for comparison.
%    FLDNAM   Names of fields to compare.
%    NCPE     NetCDF identifier of input PE file to be tested.
%    NCPI     NetCDF identifier of input PE_initial assimilation "truth" file
%    OUTID    Identifier for output ASCII results file.
%    LON,LAT  Horizontal positions.

%-------------------------------------------------------------------------------
% Get basic parameters.
%-------------------------------------------------------------------------------

nz = length(ncpi('level'));

mask   = squeeze(ncpe{'landv'}(:));
actval = 2;
actval = 2;
if isempty(mask)
   nx = length(ncpe('vlon'));
   ny = length(ncpe('vlat'));
   mask = actval.*ones(ny,nx);
end;

[ny nx] = size(mask);
fdiff = NaN.*ones(ny,nx,nz);
zdiff = NaN.*ones(ny,nx,length(zlev));
zwk   = abs(squeeze(ncpe{'tgrid3'}(:,:,:,3)));

vfill = ncpe{fldnam}.FillValue_(:);
vifill = ncpi{fldnam}.FillValue_(:);
if ( isempty(vifill) & ~isempty(vfill) )
   vifill = vfill;
 elseif ( isempty(vfill) & ~isempty(vifill) )
   vfill = vifill;
 elseif ( isempty(vfill) & isempty(vifill) )
   vifill = 1.0e35;
   vfill  = vifill;
end;

urms = NaN.*ones(1,length(zlev)+1);
vrms = urms;

%-------------------------------------------------------------------------------
% Loop over all times, fields & levels, computing restricted RMS errors.
%-------------------------------------------------------------------------------

%----------------------------------
%--- Zonal Velocity RMS errors. ---
%----------------------------------

fprintf (outid,'\n   Zonal\n\n');

velind = 1;
errsum = 0;
nsum   = 0;

for n = 1:nz
   pefld = squeeze(ncpe{fldnam}(petind,:,:,n,velind));
   pifld = squeeze(ncpi{fldnam}(pitind,:,:,n,velind));
   erfld = squeeze(ncpi{[fldnam,'err']}(pitind,:,:,n,velind));
   if isempty(erfld)
      erfld = zeros(size(pefld));
   end;

   ind = find ( (erfld<=errtol) & (lon>=lonlim(1)) & (lon<=lonlim(2)) ...
                 & (lat>=latlim(1)) & (lat<=latlim(2)) & (mask==actval) & ...
                 (pefld~=vfill) & (pifld~=vifill) );

   if ~isempty(ind)
      ind3d = ind + (n-1).*nx.*ny;
      fdiff(ind3d) = pefld(ind) - pifld(ind);
   end;
end;

for j = 1:ny
for i = 1:nx
   zdiff(j,i,:) = interp1 (squeeze(zwk(j,i,:)),squeeze(fdiff(j,i,:)),zlev);
end;
end;

for n = 1:length(zlev)
   ind = find (~isnan(zdiff(:,:,n)));
   cnt(n) = length(ind);
   if ~isempty(ind)
      ind3d = ind + (n-1).*nx.*ny;
      urms(n) = sqrt(mean( (zdiff(ind3d)).^2 ));

      fprintf (outid,'         %f    %f\n',[zlev(n) urms(n)]);

      errsum = errsum + cnt(n).*urms(n).*urms(n);
      nsum   = nsum + cnt(n);
   end;
end;

urms(length(zlev)+1) = sqrt(errsum./nsum);

fprintf (outid,'\n         Volume RMS Error = %f\n',urms(length(zlev)+1));

%---------------------------------------
%--- Meridional Velocity RMS errors. ---
%---------------------------------------

fprintf (outid,'\n   Meridional\n\n');

velind = 2;
errsum = 0;
nsum   = 0;

for n = 1:nz
   pefld = squeeze(ncpe{fldnam}(petind,:,:,n,velind));
   pifld = squeeze(ncpi{fldnam}(pitind,:,:,n,velind));
   erfld = squeeze(ncpi{[fldnam,'err']}(pitind,:,:,n,velind));
   if isempty(erfld)
      erfld = zeros(size(pefld));
   end;

   ind = find ( (erfld<=errtol) & (lon>=lonlim(1)) & (lon<=lonlim(2)) ...
                 & (lat>=latlim(1)) & (lat<=latlim(2)) & (mask==actval) & ...
                 (pefld~=vfill) & (pifld~=vifill) );

   if ~isempty(ind)
      ind3d = ind + (n-1).*nx.*ny;
      fdiff(ind3d) = pefld(ind) - pifld(ind);
   end;
end;

for j = 1:ny
for i = 1:nx
   zdiff(j,i,:) = interp1 (squeeze(zwk(j,i,:)),squeeze(fdiff(j,i,:)),zlev);
end;
end;

for n = 1:length(zlev)
   ind = find (~isnan(zdiff(:,:,n)));
   cnt(n) = length(ind);
   if ~isempty(ind)
      ind3d = ind + (n-1).*nx.*ny;
      vrms(n) = sqrt(mean( (zdiff(ind3d)).^2 ));

      fprintf (outid,'         %f    %f\n',[zlev(n) vrms(n)]);

      errsum = errsum + cnt(n).*vrms(n).*vrms(n);
      nsum   = nsum + cnt(n);
   end;
end;

vrms(length(zlev)+1) = sqrt(errsum./nsum);

fprintf (outid,'\n         Volume RMS Error = %f\n',vrms(length(zlev)+1));

fprintf (outid,'\n         Volume RMS Error = %f\n',sqrt(errsum./nsum));
