function status = comb_oa (temp,terr,tstd,zmld,jdate,oafile);
%
% function status = comb_oa (temp,terr,tstd,zmld,jdate,oafile);
%
% This function combines the Temperature field in an OA(G)/FM
% file with the SST & error fields.

%-------------------------------------------------------------------------------
% Open netCDF file.
%-------------------------------------------------------------------------------

status = mexcdf ('setopts',0);

ncid = netcdf (oafile,'write');

if isempty(ncid)
   disp (' ');
   disp ('***Error:  COMB_OA - unable to open OA(G)/FM file:');
   disp (['           ',setstr(34),oafile,setstr(34)]);
   disp (' ');
   status = -1;
   return;
end;

%-------------------------------------------------------------------------------
% Determine number of vertical levels & the depths.
%-------------------------------------------------------------------------------

name = 'level';
nz = size(ncid(name));

if isempty(nz)
   disp (' ');
   disp (['***Error:  COMB_OA - unable to read dimension ',setstr(34), ...
          name,setstr(34),' in OA(G)/FM file:']);
   disp (['           ',setstr(34),oafile,setstr(34)]);
   disp (' ');
   status = -1;
   return;
end;

name = 'zout';
z = squeeze(ncid{name}(:,3));

if isempty(z)
   disp (' ');
   disp (['***Error:  COMB_OA - unable to read variable ',setstr(34), ...
          name,setstr(34),' in OA(G)/FM file:']);
   disp (['           ',setstr(34),oafile,setstr(34)]);
   disp (' ');
   status = -1;
   return;
end;

%-------------------------------------------------------------------------------
% Find nearest time to modify.
%-------------------------------------------------------------------------------

name = 'time';
oatimes = ncid{name}(:);

if isempty(oatimes)
   disp (' ');
   disp (['***Error:  COMB_OA - unable to read variable ',setstr(34), ...
          name,setstr(34),' in OA(G)/FM file:']);
   disp (['           ',setstr(34),oafile,setstr(34)]);
   disp (' ');
   status = -1;
   return;
end;

dt = abs(oatimes - jdate);
dtmn = min(dt);
tind = min(find(dt==dtmn));

%-------------------------------------------------------------------------------
% Reset SST error based on temporal mismatch.
%-------------------------------------------------------------------------------

name  = 'tdcay';
tdcay = ncid{name}(:);

if isempty(tdcay)
   disp (' ');
   disp (['***Error:  COMB_OA - unable to read variable ',setstr(34), ...
          name,setstr(34),' in OA(G)/FM file:']);
   disp (['           ',setstr(34),oafile,setstr(34)]);
   disp (' ');
   status = -1;
   return;
end;

wt = dtmn/tdcay;
wt = exp(-0.5.*wt.*wt);

terr = terr.*wt + tstd.*(1-wt);

%-------------------------------------------------------------------------------
% Reset SST to a variance error.
%-------------------------------------------------------------------------------

terr = terr.*terr;
tvar = tstd.*tstd;

%-------------------------------------------------------------------------------
% Replace SST in upper levels.
%-------------------------------------------------------------------------------

nrep = max(find(abs(z)<=max(abs(zmld(:)))));

for n = 1:nrep;

   %--------------------------------
   %--- Read temperature fields. ---
   %--------------------------------

   twk = squeeze(ncid{'temp'}(tind,:,:,n));
   trrwk = squeeze(ncid{'temperr'}(tind,:,:,n));
   tmnwk = squeeze(ncid{'tempmean'}(tind,:,:,n));
   tscl = squeeze(ncid{'temprmsdev'}(tind,n));

   %---------------------------------------
   %--- Turn Error into Variance error. ---
   %---------------------------------------

   trrwk = trrwk.*trrwk;
   tscl  = tscl.*tscl;

   %-----------------------------------------------------------
   %--- Compute the correlation between in situ data & SST. ---
   %-----------------------------------------------------------

   ind = find ((terr<=(0.5.*tvar)) & (trrwk<=(0.5.*tscl)));

   dtwk = twk(ind)-tmnwk(ind);
   dtmp = temp(ind)-tmnwk(ind);
   corr = sum(dtwk.*dtmp) / sqrt(sum(dtwk.*dtwk)*sum(dtmp.*dtmp));

   %---------------------------------------
   %--- Downgrade SST error with depth. ---
   %---------------------------------------

   ewk = terr;

   wt = abs(z(n)./zmld);
   ind = find (abs(zmld)==0);
   if ~isempty(ind)
      wt(ind) = 1;
   end;
   ind = find (wt>1);
   if ~isempty(ind)
      wt(ind) = 1;
   end;
   wt = wt.^4;

   ewk = ewk.*(1-wt) + tvar.*wt;

   %-------------------------------------------
   %--- Combine SST with OA(G)/FM estimate. ---
   %-------------------------------------------

   errprod  = ewk.*trrwk;
   errmnfac = corr.*sqrt(errprod);
   denom    = ewk + trrwk - 2.*errmnfac;
   wt       = (ewk - errmnfac) ./ denom;

   twk = wt.*twk + (1-wt).*temp;
   ncid{'temp'}(tind,:,:,n) = twk;

   trrwk = errprod.*(1-corr.*corr) ./ (denom + errmnfac);

   %---------------------------
   %--- Reset error to STD. ---
   %---------------------------

   trrwk = sqrt(trrwk);

   ncid{'temperr'}(tind,:,:,n) = trrwk;

end;

%-------------------------------------------------------------------------------
% Close netCDF file.
%-------------------------------------------------------------------------------

status = close (ncid);

if ~isempty(status)
   disp (' ');
   disp ('***Error:  COMB_OA - unable to close OA(G)/FM file:');
   disp (['           ',setstr(34),oafile,setstr(34)]);
   disp (' ');
   status = -1;
   return;
end;

status = 0;
