function [tlon,tlat,sst,ssterr,sststd] = get_sst (errmin,xpnddt,xpnder, ...
                                                  sstfile,oafile);
%
% function [tlon,tlat,sst,ssterr,sststd] = get_sst (errmin,xpnddt,xpnder, ...
%                                                   sstfile,oafile);
%
% This function reads the data necessary to convert a satellite image to a
% MODS file.
%
% ------
% Input:
% ------
%
%    ERRMIN....Minimum value for non-dimensional error field.
%    XPNDDT....Length (neighbors) for expanding range of bad values
%    XPNDER....Length (neighbors) for transitioning errors
%    SSTFILE...Name of GIF file with SST image.
%    OAFILE....Name of OA(G)/FM netCDF file.
%
% -------
% Output:
% -------
%
%    TLON.....Vector of image longitudes.
%    TLAT.....Vector im image latitudes.
%    SST......Sea Surface Temperature matrix.
%    SSTERR...Sea Surface Temperature dimensional error matrix.
%    SSTSTD...Scaling for SSTERR.

%-------------------------------------------------------------------------------
% Initialize output.
%-------------------------------------------------------------------------------

tlon   = [];
tlat   = [];
sst    = [];
ssterr = [];
sststd = [];

%-------------------------------------------------------------------------------
% Get SST data.
%-------------------------------------------------------------------------------

disp ('   reading SST data.');

[sstim,sstmap] = imread (sstfile,'gif');

clear sstfile

%-------------------------------------------------------------------------------
% Get Navigation data.
%-------------------------------------------------------------------------------

[lon,lat,x,y,excldx,excldy,barx,bary,trange,relrng] = parse_sst (sstim,sstmap);

clear sstmap

%-------------------------------------------------------------------------------
% Get color mapping.
%-------------------------------------------------------------------------------

disp ('   determine mapping.');

[colind,tmpvec,tervec] = get_colbar (sstim,trange,relrng,barx,bary);

clear barx bary relrng trange;

%-------------------------------------------------------------------------------
% Block colorbar from SST image.
%-------------------------------------------------------------------------------

i1 = floor(min(excldx));
i2 = ceil(max(excldx));
j1 = floor(min(excldy));
j2 = ceil(max(excldy));

sstim(j1:j2,i1:i2) = NaN;

clear excldx excldy i1 i2 j1 j2

%-------------------------------------------------------------------------------
% Get SST.
%-------------------------------------------------------------------------------

disp ('   constructing SST temperature field.');

%----------------------------
%--- Construct SST field. ---
%----------------------------

sstfull = NaN.*ones(size(sstim));
ssterrf = sstfull;

for n = 1:length(colind)

   ind = find ( sstim == colind(n) );
   if ~isempty(ind)
      sstfull(ind) = tmpvec(n);
      ssterrf(ind) = tervec(n);
   end;

end;

clear colind sstim tervec tmpvec

%------------------------
%--- Normalize error. ---
%------------------------

ind = find(~isnan(sstfull));

nsst = length(ind);

if (nsst>1)
   sstxpt = median (sstfull(ind));
   dsst = sstfull(ind) - sstxpt;
   sststd = sqrt(sum(dsst.*dsst) ./ (nsst-1));
   clear dsst sstxpt;
   ssterrf = (ssterrf./sststd).^2;
   ind = find (~isnan(ssterrf) & (ssterrf<errmin));
   if ~isempty(ind)
      ssterrf(ind) = errmin;
   end;
 else
   disp('*** Insufficient useful data.');
   return;
end;

clear ind nsst errmin;

%----------------------------------------------
%--- Block neighbors to questionable values ---
%----------------------------------------------

if (xpnddt>0)
   [ny nx] = size(sstfull);

   msk = ones(ny,nx);
   ind = find(isnan(sstfull));
   msk(ind) = 0;
   n   = [2:ny ny];
   s   = [1 1:(ny-1)];
   e   = [2:nx nx];
   w   = [1 1:(nx-1)];

   for i = 1:xpnddt
      msk = msk(n,w).*msk(n,:).*msk(n,e).* ...
            msk(:,w).*msk     .*msk(:,e).* ...
            msk(s,w).*msk(s,:).*msk(s,e);
   end;

   ind = find (msk==0);
   if ~isempty(ind)
      sstfull(ind) = NaN;
      ssterrf(ind) = NaN;
   end;

   clear ind msk
end;

clear xpnddt

%-------------------------------------------------------------------------------
% Positions.
%-------------------------------------------------------------------------------

disp ('   constructing SST Positions.');

[ny nx] = size(sstfull);

%------------------
%--- Longitude. ---
%------------------

m = diff(lon)/diff(x);
b = lon(1) - m*x(1);
tlonfull = m.*(1:nx) + b;

%-----------------
%--- Latitude. ---
%-----------------

m = diff(lat)/diff(y);
b = lat(1) - m*y(1);
tlatfull = m.*(1:ny) + b;

clear b lat lon m nx ny x y;

%-------------------------------------------------------------------------------
% Make diagnostic plots.
%-------------------------------------------------------------------------------

disp ('   making diagnostic plots.');

figure;
pcolor (tlonfull,tlatfull,sstfull);shading flat;colormap(jet);colorbar;
set (gca,'XLim',extrem(tlonfull), ...
         'YLim',extrem(tlatfull), ...
         'DataAspectRatio',[1 cos(mean(tlatfull)*(pi/180)) 1], ...
         'FontName','Times','FontSize',14);

%-------------------------------------------------------------------------------
% Extract domain data.
%-------------------------------------------------------------------------------

disp ('   extracting domain data.');

%---------------------------
%--- Open OA(G)/FM file. ---
%---------------------------

ncid = netcdf (oafile);

if isempty(ncid)
   disp (' ');
   disp ('***Error:  GET_SST - unable to open OA(G)/FM netCDF file:');
   disp (['           ',setstr(34),oafile,setstr(34)]);
   disp (' ');
   return;
end;

%-----------------------
%--- Read positions. ---
%-----------------------

name = 'grid3';
tlon = squeeze(ncid{name}(:,:,1));

if isempty(tlon)
   disp (' ');
   disp (['***Error:  GET_SST - unable to read ',setstr(34),name, ...
          setstr(34),' in OA(G)/FM netCDF file:']);
   disp (['           ',setstr(34),oafile,setstr(34)]);
   disp (' ');
   return;
end;

tlat = squeeze(ncid{name}(:,:,2));

if isempty(tlat)
   disp (' ');
   disp (['***Error:  GET_SST - unable to read ',setstr(34),name, ...
          setstr(34),' in OA(G)/FM netCDF file:']);
   disp (['           ',setstr(34),oafile,setstr(34)]);
   disp (' ');
   return;
end;

clear name;

%----------------------------
%--- Close OA(G)/FM file. ---
%----------------------------

status = close (ncid);

if ~isempty(status)
   disp (' ');
   disp ('***Error:  GET_SST - unable to close OA(G)/FM netCDF file:');
   disp (['           ',setstr(34),oafile,setstr(34)]);
   disp (' ');
   return;
end;

clear oafile ncid status

%-------------------------------------------------------------------------------
% Grid SST data.
%-------------------------------------------------------------------------------

disp ('   interpolating SST fields.');

sst = interp2 (tlonfull,tlatfull,sstfull,tlon,tlat);
ssterr = interp2 (tlonfull,tlatfull,ssterrf,tlon,tlat);

clear ssterrf sstfull tlatfull tlonfull

disp ('   plotting interpolated SST fields.');

figure;
pcolor (tlon,tlat,sst);shading flat;colormap(jet);colorbar;
set (gca,'XLim',extrem(tlon), ...
         'YLim',extrem(tlat), ...
         'DataAspectRatio',[1 cos(mean(tlat(:))*(pi/180)) 1], ...
         'FontName','Times','FontSize',14);
title ('interpolated SST');

%-------------------------------------------------------------------------------
% Fill SST under problem areas
%-------------------------------------------------------------------------------

[ny nx] = size(sst);

msk = ones(ny,nx);
ind = find(isnan(sst));
msk(ind) = 0;

sst = smooth_under_iw (sst,msk);

clear ind;

figure;
pcolor (tlon,tlat,sst);shading flat;colormap(jet);colorbar;
set (gca,'XLim',extrem(tlon), ...
         'YLim',extrem(tlat), ...
         'DataAspectRatio',[1 cos(mean(tlat(:))*(pi/180)) 1], ...
         'FontName','Times','FontSize',14);
title ('filled SST');

%-------------------------------------------------------------------------------
% Fill SST error under problem areas
%-------------------------------------------------------------------------------

disp ('   constructing SST error field.');

%---------------------
%--- Replace NaNs. ---
%---------------------

emax = max([1 max(ssterr(:))]);
ind = find(isnan(ssterr));
if ~isempty(ind)
   ssterr(ind) = emax;
end;

clear emax;

%-----------------------------------
%--- Introduce transition areas. ---
%-----------------------------------

msk2 = msk;

n   = [2:ny ny];
s   = [1 1:(ny-1)];
e   = [2:nx nx];
w   = [1 1:(nx-1)];

for i = 1:xpnder
   msk2 = msk2(n,w).*msk2(n,:).*msk2(n,e).* ...
          msk2(:,w).*  msk2   .*msk2(:,e).* ...
          msk2(s,w).*msk2(s,:).*msk2(s,e);
end;

msk = 1 + msk2 - msk;

ssterr = smooth_under_iw (ssterr,msk);

clear e i msk msk2 n nx ny s w xpnder;

%-------------------------
%--- Rescale to deg C. ---
%-------------------------

ssterr = sqrt(ssterr) .* sststd;

%-------------------------
%--- Plot error field. ---
%-------------------------

figure;
pcolor (tlon,tlat,ssterr);shading flat;colormap(jet);colorbar;
set (gca,'XLim',extrem(tlon), ...
         'YLim',extrem(tlat), ...
         'DataAspectRatio',[1 cos(mean(tlat(:))*(pi/180)) 1], ...
         'FontName','Times','FontSize',14);
