function [colind,tmpvec,tervec] = get_colbar (sst,trange,relrng,xlim,ylim);
%
% function [colind,tmpvec,tervec] = get_colbar (sst,trange,relrng,xlim,ylim);
%
% This function extracts the colorbar from the SST image and constructs
% the mapping between SST index and temperature.
%
% ------
% Input:
% ------
%
%    SST      SST image map, an array of indices into a colormap.
%    TRANGE   Total range of temperature values covered by colorbar.
%    RELRNG   Range of reliable temperatures in SST image.
%    XLIM     Horizontal location of color portion of colorbar on SST image.
%    YLIM     Vertical location of color portion of colorbar on SST image.
%
% -------
% Output:
% -------
%
%    COLIND   SST indices corresponding to colorbar values.
%    TMPVEC   Temperatures corresponding to COLIND.
%    TERVEC   Temperature errors corresponding to COLIND.

%-------------------------------------------------------------------------------
% Extract colorbar from SST image.
%-------------------------------------------------------------------------------

i1 = ceil (xlim(1));
i2 = floor (xlim(2));

j = round(mean(ylim));

ivec  = i1:i2;
cbind = sst(j,ivec);

clear i1 i2 j;

%-------------------------------------------------------------------------------
% Construct mapping between colobar i-index and temperature.
%-------------------------------------------------------------------------------

itslope = diff(trange)/diff(xlim);
itintcp = trange(1) - itslope*xlim(1);

%-------------------------------------------------------------------------------
% Remove duplicate values from colorbar indices, retaining original order.
% Construct corresponding temperature & error vectors.
%-------------------------------------------------------------------------------

colind = [];
tmpvec = [];
tervec = [];

dt = itslope*(max(ivec)-min(ivec))/length(ivec);

while ~isempty(cbind)
   ctst = cbind(1);
   xind = find(cbind==ctst);
   iwk  = ivec(xind);
   cbind(xind) = [];
   ivec(xind) = [];

   t1 = itslope*min(iwk) + itintcp;
   tn = itslope*max(iwk) + itintcp;

   colind = [colind ctst];
   tmpvec = [tmpvec mean([t1 tn])];
%   tervec = [tervec abs(tn-t1)/2];
   tervec = [tervec length(iwk)*dt/2];
end;

clear cbind ctst itintcp itslope ivec iwk t1 tn xind;

%-------------------------------------------------------------------------------
% Remove unreliable portion of temperature mapping.
%-------------------------------------------------------------------------------

ind = find ( (tmpvec<relrng(1)) | (tmpvec>relrng(2)) );

if ~isempty(ind)
   colind(ind) = [];
   tmpvec(ind) = [];
   tervec(ind) = [];
end;

clear ind;
