function [heatflux,windflux] = read_flx2 (fluxname,ssttime);
%
% function [heatflux,windflux] = read_flx2 (fluxname,ssttime);
%
% This function read the surface heat flux at the given positions
% and times.
%
% ------
% Input:
% ------
%
%    FLUXNAME...Forcing netCDF file name.
%    SSTTIME....Times at which heat fluxes are desired.
%
% -------
% Output:
% -------
%
%    HEATFLUX...Surface heat flux for Neumann BC     (deg C cm^-1)
%    WINDFLUX...Wind stress for Eckman calculation.  (cm)

%-------------------------------------------------------------------------------
% Initialize output.
%-------------------------------------------------------------------------------

heatflux = [];
windflux = [];

%-------------------------------------------------------------------------------
% Set Conversion factors.
%-------------------------------------------------------------------------------

s_std = 35.0;
t_std = 20.0;
p_std = 0.0;

rho0 = sw_dens0 (s_std,t_std);      % Density sea-water:               kg/(m^3)
Cp   = sw_cp (s_std,t_std,p_std);   % Heat capcity sea-water:          J/(kg C)
k_ht = 1e-7;                        % Kinematic thermal conductivity:  m^2/s
m2cm = 100.0;

crho0 = rho0/1000;                  % Density sea-water  gm/(cm^3)
omega = 2*pi*(366.25/365.25)/86400; % Earth angular rotation.

fnoc2neumann = 1/(rho0*Cp*k_ht*m2cm);

%-------------------------------------------------------------------------------
% Open forcing file.
%-------------------------------------------------------------------------------

ncid = netcdf (fluxname);

if isempty(ncid),
   disp (' ');
   disp ('***Error:  READ_FLX - unable to open forcing file');
   disp(['           ',setstr(34),fluxname,setstr(34)]);
   disp (' ');
   return;
end;

%-------------------------------------------------------------------------------
% Get surface forcing times.
%-------------------------------------------------------------------------------

%----------------------------
%--- Get heat flux times. ---
%----------------------------

name = 'shf_time';

nftim = length (ncid(name));

if ~isempty(nftim),

   flxtim = ncid{name}(:);
   if isempty(flxtim),
      disp (' ');
      disp (['***Error:  READ_FLX - unable to read variable ',setstr(34), ...
              name,setstr(34),' in forcing file']);
      disp(['           ',setstr(34),fluxname,setstr(34)]);
      disp (' ');
      return;
   end;

else

   flxtim = [];

end;

%------------------------------
%--- Get wind stress times. ---
%------------------------------

name = 'smf_time';

nftim = length (ncid(name));

if ~isempty(nftim),

   vflxtim = ncid{name}(:);
   if isempty(vflxtim),
      disp (' ');
      disp (['***Error:  READ_FLX - unable to read variable ',setstr(34), ...
              name,setstr(34),' in forcing file']);
      disp(['           ',setstr(34),fluxname,setstr(34)]);
      disp (' ');
      return;
   end;

else

   vflxtim = [];

end;

%-------------------------------------------------------------------------------
% Read heat fluxes.
%-------------------------------------------------------------------------------

if ~isempty(flxtim)

   st = max(find(flxtim<=ssttime));

   name = 'shflux';

   flx0 = squeeze(ncid{name}(st,:,:));

   if isempty(flx0),
      disp (' ');
      disp (['***Error:  READ_FLX - unable to read variable ',setstr(34), ...
             name,setstr(34),' in forcing file']);
      disp (['           ',setstr(34),fluxname,setstr(34)]);
      disp (' ');
      return;
   end;

   flx0 = fnoc2neumann.*flx0;

   flx1 = squeeze(ncid{name}((st+1),:,:));

   if isempty(flx1),
      disp (' ');
      disp (['***Error:  READ_FLX - unable to read variable ',setstr(34), ...
             name,setstr(34),' in forcing file']);
      disp (['           ',setstr(34),fluxname,setstr(34)]);
      disp (' ');
      return;
   end;

   flx1 = fnoc2neumann.*flx1;
end;

%-------------------------------------------------------------------------------
% Read wind stresses.
%-------------------------------------------------------------------------------

if ~isempty(vflxtim)

   name = 'vgrid2';

   vlat = squeeze(ncid{name}(:,:,2));

   if isempty(vlat),
      disp (' ');
      disp (['***Error:  READ_FLX - unable to read variable ',setstr(34), ...
             name,setstr(34),' in forcing file']);
      disp (['           ',setstr(34),fluxname,setstr(34)]);
      disp (' ');
      return;
   end;

   f0 = 2.*omega.*abs(sin(vlat.*pi./180));

   vst = max(find(vflxtim<=ssttime));

   name = 'smflux';

   uflx = squeeze(ncid{name}(vst,:,:,1));

   if isempty(uflx),
      disp (' ');
      disp (['***Error:  READ_FLX - unable to read variable ',setstr(34), ...
             name,setstr(34),' in forcing file']);
      disp (['           ',setstr(34),fluxname,setstr(34)]);
      disp (' ');
      return;
   end;

   vflx = squeeze(ncid{name}(vst,:,:,2));

   if isempty(vflx),
      disp (' ');
      disp (['***Error:  READ_FLX - unable to read variable ',setstr(34), ...
             name,setstr(34),' in forcing file']);
      disp (['           ',setstr(34),fluxname,setstr(34)]);
      disp (' ');
      return;
   end;

   wflx0 = sqrt(uflx.*uflx+vflx.*vflx);
   wflx0 = sqrt(wflx0./crho0)./f0;

   uflx = squeeze(ncid{name}((vst+1),:,:,1));

   if isempty(uflx),
      disp (' ');
      disp (['***Error:  READ_FLX - unable to read variable ',setstr(34), ...
             name,setstr(34),' in forcing file']);
      disp (['           ',setstr(34),fluxname,setstr(34)]);
      disp (' ');
      return;
   end;

   vflx = squeeze(ncid{name}((vst+1),:,:,2));

   if isempty(vflx),
      disp (' ');
      disp (['***Error:  READ_FLX - unable to read variable ',setstr(34), ...
             name,setstr(34),' in forcing file']);
      disp (['           ',setstr(34),fluxname,setstr(34)]);
      disp (' ');
      return;
   end;

   wflx1 = sqrt(uflx.*uflx+vflx.*vflx);
   wflx1 = sqrt(wflx1./crho0)./f0;

end;

%-------------------------------------------------------------------------------
% Interpolate fluxes.
%-------------------------------------------------------------------------------

if ~isempty(flxtim)
   wt0 = (flxtim(st+1)-ssttime)./(flxtim(st+1)-flxtim(st));
   heatflux = flx0.*wt0 + flx1.*(1-wt0);
end;

if ~isempty(vflxtim)
   wt0 = (vflxtim(vst+1)-ssttime)./(vflxtim(vst+1)-vflxtim(vst));
   windflux = wflx0.*wt0 + wflx1.*(1-wt0);
end;

%-------------------------------------------------------------------------------
% Close forcing file.
%-------------------------------------------------------------------------------

status = close (ncid);

if ~isempty(status)
   disp (' ');
   disp ('***Error:  READ_FLX - unable to close forcing file');
   disp(['           ',setstr(34),fluxname,setstr(34)]);
   disp (' ');
   return;
end;
