      subroutine set_depths
c
c=======================================================================
c                                                                    ===
c  This routine returns the depth at each grid point along a slab    ===
c  of constant J in the hybrid coordinate system.                    ===
c                                                                    ===
c  ------                                                            ===
c  Input:                                                            ===
c  ------                                                            ===
c                                                                    ===
c  Common Blocks:                                                    ===
c                                                                    ===
c  /HYBRID/                                                          ===
c                                                                    ===
c     HAVE     Total thickness of reference                          ===
c              depths in lower levels.              (real; cm)       ===
c     HC       Total thickness of reference                          ===
c               depths in upper levels.             (real; cm)       ===
c     HZ       Thicknesses of the flat grid                          ===
c              vertical boxes.                      (real; cm)       ===
c     KC       Coordinate interface level           (integer)        ===
#ifdef dblsigma
c     TINTER   Interface depth at tracer points.    (real array; cm) ===
c     VINTER   Interface depth at velocity points.  (real array; cm) ===
#endif
c                                                                    ===
c  /VERTSLABS/                                                       ===
c                                                                    ===
c     HD    Bottom depth at tracer grid points.     (real array; cm) ===
c     HDV   Bottom depth at velocity grid points.   (real array; cm) ===
c                                                                    ===
c  -------                                                           ===
c  Output:                                                           ===
c  -------                                                           ===
c                                                                    ===
c  Common Blocks:                                                    ===
c                                                                    ===
c  /VERTICAL/                                                        ===
c                                                                    ===
c     DZT   Thicknesses of T vertical boxes.        (real array; cm) ===
c     DZV   Thicknesses of UV vertical boxes.       (real array; cm) ===
c     HVZ   Depths at UV 3D grid points.            (real array; cm) ===
c     HTZ   Depths at tracer 3D grid points.        (real array; cm) ===
c                                                                    ===
c  /VERTSLABS/                                                       ===
c                                                                    ===
c     HVAV   Reciprocal thickness, velocity points. (real array; cm) ===
c                                                                    ===
c  Calls:  none                                                      ===
c                                                                    ===
c=======================================================================
c
c-----------------------------------------------------------------------
c  Define global data.
c-----------------------------------------------------------------------
c
#include <cdefs.h>
#include <param.h>
#include <pconst.h>
#include <hybrid.h>
#include <vertslabs.h>
#include <vertical.h>
c
c-----------------------------------------------------------------------
c  Define local data.
c-----------------------------------------------------------------------
c
      logical first
      integer i,j,k
#ifndef barotropic
     *        ,km1
#endif
      FLOAT
     *      dsigma,htmft,hvmfv,rhave,rhc
c
      save first,rhave,rhc
c
      data first /.true./
c
c=======================================================================
c  Begin executable code.
c=======================================================================
c
c-----------------------------------------------------------------------
c  Set often used interface parameters, on first call.
c-----------------------------------------------------------------------
c
      if (first) then
         if (kc.gt.0) then
            rhc = c1/hc
          else
            rhc = c0
         endif
         rhave = c1/have
         first = .false.
      endif
c
c-----------------------------------------------------------------------
c  Compute the thicknesses of the tracer and velocity boxes.
c-----------------------------------------------------------------------
c
      do 30 j = 1, jmt
      do 30 i = 1, imt
c
         do 10 k = 1, kc
            dsigma     = hz(k)*rhc
#ifdef dblsigma
            dzt(i,j,k) = dsigma*tinter(i,j)
            dzv(i,j,k) = dsigma*vinter(i,j)
#else
            dzt(i,j,k) = dsigma*hc
            dzv(i,j,k) = dsigma*hc
#endif
  10     continue
c
#ifdef dblsigma
         htmft = hd(i,j)  - tinter(i,j)
         hvmfv = hdv(i,j) - vinter(i,j)
#else
         htmft = hd(i,j)  - hc
         hvmfv = hdv(i,j) - hc
#endif
c
         do 20 k = kc+1, km
            dsigma     = hz(k)*rhave
            dzt(i,j,k) = dsigma*htmft
            dzv(i,j,k) = dsigma*hvmfv
  20     continue
c
  30  continue
c
#ifdef cyclic
c-----------------------------------------------------------------------
c  Ensure periodicity.
c-----------------------------------------------------------------------
c
      do 40 k = 1, km
      do 40 j = 1, jmt
         dzt(1,j,k)   = dzt(imtm1,j,k)
         dzt(imt,j,k) = dzt(2,j,k)
         dzv(1,j,k)   = dzv(imtm1,j,k)
         dzv(imt,j,k) = dzv(2,j,k)
  40  continue
c
#endif
c-----------------------------------------------------------------------
c  Compute depths at the middles of the tracer and velocity boxes.
c-----------------------------------------------------------------------
c
      do 50 j = 1, jmt
      do 50 i = 1, imt
         htz(i,1,j) = p5*dzt(i,j,1)
         hvz(i,1,j) = p5*dzv(i,j,1)
  50  continue
c
#ifndef barotropic
      do 60 k = 2, km
      do 60 j = 1, jmt
      do 60 i = 1, imt
         km1        = k - 1
         htz(i,k,j) = htz(i,km1,j) + p5*(dzt(i,j,km1)+dzt(i,j,k))
         hvz(i,k,j) = hvz(i,km1,j) + p5*(dzv(i,j,km1)+dzv(i,j,k))
  60  continue
c
#endif
c-----------------------------------------------------------------------
c  Compute total thickness of the water column over velocity grid.
c-----------------------------------------------------------------------
c
      do 70 j = 1, jmt
      do 70 i = 1, imt
         hvav(i,j) = dzv(i,j,1)
  70  continue
c
#ifndef barotropic
      do 80 k = 2, km
      do 80 j = 1, jmt
      do 80 i = 1, imt
         hvav(i,j) = hvav(i,j) + dzv(i,j,k)
  80  continue
c
#endif
      do 90 j = 1, jmt
      do 90 i = 1, imt
         hvav(i,j) = c1/hvav(i,j)
  90  continue
c
      return
      end
