      subroutine state(tx,sx,depth,rho)
c
c=======================================================================
c                                                                   ====
c  Nonlinear or linear equation of state for seawater.              ====
c                                                                   ====
c  On Input:                                                        ====
c                                                                   ====
#ifndef pottemp
c     T       in situ temperature (real array; degress Celcius)     ====
#else
c     T       potential temperature (real array; degress Celcius)   ====
#endif
c     S       salinity anomaly from SMEAN (real array; PSS)         ====
c             (SMEAN is added locally in order to compute density)  ====
c     DEPTH   depths (real array; cm)                               ====
c                                                                   ====
c  On Output:                                                       ====
c                                                                   ====
c     RHO     density anomaly (real array; nondimensional)          ====
c                                                                   ====
c=======================================================================
c
c-----------------------------------------------------------------------
c  Define global data
c-----------------------------------------------------------------------
c
#include <cdefs.h>
#include <param.h>
#include <rhomean.h>
c
c-----------------------------------------------------------------------
c  Define local data
c-----------------------------------------------------------------------
c
      integer i,k
      FLOAT
     *      depth(imt,km),tx(imt,km),sx(imt,km),rho(imt,km)
#if defined unesco
      FLOAT
     *      a,aw,a1,b,bw,b1,c,d,d350,dk,dr35p,dr350,dvan,e,gam,kw,k0,
     *      k35,pk,r1,r2,r3,r4,r3500,t,s,sig,sigma,sr,sva,v350p,z
c
      save dr350,r3500,r4
c
      data dr350,r3500,r4 /28.106331,1028.1063,4.8314e-4/
#elif defined pottemp
      FLOAT
     *      pp,t,t2t,s
      FLOAT
     *      a(7),b(7),cf(7),g(7)
c
      save a,b,g
c
      data a /-1.36471e-1,4.681810e-2,8.070040e-1,-7.45353e-3,
     &        -2.94418e-3,3.435700e-5,3.486580e-5/
      data b /5.064230e-1,-3.57109e-3,-8.76148e-4,5.252430e-5,
     &        1.579760e-5,-3.46686e-7,-1.68764e-7/
      data g /-5.52640e-4,4.885840e-6,9.960270e-7,-7.25139e-8,
     &        -3.98736e-9,4.006310e-10,8.263680e-11/
#endif
c
c=======================================================================
c  Begin executable code.
c=======================================================================
c
#if defined unesco
c-----------------------------------------------------------------------
c  International equation of state for seawater (Millero et al., 1980:
c  A new high-pressure equation of state for seawater. Deep-Sea Res.,
c  27a, 255-264).
c-----------------------------------------------------------------------
c
      do 20 k=1,km
        do 10 i=1,imt
          z=depth(i,k)*0.001
          t=tx(i,k)
          s=sx(i,k)+smean
          sr=sqrt(abs(s))
          r1=((((6.536332e-9*t-1.120083e-6)*t+1.001685e-4)*t
     *       -9.095290e-3)*t+6.793952e-2)*t-28.263737
          r2=(((5.3875e-9*t-8.2467e-7)*t+7.6438e-5)*t-4.0899e-3)*t
     *       +8.24493e-1
          r3=(-1.6546e-6*t+1.0227e-4)*t-5.72466e-3
          sig=(r4*s+r3*sr+r2)*s+r1
          v350p=1.0/r3500
          sva=-sig*v350p/(r3500+sig)
          sigma=dr350-(sva/(v350p*(v350p+sva)))
          e=(9.1697e-10*t+2.0816e-8)*t-9.9348e-7
          bw=(5.2787e-8*t-6.12293e-6)*t+3.47718e-5
          b=bw+e*s
          d=1.91075e-4
          c=(-1.6078e-6*t-1.0981e-5)*t+2.2838e-3
          aw=((-5.77905e-7*t+1.16092e-4)*t+1.43713e-3)*t-0.1194975
          a=(d*sr+c)*s+aw
          b1=(-5.3009e-4*t+1.6483e-2)*t+7.944e-2
          a1=((-6.1670e-5*t+1.09987e-2)*t-0.603459)*t+54.6746
          kw=(((-5.155288e-5*t+1.360477e-2)*t-2.327105)*t+148.4206)*t
     *       -1930.06
          k0=(b1*sr+a1)*s+kw
          dk=(b*z+a)*z+k0
          k35=(5.03217e-5*z+3.359406)*z+21582.27
          gam=z/k35
          pk=1.0-gam
          sva=sva*pk+(v350p+sva)*z*dk/(k35*(k35+dk))
          v350p=v350p*pk
          d350=gam/pk
          dr35p=r3500*d350
          dvan=sva/(v350p*(v350p+sva))
          sigma=dr350+dr35p-dvan
          rho(i,k)=sigma/rho0
  10    continue
  20  continue
c
#elif defined pottemp
c-----------------------------------------------------------------------
c  This the equation of state which has a simpler algebraic
c  form and it is from JGR, vol. 104, No. C1, pp. 1537-1540 (1999)
c  (Authors David Brydon, Shan Sun and Rainer Bleck).
c-----------------------------------------------------------------------
c
      do 20 k=1,km
        do 10 i=1,imt
          pp=depth(i,k)*1.0e-4
          t=tx(i,k)
          s=sx(i,k)+smean
          t2t=t*t
          cf(1)=a(1)+(b(1)+g(1)*pp)*pp
          cf(2)=a(2)+(b(2)+g(2)*pp)*pp
          cf(3)=a(3)+(b(3)+g(3)*pp)*pp
          cf(4)=a(4)+(b(4)+g(4)*pp)*pp
          cf(5)=a(5)+(b(5)+g(5)*pp)*pp
          cf(6)=a(6)+(b(6)+g(6)*pp)*pp
          cf(7)=a(7)+(b(7)+g(7)*pp)*pp
          rho(i,k)=cf(1)+(cf(2)+cf(5)*s+cf(6)*t2t)*t+cf(4)*t2t
     *            +(cf(3)+cf(7)*t2t)*s
          rho(i,k)=rho(i,k)/rho0
  10    continue
  20  continue
c
#elif defined leqstate
c-----------------------------------------------------------------------
c  Linear equation of state. The user needs to select the appropriate
c  values for the mean temperature TBAR, thermal expansion coefficient
c  TCOEF, mean salinity SBAR, and saline contraction coefficeint SCOEF.
c  These values are specified in common block RHOMEAN.
c-----------------------------------------------------------------------
c
      do 10 k=1,km
      do 10 i=1,imt
        rho(i,k)=-tcoef*(tx(i,k)-tbar)+scoef*((sx(i,k)+smean)-sbar)
  10  continue
c
#else
c-----------------------------------------------------------------------
c  Simplified nonlinear equation of state.
c-----------------------------------------------------------------------
c
      do 20 k=1,km
        do 10 i=1,imt
          rho(i,k)=5891.+depth(i,k)/1013.+3.*(sx(i,k)+smean)+
     *             (38.-0.375*tx(i,k))*tx(i,k)
  10    continue
  20  continue
      do 40 k=1,km
        do 30 i=1,imt
          rho(i,k)=(1779.5+(11.25-0.0745*tx(i,k))*tx(i,k)
     *             -(3.8+0.01*tx(i,k))*(sx(i,k)+smean))/rho(i,k)
  30    continue
  40  continue
      do 60 k=1,km
        do 50 i=1,imt
          rho(i,k)=1.0/(0.698+rho(i,k))-1.02
  50    continue
  60  continue
c
#endif
      return
      end
