      subroutine cgpois (r,x,h,p,xr,eps,itermax,im,jm,yxmrat,xymrat,
     *                   nmvec,iguess,ipt,ier)
c
c=======================================================================
c                                                                    ===
c  This routine solves Poisson equation using conjugate gradient.    ===
c                                                                    ===
c  On Input:                                                         ===
c                                                                    ===
c     XR      forcing field (interior and boundary conditions).      ===
c     IPT     type of finite differences stencil for the forcing     ===
c             field:                                                 ===
c             [0] five points PLUS  (+) Laplacian.                   ===
c             [1] five points TIMES (x) Laplacian.                   ===
c     IM      number of points in the x-direction.                   ===
c     JM      number of points in the y-direction.                   ===
c     YXMRAT  the ratio of the y-metric terms over the x-metric      ===
c             terms:                                                 ===
c             IPT=0 -> XYMRAT(i,j) evaluated at (i+1/2, j).          ===
c             IPT=1 -> XYMRAT(i,j) evaluated at (i+1/2, j+1/2).      ===
c     XYMRAT  the ratio of the x-metric terms over the y-metric      ===
c             terms:                                                 ===
c             IPT=0 -> XYMRAT(i,j) evaluated at (i, j+1/2).          ===
c             IPT=1 -> XYMRAT(i,j) evaluated at (i+1/2, j+1/2).      ===
c     NMVEC   first dimension for working arrays.                    ===
c     ITERMAX maximum number of iterations.                          ===
c     EPS     convergence criterion (percent of RMS).                ===
c     IGUESS  flag for first guess solution:                         ===
c             [0] first guess for solution is a "zero" vector.       ===
c             [1] first guess solution in vector P (destroyed at     ===
c                 output).                                           ===
c     R       working array dimensioned (NMVEC,2).                   ===
c     X       working array dimensioned (NMVEC,2).                   ===
c     H       working array dimensioned (NMVEC,3).                   ===
c                                                                    ===
c  On Ouput:                                                         ===
c                                                                    ===
c     XR      solution to Poisson equation.                          ===
c     IER     error flag:                                            ===
c             [0] successful computation.                            ===
c             [1] underdimensioned working arrays.                   ===
c             [2] illegal Laplacian finite difference stencil.       ===
c             [3] outrageous convergence criteria (EPS, ITER).       ===
c                                                                    ===
c   Calls:   OP                                                      ===
c                                                                    ===
c=======================================================================
c
c-----------------------------------------------------------------------
c  Define local data.
c-----------------------------------------------------------------------
c
      integer i,idown,ier,iguess,im,ip,ipt,iter,itermax,iup,j,jm,nmvec
      real c0,c1,bot,eps,error,error2,gamma,lambda,size,
     *     top,p5,c2
     *     ,vel_err,vtest
      real h(nmvec,0:2),p(nmvec),r(nmvec,0:1),x(nmvec,0:1),xr(im,jm),
     *     xymrat(im,jm),yxmrat(im,jm),fn(0:1)
      parameter (c0=0.0,c1=1.0,p5=0.5,c2=2.0)
c
c-----------------------------------------------------------------------
c  Begin executable code.
c-----------------------------------------------------------------------
c
c  Check input for gross errors.
c
      if(nmvec.lt.im*jm) then
        ier=1
      elseif((ipt.gt.1).or.(ipt.lt.0)) then
        ier=2
      elseif((eps.lt.1.0e-7).or.(itermax.lt.1)) then
        ier=3
      else
        ier=0
      endif
      if(ier.ne.0) return
c
c-----------------------------------------------------------------------
c  Initialize several quantities.
c-----------------------------------------------------------------------
c
c  Set pointer IUP.
c
      iup=0
c
c  Set first guess to zero.
c
      if(iguess.eq.0) then
        do 10 ip=1,im*jm
          x(ip,iup)=c0
  10    continue
c
c  Use first guess contained in P.
c
      else
        do 20 ip=1,im*jm
          x(ip,iup)=p(ip)
  20    continue
      endif
c
c  Compute residual, R=Laplacian[X]-XR.
c
      call op(im,jm,yxmrat,xymrat,x(1,iup),r(1,iup),xr,ipt)
c
c  Initialize conjugate directions array H.
c
       do 30 ip=1,im*jm
         h(ip,iup)=r(ip,iup)
  30   continue
c
c  Set array P to zero.
c
      do 40 ip=1,im*jm
        p(ip)=c0
  40  continue
c
c  Compute initial RMS and set convergence criteria.
c
      call op(im,jm,yxmrat,xymrat,x(1,iup),h(1,2),p,ipt)
      iter=0
      error2=eps*p5
      bot=c0
      fn(iup)=c0
      do 60 j=2,jm-1
        do 50 i=2,im-1
          ip=i+(j-1)*im
          bot=bot + x(ip,iup)*h(ip,2)
          fn(iup)=fn(iup) + xr(i,j)*x(ip,iup)
  50    continue
  60  continue
      fn(iup) = p5*bot - fn(iup)
      idown=1-iup
      fn(idown) = c2*fn(iup)+c1
      vtest = c1 + c2*eps
c
c-----------------------------------------------------------------------
c  Iterate util convergence or to maximum number of iterations.
c-----------------------------------------------------------------------
c
      do 170 while ( ( (abs(fn(iup)-fn(idown)) .gt.
     &              error2*abs(fn(iup)+fn(idown))) .or. (vtest.gt.eps) )
     &                                       .and. (iter.le.itermax))
        iup=1-iup
        idown=1-iup
c
c  Compute H_2=Laplacian[H].
c
         call op(im,jm,yxmrat,xymrat,h(1,idown),h(1,2),p,ipt)
c
c  Compute LAMBDA=-(R*H)/(H*H_2).
c
         top=c0
         bot=c0
         do 80 j=2,jm-1
           do 70 i=2,im-1
             ip=i+(j-1)*im
             top=top+r(ip,idown)*h(ip,idown)
             bot=bot+h(ip,idown)*h(ip,2)
  70       continue
  80     continue
         lambda=-top/bot
c
c  Compute X+, R+
c
         error=c0
         size=c0
         do 100 j=2,jm-1
           do 90 i=2,im-1
             ip=i+(j-1)*im
             top=lambda*h(ip,idown)
             x(ip,iup)=x(ip,idown)+top
             r(ip,iup)=r(ip,idown)+lambda*h(ip,2)
             error=error+top*top
             size=size+x(ip,iup)*x(ip,iup)
  90       continue
 100     continue
c
c  Compute maximum relative change in barotropic velocity.
c
         vtest = vel_err (im,jm,x(1,iup),h(1,idown),lambda)
c
c  Compute GAMMA.
c
         top=c0
         do 120 j=2,jm-1
           do 110 i=2,im-1
             ip=i+(j-1)*im
             top=top+r(ip,iup)*h(ip,2)
 110       continue
 120     continue
         gamma=-top/bot
c
c  Compute new direction H+
c
         do 140 j=2,jm-1
           do 130 i=2,im-1
             ip=i+(j-1)*im
             h(ip,iup)=r(ip,iup)+gamma*h(ip,idown)
 130       continue
 140     continue
c
c  Compute convergence criteria.
c
         fn(iup) = p5*bot
         do 160 j=2,jm-1
           do 150 i=2,im-1
             ip=i+(j-1)*im
             fn(iup) = fn(iup) - xr(i,j)*x(ip,iup)
 150       continue
 160     continue
         iter=iter+1
 170  continue
c
c  Report convergence status.
c
      error=sqrt(error/float((im-2)*(jm-2)))
      size=sqrt(size/float((im-2)*(jm-2)))
      print 180, iter,itermax,eps,p5*abs(fn(iup)+fn(idown)),
     &                                 abs(fn(iup)-fn(idown)),error,size
 180  format(/,' CGPOIS      iteration = ',i10,
     &        /'    Max. allowed iter. = ',i10
     *       /,'           convergence = ',1pe15.8,
     *       /,'    objective function = ',1pe15.8,
     *       /,'    change in obj. fn. = ',1pe15.8,
     *       /,'RMS change in solution = ',1pe15.8,
     *       /,'          RMS solution = ',1pe15.8)
      print 190, vtest
 190  format('  Max. Rel. Vel. Chng. = ',1pe15.8)
      call op(im,jm,yxmrat,xymrat,x(1,iup),r(1,iup),xr,ipt)
      error=c0
      size=c0
      do 210 j=2,jm-1
        do 200 i=2,im-1
          ip=i+(j-1)*im
          error=error+r(ip,iup)*r(ip,iup)
          size=size+xr(i,j)*xr(i,j)
 200    continue
 210  continue
      error=sqrt(error/float((im-2)*(jm-2)))
      size=sqrt(size/float((im-2)*(jm-2)))
      print 220, error,size
 220  format('          RMS residual = ',1pe15.8/
     *       '           RMS forcing = ',1pe15.8)
c
c  Write out solution in the interior of array XR.
c
      do 240 j=2,jm-1
        do 230 i=2,im-1
          ip=i+(j-1)*im
          xr(i,j)=x(ip,iup)
 230    continue
 240  continue
      return
      end
