      subroutine get_dom (gname,coord,nx,ny,dx,dy,clon,clat,delx,dely,
     &                    theta,ncst,nisle,lencst,icst,jcst,landt)
c
c=======================================================================
c                                                                    ===
c  This routine extracts the needed domain parameters from GRIDS     ===
c  netCDF file.                                                      ===
c                                                                    ===
c  ------                                                            ===
c  Input:                                                            ===
c  ------                                                            ===
c                                                                    ===
c   GNAME    Name of GRIDS netCDF file.              (string.)       ===
c                                                                    ===
c   Common Blocks:                                                   ===
c                                                                    ===
c   /IOUNITS/                                                        ===
c                                                                    ===
c   STDOUT   standard output unit.                   (integer)       ===
c                                                                    ===
c  -------                                                           ===
c  Output:                                                           ===
c  -------                                                           ===
c                                                                    ===
c   COORD    flag for coordinate type.               (integer)       ===
c               [0]  Cartesian (tangent plane) grid.                 ===
c               [1]  unrotated spherical grid.                       ===
c               [2]  rotated spherical grid.                         ===
c   DELX,                                                            ===
c   DELY     Offset from transformation point to                     ===
c            grid center.                            (real; cm|deg)  ===
c   DX, DY   grid spacing in x & y directions.       (real; cm|deg)  ===
c   NX, NY   number of grid points in x & y dir.s.   (integer)       ===
c   CLON     longitude at center of grid             (real; deg)     ===
c   CLAT     latitude at center of grid              (real; deg)     ===
c   THETA    angle grid is rotated w/ respect to E.  (real; deg)     ===
c   NCST     Number of coastlines.                   (integer)       ===
c   NISLE    Number of islands.                      (integer)       ===
c   LENCST   Length of coastlines.                   (integer vector)===
c   ICST     X-index of coastlines.                  (integer array) ===
c   JCST     Y-index of coastlines.                  (integer array) ===
c   LANDT    Tracer-grid land mask.                  (integer array) ===
c                                                                    ===
c         Calls:  EXITUS,          LENGTH                            ===
c  NetCDF Calls:  NF_GET_VAR_INT,  NF_GET_VAR_REAL,  NF_GET_VARA_INT,===
c                 NF_INQ_VARID,    NF_OPEN                           ===
c                                                                    ===
c=======================================================================
c
c-----------------------------------------------------------------------
c  Define global data.
c-----------------------------------------------------------------------
c
#include <param.h>
#include <netcdf.inc>
#include <iounits.h>
c
c-----------------------------------------------------------------------
c  Define local data.
c-----------------------------------------------------------------------
c
      integer coord,gbgn,gend,glen,mxcst,n,ncid,ncst,nisle,nx,ny,status,
     &        varid
      integer count(2),icst(mpseg,mcseg),jcst(mpseg,mcseg),landt(np),
     &        lencst(mcseg),start(2)
      real          clon,clat,dx,dy,delx,dely,m2cm,theta
      character*(*) gname
c
      parameter (m2cm=100.0)
c
      data start /2*1/
      data count /2*1/
c
c=======================================================================
c  Begin executable code.
c=======================================================================
c
c-----------------------------------------------------------------------
c  Open GRIDS netCDF file.
c-----------------------------------------------------------------------
c
c  Determine length of GRIDS file name.
c
      call length (gname,glen,gbgn,gend)
c
      if (glen.lt.1) then
         write (stdout,900)
         call exitus ('GET_DOM')
      end if
c
c  Open GRIDS netCDF file.
c
      status = nf_open (gname(gbgn:gend),NF_NOWRITE,ncid)
c
      if (status.ne.NF_NOERR) then
         write (stdout,910) 'open',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
c-----------------------------------------------------------------------
c  Read domain parameters.
c-----------------------------------------------------------------------
c
c  Read coordinate type.
c
      status = nf_inq_varid (ncid,'coord',varid)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'find','coord',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
      status = nf_get_var_int (ncid,varid,coord)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'read','coord',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
c  Read domain size.
c
      status = nf_inq_varid (ncid,'imt',varid)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'find','imt',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
      status = nf_get_var_int (ncid,varid,nx)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'read','imt',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
      status = nf_inq_varid (ncid,'jmt',varid)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'find','jmt',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
      status = nf_get_var_int (ncid,varid,ny)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'read','jmt',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
c  Read grid spacing.
c
      status = nf_inq_varid (ncid,'meandx',varid)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'find','meandx',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
      status = nf_get_var_real (ncid,varid,dx)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'read','meandx',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
      status = nf_inq_varid (ncid,'meandy',varid)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'find','meandy',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
      status = nf_get_var_real (ncid,varid,dy)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'read','meandy',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
c  Read transformation centroid.
c
      status = nf_inq_varid (ncid,'rlngd',varid)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'find','rlngd',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
      status = nf_get_var_real (ncid,varid,clon)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'read','rlngd',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
      status = nf_inq_varid (ncid,'rlatd',varid)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'find','rlatd',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
      status = nf_get_var_real (ncid,varid,clat)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'read','rlatd',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
c  Read offset between grid & transformation centroids.
c
      status = nf_inq_varid (ncid,'delx',varid)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'find','delx',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
      status = nf_get_var_real (ncid,varid,delx)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'read','delx',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
      status = nf_inq_varid (ncid,'dely',varid)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'find','dely',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
      status = nf_get_var_real (ncid,varid,dely)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'read','dely',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
c  Read domain rotation angle.
c
      status = nf_inq_varid (ncid,'thetad',varid)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'find','thetad',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
      status = nf_get_var_real (ncid,varid,theta)
      if (status.ne.NF_NOERR) then
         write (stdout,920) 'read','thetad',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
c-----------------------------------------------------------------------
c  Read land masking parameters.
c-----------------------------------------------------------------------
c
c  Read number of distinct coastlines, if any.
c
      status = nf_inq_varid (ncid,'ncseg',varid)
c
      if (status.ne.NF_NOERR) then
c
         ncst  = 0
         nisle = 0
c
       else
c
         status = nf_get_var_int (ncid,varid,ncst)
         if (status.ne.NF_NOERR) then
            write (stdout,920) 'read','ncseg',gname(gbgn:gend)
            call exitus ('GET_DOM')
         end if
c
         if (ncst.gt.mcseg) then
            write (stdout,930) 'mcseg','mcseg',mcseg,'ncst',ncst,'mcseg'
            call exitus ('GET_DOM')
         end if
c
c     Read lengths of coastlines.
c
         status = nf_inq_varid (ncid,'lencoast',varid)
         if (status.ne.NF_NOERR) then
            write (stdout,920) 'find','lencoast',gname(gbgn:gend)
            call exitus ('GET_DOM')
         end if
c
         status = nf_get_vara_int (ncid,varid,1,ncst,lencst)
         if (status.ne.NF_NOERR) then
            write (stdout,920) 'read','lencoast',gname(gbgn:gend)
            call exitus ('GET_DOM')
         end if
c
c     Read maximum length of coastlines.
c
         status = nf_inq_varid (ncid,'lcseg',varid)
         if (status.ne.NF_NOERR) then
            write (stdout,920) 'find','lcseg',gname(gbgn:gend)
            call exitus ('GET_DOM')
         end if
c
         status = nf_get_var_int (ncid,varid,mxcst)
         if (status.ne.NF_NOERR) then
            write (stdout,920) 'read','lcseg',gname(gbgn:gend)
            call exitus ('GET_DOM')
         end if
c
         if (mxcst.gt.mpseg) then
            write (stdout,930) 'mpseg','mpseg',mpseg,'lcseg',mxcst,
     &                         'mpseg'
            call exitus ('GET_DOM')
         end if
c
c     Read coastlines.
c
         count(2) = 1
c
         do 10 n = 1, ncst
c
            start(2) = n
            count(1) = lencst(n)
c
            status = nf_inq_varid (ncid,'icoast',varid)
            if (status.ne.NF_NOERR) then
               write (stdout,920) 'find','icoast',gname(gbgn:gend)
               call exitus ('GET_DOM')
            end if
c
            status = nf_get_vara_int (ncid,varid,start,count,icst(1,n))
            if (status.ne.NF_NOERR) then
               write (stdout,920) 'read','icoast',gname(gbgn:gend)
               call exitus ('GET_DOM')
            end if
c
            status = nf_inq_varid (ncid,'jcoast',varid)
            if (status.ne.NF_NOERR) then
               write (stdout,920) 'find','jcoast',gname(gbgn:gend)
               call exitus ('GET_DOM')
            end if
c
            status = nf_get_vara_int (ncid,varid,start,count,jcst(1,n))
            if (status.ne.NF_NOERR) then
               write (stdout,920) 'read','jcoast',gname(gbgn:gend)
               call exitus ('GET_DOM')
            end if
c
  10     continue
c
c     Read landmask.
c
         start(2) = 1
         count(1) = nx
         count(2) = ny
c
         status = nf_inq_varid (ncid,'landt',varid)
         if (status.ne.NF_NOERR) then
            write (stdout,920) 'find','landt',gname(gbgn:gend)
            call exitus ('GET_DOM')
         end if
c
         status = nf_get_vara_int (ncid,varid,start,count,landt)
         if (status.ne.NF_NOERR) then
            write (stdout,920) 'read','landt',gname(gbgn:gend)
            call exitus ('GET_DOM')
         end if
c
c     Read number of islands, if any.
c
         status = nf_inq_varid (ncid,'nisle',varid)
c
         if (status.ne.NF_NOERR) then
c
            nisle = 0
c
          else
c
            status = nf_get_var_int (ncid,varid,nisle)
            if (status.ne.NF_NOERR) then
               write (stdout,920) 'read','nisle',gname(gbgn:gend)
               call exitus ('GET_DOM')
            end if
c
         end if
c
      end if
c
c-----------------------------------------------------------------------
c  Close GRIDS netCDF file.
c-----------------------------------------------------------------------
c
      status = nf_close (ncid)
c
      if (status.ne.NF_NOERR) then
         write (stdout,910) 'close',gname(gbgn:gend)
         call exitus ('GET_DOM')
      end if
c
c-----------------------------------------------------------------------
c  Convert to CGS.
c-----------------------------------------------------------------------
c
      if (coord.eq.0) then
         dx   = dx*m2cm
         dy   = dy*m2cm
         delx = delx*m2cm
         dely = dely*m2cm
      end if
c
      return
c
 900  format (/'***Error:  GET_DOM - zero length name for GRIDS netCDF',
     &        ' file.')
 910  format (/'***Error:  GET_DOM - unable to ',a,' GRIDS netCDF file:'
     &        /11x,1h",a,1h")
 920  format (/'***Error:  GET_DOM - unable to ',a,' variable ',1h",a,
     &        1h",' in GRIDS netCDF file:'/11x,1h",a,1h")
 930  format (/'***Error:  GET_DOM - undersized dimension ',1h",a,1h"/
     &        11x,1h",a,1h",' = ',i10/11x,1h",a,1h",' = ',i10/11x,
     &        'Reset ',1h",a,1h",', recompile and rerun.')
c
      end
