      subroutine check_depths
c
c=======================================================================
c                                                                    ===
c  This routine compares model levels to the supplied data levels.   ===
c                                                                    ===
c  ------                                                            ===
c  Input:                                                            ===
c  ------                                                            ===
c                                                                    ===
c  Common Blocks:                                                    ===
c                                                                    ===
c  /HYBRID/                                                          ===
c                                                                    ===
c     DEPTHMT   depths at middle of T boxes.   (real array; cm)      ===
c     DEPTHMV   depths at middle of V boxes.   (real array; cm)      ===
c                                                                    ===
c  /IOUNITS/                                                         ===
c                                                                    ===
c     STDOUT   standard output unit.   (integer)                     ===
c                                                                    ===
c  /MEANTS/                                                          ===
c                                                                    ===
c     NPROF   number of points in profile.   (integer)               ===
c     ZM      input mean profile depths.     (real vector; meter)    ===
c                                                                    ===
c  /NDIMEN/                                                          ===
c                                                                    ===
c     IM   number of points in the x-direction.   (integer)          ===
c     JM   number of points in the y-direction.   (integer)          ===
c     KM   number of points in the z-direction.   (integer)          ===
c                                                                    ===
c  /ZDAT/                                                            ===
c                                                                    ===
c     KFLD   number of levels in input data.       (integer)         ===
c     ZBOT   depths used to set flat topography.   (real vector; cm) ===
c                                                                    ===
c=======================================================================
c
c-----------------------------------------------------------------------
c  Define global data.
c-----------------------------------------------------------------------
c
#include <param.h>
#include <hybrid.h>
#include <iounits.h>
#include <meants.h>
#include <ndimen.h>
#include <zdat.h>
#include <switches.h>
c
c-----------------------------------------------------------------------
c  Define local data.
c-----------------------------------------------------------------------
c
      real       cm2m,m2cm
      parameter (cm2m=0.01,m2cm=100.0)
#ifdef surfpress
      real       c0
      parameter (c0=0.0)
#endif
c
      integer ip
      real    zpemax,zpemin
c
c=======================================================================
c  Begin executable code.
c=======================================================================
c
c-----------------------------------------------------------------------
c  Find maximum & minimum depths of model grid.
c-----------------------------------------------------------------------
c
      zpemax = max( depthmt(1,km), depthmv(1,km) )
      zpemin = min( depthmt(1,1), depthmv(1,1) )
c
      do 10 ip = 2, im*jm
         zpemax = max( zpemax, depthmt(ip,km), depthmv(ip,km) )
         zpemin = min( zpemin, depthmt(ip,1), depthmv(ip,1) )
  10  continue
#ifdef surfpress
      if(ptype.gt.1) then
         zpemin=c0
      endif
#endif
c
c-----------------------------------------------------------------------
c  Provide error message if data levels do not bracket model grid.
c-----------------------------------------------------------------------
c
      if (zpemin.lt.zfld(1)) then
        write(stdout,20) zfld(1)*cm2m,zpemin*cm2m
  20    format(/'+++ Warning:  CHECK_DEPTHS - there is no input data ',
     &         'available for depths shallower'/26x,'than, Z = ',1pg11.3
     &         ,' meters'/
     &         26x,'Shallowest model level is: Z = ',1pg11.3,' meters.')
      endif
c
      if (zpemax.gt.zfld(kfld)) then
        write(stdout,30) zfld(kfld)*cm2m,zpemax*cm2m
  30    format(/'+++ Warning:  CHECK_DEPTHS - there is no input data ',
     &         'available for depths deeper'/26x,'than, Z = ',1pg11.3,
     &         ' meters'/
     &         26x,'Deepest model level is: Z = ',1pg11.3,' meters.')
      endif
c
c-----------------------------------------------------------------------
c  Provide error message if mean T-S levels do not bracket model grid.
c-----------------------------------------------------------------------
c
      if (zpemin.lt.abs(m2cm*zm(1))) then
        write(stdout,40) abs(zm(1)),zpemin*cm2m
  40    format(/'+++ Warning:  CHECK_DEPTHS - there is no mean T-S ',
     &         'data available for depths shallower'/26x,'than, Z = ',
     &         1pg11.3,' meters'/
     &         26x,'Shallowest model level is: Z = ',1pg11.3,' meters.'/
     &         26x,'This is a fatal error for PE versions below 9.6')
      endif
c
      if (zpemax.gt.abs(m2cm*zm(nprof))) then
        write(stdout,50) abs(zm(nprof)),zpemax*cm2m
  50    format(/'+++ Warning:  CHECK_DEPTHS - there is no mean T-S ',
     &         'data available for depths deeper'/26x,'than, Z = ',
     &         1pg11.3,' meters'/
     &         26x,'Deepest model level is: Z = ',1pg11.3,' meters.'/
     &         26x,'This is a fatal error for PE versions below 9.6')
      endif
c
#ifdef sunflush
      call flush(stdout)
c
#endif
      return
      end
