      subroutine vertgrid
c
c=======================================================================
c                                                                    ===
c  This subroutine sets the depths of each tracer and velocity grid  ===
c  point in the hybrid (single or double transformation) coordinate  ===
c  system:                                                           ===
c                                                                    ===
c  ------                                                            ===
c  Input:                                                            ===
c  ------                                                            ===
c                                                                    ===
c  Common Blocks:                                                    ===
c                                                                    ===
c  /HYBRID/                                                          ===
c                                                                    ===
c     HV     The bottom depth at V-grid points.     (real array; cm) ===
c     HAVG   Average thickness of hybrid levels.    (real; cm)       ===
c     HC     Depth of the interface.                (real; cm)       ===
c     HZ     Flat level thicknesses                 (real array; cm) ===
c     KC     Coordinate interface level             (integer)        ===
c                                                                    ===
c  /NDIMEN/                                                          ===
c                                                                    ===
c     IM   Number of points in the x-direction.      (integer)       ===
c     JM   Number of points in the y-direction.      (integer)       ===
c     KM   Number of points in the z-direction.      (integer)       ===
c                                                                    ===
c  /ZDAT/                                                            ===
c                                                                    ===
c     H        The bottom depth at T-grid points.   (real array; cm) ===
c     TINTER   The depth of interface over T-grid.  (real array; cm) ===
c     VINTER   The depth of interface over V-grid.  (real array; cm) ===
c                                                                    ===
c  -------                                                           ===
c  Output:                                                           ===
c  -------                                                           ===
c                                                                    ===
c  Common Blocks:                                                    ===
c                                                                    ===
c  /HYBRID/                                                          ===
c                                                                    ===
c     DEPTHMT   Depths at the middle of T-boxes.    (real array; cm) ===
c     DEPTHMV   Depths at the middle of V-boxes.    (real array; cm) ===
c     DZT       Thicknesses over T-grid.            (real array; cm) ===
c     DZV       Thicknesses over V-grid.            (real array; cm) ===
c     HVZ       Total thickness over V-grid.        (real array; cm) ===
c                                                                    ===
c  Calls:  none                                                      ===
c                                                                    ===
c=======================================================================
c
c-----------------------------------------------------------------------
c  Define global data.
c-----------------------------------------------------------------------
c
#include <param.h>
#include <hybrid.h>
#include <ndimen.h>
#include <zdat.h>
c
c-----------------------------------------------------------------------
c  Define local data.
c-----------------------------------------------------------------------
c
      integer imjm,ip,k,km1
      real    c0,c1,dsigma,htmft,hvmfv,p5,rhavg,rhc
c
      parameter (c0=0.0, c1=1.0, p5=0.5)
c
c=======================================================================
c  Begin executable code.
c=======================================================================
c
c-----------------------------------------------------------------------
c  Compute the thicknesses of the tracer and velocity boxes.
c-----------------------------------------------------------------------
c
      imjm  = im*jm
      if (kc.gt.0) then
         rhc = c1/hc
       else
         rhc = c0
      end if
      rhavg = c1/havg
c
      do 30 ip = 1, imjm
c
         do 10 k = 1, kc
            dsigma    = hz(k)*rhc
            dzt(ip,k) = dsigma*tinter(ip)
            dzv(ip,k) = dsigma*vinter(ip)
  10     continue
c
         htmft = h(ip)  - tinter(ip)
         hvmfv = hv(ip) - vinter(ip)
c
         do 20 k = kc+1, km
            dsigma    = hz(k)*rhavg
            dzt(ip,k) = dsigma*htmft
            dzv(ip,k) = dsigma*hvmfv
  20     continue
c
  30  continue
c
c-----------------------------------------------------------------------
c  Compute depths at the middles of the tracer and velocity boxes.
c-----------------------------------------------------------------------
c
      do 40 ip = 1, imjm
         depthmt(ip,1) = p5*dzt(ip,1)
         depthmv(ip,1) = p5*dzv(ip,1)
  40  continue
c
      do 50 k = 2, km
      do 50 ip = 1, imjm
         km1           = k - 1
         depthmt(ip,k) = depthmt(ip,km1) + p5*(dzt(ip,km1)+dzt(ip,k))
         depthmv(ip,k) = depthmv(ip,km1) + p5*(dzv(ip,km1)+dzv(ip,k))
  50  continue
c
c-----------------------------------------------------------------------
c  Compute total thickness of the water column over velocity grid.
c-----------------------------------------------------------------------
c
      do 60 ip = 1, imjm
         hvz(ip) = dzv(ip,1)
  60  continue
c
      do 70 k = 2, km
      do 70 ip = 1, imjm
         hvz(ip) = hvz(ip) + dzv(ip,k)
  70  continue
c
c
c Copy information to levels associated to mean sea surface
c
#ifdef surfpress
      do 90 k=1,km
         do 80 ip=1,im*jm
            depthft(ip,k)=depthmt(ip,k)
            depthfv(ip,k)=depthmv(ip,k)
 80      continue
 90   continue
#endif
      return
      end
